<?php

namespace App\Services;

use App\Models\Item;
use App\Models\Cart;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class PromoHandler
{
    /**
     * Calculate price, discount, and cost for a cart item based on promo and item.
     */
    public static function calculateCartValues($item, $qty, $price, $discount, $nprice, $promo, $promotipe, $pack, $benefit, $user)
    {
        $isConsignment = $item->consignment === 'Y';
        if ($isConsignment) {
            $supplier = DB::table('Luv2_supplier')->where('company_code', $user->company_code)->where('code', $item->supp_code)->first();
            $sharedMargin = $supplier->shared_margin ?? 0;
            $cost = ($price * ((100 - $sharedMargin) / 100)) * $qty;
        } else {
            $sharedMargin = 0;
            $cost = $item->avgprice * $qty;
        }
        return [
            'price' => $price,
            'nprice' => $nprice,
            'discount' => $discount,
            'promo' => $promo,
            'promotipe' => $promotipe,
            'pack' => $pack,
            'benefit' => $benefit,
            'cost' => $cost,
            'shared_margin' => $sharedMargin,
        ];
    }

    /**
     * Get available pack for a promo and item.
     */
    public static function getAvailablePack($itemCode, $promoCode, $promoTipe, $user)
    {
        $checkCartPacks = Cart::where('promo', $promoCode)
            ->where('promo_tipe', $promoTipe)
            ->where('user_code', $user->code)
            ->pluck('pack')
            ->toArray();
        $availablePacks = DB::table('Luv2_promo_item')
            ->where('company_code', $user->company_code)
            ->where('no', $promoCode)
            ->where('item_code', $itemCode)
            ->whereNotIn('pack', $checkCartPacks)
            ->pluck('pack')
            ->toArray();
        sort($availablePacks);
        return reset($availablePacks);
    }

    /**
     * Set promo session for fulfillment.
     */
    public static function setPromoSession($promo, $userId)
    {
        if ($promo->tipe == 3) {
            session(['PromoBundlingFulfill' . $userId => $promo->no]);
        } elseif ($promo->tipe == 4 && $promo->benefit == 1) {
            session(['Promo-GWP-FreeItem' . $userId => $promo->no]);
        } elseif ($promo->tipe == 4 && $promo->benefit == 2) {
            session(['Promo-GWP-CheapestItem' . $userId => $promo->no]);
        } elseif ($promo->tipe == 5 && $promo->benefit == 1) {
            session(['Promo-GWP-ValueFreeItemFulfil' . $userId => $promo->no]);
        } elseif ($promo->tipe == 5 && $promo->benefit == 2) {
            session(['Promo-GWP-ValueCheapestFulfil' . $userId => $promo->no]);
        } elseif ($promo->tipe == 5 && $promo->benefit == 3) {
            session(['Promo-Voucher-ValueFulfil' . $userId => $promo->no]);
        }
    }

    /**
     * Clear promo session for fulfillment.
     */
    public static function clearPromoSession($promo, $userId)
    {
        if ($promo->tipe == 3) {
            session()->forget('PromoBundlingFulfill' . $userId);
        } elseif ($promo->tipe == 4 && $promo->benefit == 1) {
            session()->forget('Promo-GWP-FreeItem' . $userId);
        } elseif ($promo->tipe == 4 && $promo->benefit == 2) {
            session()->forget('Promo-GWP-CheapestItem' . $userId);
        } elseif ($promo->tipe == 5 && $promo->benefit == 1) {
            session()->forget('Promo-GWP-ValueFreeItemFulfil' . $userId);
        } elseif ($promo->tipe == 5 && $promo->benefit == 2) {
            session()->forget('Promo-GWP-ValueCheapestFulfil' . $userId);
        } elseif ($promo->tipe == 5 && $promo->benefit == 3) {
            session()->forget('Promo-Voucher-ValueFulfil' . $userId);
        }
    }

    /**
     * Get promo result for a given item and promo.
     */
    public static function getPromoResult($promo, $item, $cart = null, $context = [])
    {
        $result = [
            'price' => $item->price,
            'discount' => 0,
            'benefit' => $promo->benefit,
            'promo' => $promo->no ?? 0,
            'promotipe' => $promo->tipe ?? 0,
            'pack' => $context['pack'] ?? 0,
        ];
        switch ($promo->tipe) {
            case 1: // Percentage discount
                $result['discount'] = $promo->disc;
                $result['price'] = $item->price * (1 - $promo->disc / 100);
                break;
            case 2: // Fixed price
                $result['discount'] = $item->price - $promo->disc_price;
                $result['price'] = $promo->disc_price;
                break;
            case 3: // Bundling
                $result['pack'] = $context['pack'] ?? 0;
                break;
            case 4:
            case 5:
            case 6:
            case 7:
                $result['discount'] = $promo->disc;
                break;
        }
        return $result;
    }

    /**
     * Update cart items after promo changes (for fulfillment, price update, etc.)
     */
    public static function updateCartItemsAfterPromo($promo, $user)
    {
        $cartItems = Cart::where('promo', $promo->no)->where('user_code', $user->code)->get();
        // Example: update price/discount for all items in the promo
        foreach ($cartItems as $cart) {
            $item = Item::where('code', $cart->code)->where('company_code', $user->company_code)->first();
            $result = self::getPromoResult($promo, $item, $cart);
            $cart->price = $result['price'];
            $cart->discount = $result['discount'];
            $cart->total_price = $cart->price * $cart->qty;
            $cart->save();
        }
    }

    /**
     * Utility: check if a promo is fulfilled (all packs/items present).
     */
    public static function isPromoFulfilled($promo, $user)
    {
        $cart = Cart::where('promo', $promo->no)->where('user_code', $user->code)->get();
        $syarat = intval($promo->qty_pack);
        $requiredPacks = range(1, $syarat);
        foreach ($requiredPacks as $requiredPack) {
            $packExistsInCart = false;
            foreach ($cart as $cartItem) {
                if ($cartItem->pack == $requiredPack) {
                    $packExistsInCart = true;
                    break;
                }
            }
            if (!$packExistsInCart) {
                return false;
            }
        }
        return true;
    }
} 