<?php

namespace App\Livewire\User;

use App\Models\User;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;

class UserTable extends Component
{
    use LivewireAlert;
    public $users;
    public $whscode = '';
    public $code = '';
    public $role = 'Cashier';

    #[Rule('required|string|min:8')]
    public $password = '';

    #[Rule('required|string|email|max:255|unique:Luv2_user,email')]
    public $email = '';
    #[Rule('required|string|max:12|unique:Luv2_user,username')]
    public $username = '';

    #[On('refreshtable')]
    public function mount()
    {
        $this->dispatch('refreshjs');
        $this->users = User::where('company_code', auth()->user()->company_code)
            ->where('role', '!=', 'Admin')
            ->get();

        if ($this->users->isEmpty()) {
            $this->users = [];
        }
    }

    public function updateStatus($code)
    {
        $currentTime = Carbon::now();
        $users = User::where('company_code', auth()->user()->company_code)->where('code', $code)->first();

        if ($users->active === 'Y') {
            $users->update([
                'active' => 'N',
                'updated_at' => $currentTime,
            ]);
        } else {
            $users->update([
                'active' => 'Y',
                'updated_at' => $currentTime,
            ]);
        }
        $this->alert('success', 'Status changed succesfully', [
            'position' => 'top-end',
            'timer' => 5000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
        $this->dispatch('refreshtable');
    }


    #[On('editWhsCode')]
    public function editWhs($code)
    {
        $user = User::where('code', $code)->first();
        $this->code = $user->code;
        $this->whscode = $user->whs_code;
        $this->username = $user->username;
        $this->email = $user->email;
        $this->role = $user->role;
        $this->dispatch('openTabelEditWhs');
        $this->dispatch('refreshjs');
    }

    public function saveEdit()
    {
        try {
            $user = User::where('code', $this->code)->first();

            if ($user) {
                $user->whs_code = $this->whscode;
                $user->username = $this->username;
                $user->email = $this->email;
                $user->role = $this->role;

                $user->save();

                $this->alert('success', 'User information updated successfully', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);

                $this->dispatch('tocloseTabelEditWhs');
            } else {
                $this->alert('warning', 'User Not Found', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }
        } catch (\Exception $e) {
            $this->alert('error', 'An error occurred: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }


    public function createUser()
    {
        $this->validate();

        $this->code = Carbon::now()->format('ymdHis');

        try {
            DB::beginTransaction();

            $maxId = DB::table('Luv2_user')->max('id');


            DB::table('Luv2_user')->insert([
                'id' => $maxId + 1,
                'company_code' => auth()->user()->company_code,
                'whs_code' => $this->whscode,
                'code' => $this->code,
                'username' => $this->username,
                'email' => $this->email,
                'password' => Hash::make($this->password),
                'active' => 'Y',
                'role' => $this->role,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $menus = [
                'Home',
                'Cashier',
                'Member',
                'Master',
                'Opname',
                'Goods Receipt',
                'Goods Return',
                'Item Transfer'
            ];

            foreach ($menus as $menu) {
                DB::table('Luv2_tutorial')->insert([
                    'id_user' => $maxId + 1,
                    'menu' => $menu,
                    'active' => 'Y',
                ]);
            }

            DB::commit();

            $this->alert('success', 'New user created succesfully', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('closeTabelCreateUser');
            $this->dispatch('refreshtable');
        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('warning', 'Failed to create new user', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('closeTabelCreateUser');
            $this->dispatch('refreshtable');
        }

        $this->dispatch('tocloseTableEditWhs');
    }

    #[On('tocloseTabelEditWhs')]
    public function closeTabel()
    {
        $this->code = '';
        $this->whscode = '';
        $this->username = '';
        $this->email = '';
        $this->password = '';
        $this->dispatch('refreshjs');
        $this->dispatch('closeTabelEditWhs');
    }
    public function render()
    {
        $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();
        return view('livewire.user.user-table', [
            'warehouse' => $warehouse,
        ]);
    }
}
