<?php

namespace App\Livewire\Report;

use App\Exports\ExportTransfer;
use App\Exports\ExportTransferDetail;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;

class Transfer extends Component
{
    use LivewireAlert;

    public $focusedHeader = true;
    public $supplier;
    public $startDate;
    public $endDate;
    public $selectedDate;
    public $dataDetail;
    public $selectedWarehouse = null;
    public $selectedCodeWarehouse;
    public $selectedIndexWarehouse;
    public $searchwarehouse;
    public $warehouse;
    public $selectedWarehouseTo = null;
    public $selectedCodeWarehouseTo;
    public $selectedIndexWarehouseTo;
    public $searchwarehouseto;
    public $warehouseto;
    public $transfer;
    public $transferDet;

    public function mount()
    {
        $this->startDate = Carbon::now()->format('Y-m-d');
        $this->endDate = Carbon::now()->format('Y-m-d');
    }

    public function focusedHeaderTrue()
    {
        $this->focusedHeader = true;
        $this->dispatch('refreshjs');
    }

    public function focusedHeaderFalse()
    {
        $this->focusedHeader = false;
        $this->dispatch('refreshjs');
    }

    public function filters()
    {
        $this->transfer = DB::table('Luv2_transfer')
            ->leftJoin('Luv2_warehouse as whs', 'Luv2_transfer.whs_code', '=', 'whs.code')
            ->leftJoin('Luv2_warehouse as whs_to', 'Luv2_transfer.whs_code_to', '=', 'whs_to.code')
            ->where(function ($query) {
                $query->where('Luv2_transfer.status', '=', '2')
                    ->orWhere('Luv2_transfer.status', '=', '3');
            })
            ->where('Luv2_transfer.date', '>=', $this->startDate)
            ->where('Luv2_transfer.date', '<=', $this->endDate)
            ->when($this->selectedWarehouse, function ($query) {
                $query->where('Luv2_transfer.whs_code', $this->selectedWarehouse->code);
            })
            ->when($this->selectedWarehouseTo, function ($query) {
                $query->where('Luv2_transfer.whs_code_to', $this->selectedWarehouseTo->code);
            })
            ->select('Luv2_transfer.*', 'whs.name as whs_name', 'whs_to.name as whs_to_name')
            ->orderBy('Luv2_transfer.date', 'ASC')
            ->orderBy('Luv2_transfer.no', 'ASC')
            ->get();

        $this->transferDet = DB::table('Luv2_transfer')
            ->join('Luv2_transfer_detail', 'Luv2_transfer.no', '=', 'Luv2_transfer_detail.no')
            ->join('Luv2_item', 'Luv2_transfer_detail.item_code', '=', 'Luv2_item.code')
            ->leftJoin('Luv2_warehouse as whs', 'Luv2_transfer.whs_code', '=', 'whs.code')
            ->leftJoin('Luv2_warehouse as whs_to', 'Luv2_transfer.whs_code', '=', 'whs_to.code')
            ->where(function ($query) {
                $query->where('Luv2_transfer.status', '=', '2')
                    ->orWhere('Luv2_transfer.status', '=', '3');
            })
            ->where('Luv2_transfer.date', '>=', $this->startDate)
            ->where('Luv2_transfer.date', '<=', $this->endDate)
            ->when($this->selectedWarehouse, function ($query) {
                $query->where('Luv2_transfer.whs_code', $this->selectedWarehouse->code);
            })
            ->when($this->selectedWarehouseTo, function ($query) {
                $query->where('Luv2_transfer.whs_code_to', $this->selectedWarehouseTo->code);
            })
            ->select('Luv2_transfer.no', 'Luv2_transfer.date', 'Luv2_transfer_detail.linenum', 'Luv2_transfer_detail.qty_open', 'Luv2_transfer_detail.qty_close', 'Luv2_item.*', 'whs.name as whs_name', 'whs_to.name as whs_to_name')
            ->orderBy('Luv2_transfer.date', 'ASC')
            ->orderBy('Luv2_transfer.no', 'ASC')
            ->orderBy('Luv2_transfer_detail.linenum', 'ASC')
            ->get();

        $this->dispatch('refreshjs');
    }

    public function exportExcel()
    {
        if ($this->startDate === $this->endDate) {
            $filename = "Transfer Report - {$this->startDate}.xlsx";
        } else {
            $filename = "Transfer Report - {$this->startDate} - {$this->endDate}.xlsx";
        }

        $this->dispatch('refreshjs');

        return Excel::download(new ExportTransfer($this->transfer), $filename);
    }

    public function exportExcelDetail()
    {
        if ($this->startDate === $this->endDate) {
            $filename = "Transfer Detail Report - {$this->startDate}.xlsx";
        } else {
            $filename = "Transfer Detail Report - {$this->startDate} - {$this->endDate}.xlsx";
        }

        $this->dispatch('refreshjs');
        return Excel::download(new ExportTransferDetail($this->transferDet), $filename);
    }

    public function updated()
    {
        $this->dispatch('refreshjs');
    }

    public function render()
    {
        $resultwarehouse = [];
        $resultwarehouseto = [];

        if (strlen($this->searchwarehouse) >= 1) {
            $resultwarehouse = DB::table('Luv2_warehouse')->where('name', 'ILIKE', '%' . $this->searchwarehouse . '%')
                ->where('company_code', auth()->user()->company_code)
                ->limit(10)->get();
            $this->warehouse = $resultwarehouse;
        }

        if (strlen($this->searchwarehouseto) >= 1) {
            $resultwarehouseto = DB::table('Luv2_warehouse')->where('name', 'ILIKE', '%' . $this->searchwarehouseto . '%')
                ->where('company_code', auth()->user()->company_code)
                ->limit(10)->get();
            $this->warehouseto = $resultwarehouseto;
        }
        $user = Auth::user();
        if ($user->role != 'Admin') {
            $warehouse = DB::table('Luv2_warehouse')->where('code', $user->whs_code)
                ->where('company_code', auth()->user()->company_code)
                ->first();

            if ($warehouse) {

                $this->searchwarehouse = $warehouse->name;
                $this->selectedWarehouse = $warehouse;
            }
        }

        $this->transfer = DB::table('Luv2_transfer')
            ->leftJoin('Luv2_warehouse as whs', 'Luv2_transfer.whs_code', '=', 'whs.code')
            ->leftJoin('Luv2_warehouse as whs_to', 'Luv2_transfer.whs_code_to', '=', 'whs_to.code')
            ->where(function ($query) {
                $query->where('Luv2_transfer.status', '=', '2')
                    ->orWhere('Luv2_transfer.status', '=', '3');
            })
            ->where('Luv2_transfer.date', '>=', $this->startDate)
            ->where('Luv2_transfer.date', '<=', $this->endDate)
            ->when($this->selectedWarehouse, function ($query) {
                $query->where('Luv2_transfer.whs_code', $this->selectedWarehouse->code);
            })
            ->when($this->selectedWarehouseTo, function ($query) {
                $query->where('Luv2_transfer.whs_code_to', $this->selectedWarehouseTo->code);
            })
            ->select('Luv2_transfer.*', 'whs.name as whs_name', 'whs_to.name as whs_to_name')
            ->orderBy('Luv2_transfer.date', 'ASC')
            ->orderBy('Luv2_transfer.no', 'ASC')
            ->get();

        $this->transferDet = DB::table('Luv2_transfer')
            ->join('Luv2_transfer_detail', 'Luv2_transfer.no', '=', 'Luv2_transfer_detail.no')
            ->join('Luv2_item', 'Luv2_transfer_detail.item_code', '=', 'Luv2_item.code')
            ->leftJoin('Luv2_warehouse as whs', 'Luv2_transfer.whs_code', '=', 'whs.code')
            ->leftJoin('Luv2_warehouse as whs_to', 'Luv2_transfer.whs_code', '=', 'whs_to.code')
            ->where(function ($query) {
                $query->where('Luv2_transfer.status', '=', '2')
                    ->orWhere('Luv2_transfer.status', '=', '3');
            })
            ->where('Luv2_transfer.date', '>=', $this->startDate)
            ->where('Luv2_transfer.date', '<=', $this->endDate)
            ->when($this->selectedWarehouse, function ($query) {
                $query->where('Luv2_transfer.whs_code', $this->selectedWarehouse->code);
            })
            ->when($this->selectedWarehouseTo, function ($query) {
                $query->where('Luv2_transfer.whs_code_to', $this->selectedWarehouseTo->code);
            })
            ->select('Luv2_transfer.no', 'Luv2_transfer.date', 'Luv2_transfer_detail.linenum', 'Luv2_transfer_detail.qty_open', 'Luv2_transfer_detail.qty_close', 'Luv2_item.*', 'whs.name as whs_name', 'whs_to.name as whs_to_name')
            ->orderBy('Luv2_transfer.date', 'ASC')
            ->orderBy('Luv2_transfer.no', 'ASC')
            ->orderBy('Luv2_transfer_detail.linenum', 'ASC')
            ->get();

        return view('livewire.report.transfer', [
            'resultwarehouse' => $resultwarehouse,
            'resultwarehouseto' => $resultwarehouseto,
            'transfer' => $this->transfer,
            'transferDet' => $this->transferDet,
        ]);
    }

    public function clearfilters()
    {
        $this->searchwarehouse = '';
        $this->searchwarehouseto = '';

        $this->selectedWarehouse = null;
        $this->selectedWarehouseTo = null;

        $this->startDate = Carbon::now()->format('Y-m-d');
        $this->endDate = Carbon::now()->format('Y-m-d');

        $this->transfer = collect();
        $this->transferDet = collect();

        $this->dispatch('refreshjs');

    }


    public function selectWarehouse($index)
    {
        $this->selectedIndexWarehouse = $index;
        $this->selectedCodeWarehouse = $this->warehouse[$index]->code;

        $warehouse = DB::table('Luv2_warehouse')->where('code', $this->selectedCodeWarehouse)
            ->where('company_code', auth()->user()->company_code)
            ->first();

        if ($warehouse) {

            $this->searchwarehouse = $warehouse->name;
            $this->selectedWarehouse = $warehouse;

            $this->selectedCodeWarehouse = null;
            $this->selectedIndexWarehouse = null;

            $this->alert('info', 'Warehouse: ' . $warehouse->name, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshDatatable');
        }

        $this->dispatch('refreshjs');
    }

    public function selectWarehouseByClick($warehouseCode)
    {

        $warehouse = DB::table('Luv2_warehouse')->where('code', $warehouseCode)
            ->where('company_code', auth()->user()->company_code)
            ->first();

        if ($warehouse) {

            $this->searchwarehouse = $warehouse->name;
            $this->selectedWarehouse = $warehouse;

            $this->alert('info', 'Warehouse: ' . $warehouse->name, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshDatatable');
        }

        $this->dispatch('refreshjs');
    }

    public function updatedSearchwarehouse()
    {
        $this->selectedWarehouse = null;
        $this->dispatch('refreshjs');
    }

    public function selectWarehouseTo($index)
    {
        $this->selectedIndexWarehouseTo = $index;
        $this->selectedCodeWarehouseTo = $this->warehouseto[$index]->code;

        $warehouseto = DB::table('Luv2_warehouse')->where('code', $this->selectedCodeWarehouseTo)
            ->where('company_code', auth()->user()->company_code)
            ->first();

        if ($warehouseto) {

            $this->searchwarehouseto = $warehouseto->name;
            $this->selectedWarehouseTo = $warehouseto;

            $this->selectedCodeWarehouseTo = null;
            $this->selectedIndexWarehouseTo = null;

            $this->alert('info', 'To Warehouse: ' . $warehouseto->name, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshDatatable');
        }

        $this->dispatch('refreshjs');
    }

    public function selectWarehouseToByClick($warehouseToCode)
    {
        $warehouseto = DB::table('Luv2_warehouse')->where('code', $warehouseToCode)
            ->where('company_code', auth()->user()->company_code)
            ->first();

        if ($warehouseto) {

            $this->searchwarehouseto = $warehouseto->name;
            $this->selectedWarehouseTo = $warehouseto;

            $this->alert('info', 'To Warehouse: ' . $warehouseto->name, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshDatatable');
        }

        $this->dispatch('refreshjs');
    }

    public function updatedSearchwarehouseTo()
    {
        $this->selectedWarehouseTo = null;
        $this->dispatch('refreshjs');
    }
}
