<?php

namespace App\Livewire\Report;

use App\Exports\ExportMarginReport;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\On;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Warehouse;
use Illuminate\Support\Facades\Auth;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class ReportMargin extends Component
{
    use WithPagination;
    use LivewireAlert;

    protected $paginationTheme = 'bootstrap';

    public $page = 1;
    public $perPage = 10;
    public $search = '';
    public $sortDirection = 'DESC';
    public $sortColumn = 'created_at';

    public $startDate;
    public $endDate;
    public $warehouse;
    public $type = 'All';
    public $whs_code;

    protected $queryString = [
        'search' => ['except' => ''],
        'sortColumn' => ['except' => 'created_at'],
        'sortDirection' => ['except' => 'DESC'],
        'page' => ['except' => 1],
        'perPage' => ['except' => 10]
    ];

    public function mount()
    {
        $this->startDate = Carbon::now()->format('Y-m-d');
        $this->endDate = Carbon::now()->format('Y-m-d');
        if (auth()->user()->role == 'Admin') {
            $this->warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();
            $this->whs_code = Auth::user()->whs_code;
        }
    }

    public function render()
    {
        return view('livewire.report.report-margin', [
            'invoices' => $this->loadInvoices()
        ]);
    }

    public function loadInvoices()
    {
        $query = DB::table('Luv2_invoice')
            ->select([
                'Luv2_invoice.no',
                'Luv2_invoice.created_at', 
                'Luv2_invoice.must_paid as grandtotal',
                'Luv2_invoice.note',
                'Luv2_salesemp.name as sales_name',
                'Luv2_user.username as users_name',
                'Luv2_item.code as item_code',
                'Luv2_item.name as item_name',
                'Luv2_item.barcode as item_barcode',
                'Luv2_brand.name as item_brand',
                'Luv2_supplier.name as item_supp',
                'Luv2_item.avgprice',
                'Luv2_invoice_detail.price',
                'Luv2_invoice_detail.cost',
                'Luv2_invoice_detail.qty as total_qty',
                'Luv2_invoice_detail.total_price',
                'Luv2_brand.code as brand_code',
                'Luv2_supplier.code as supp_code',
                DB::raw('SUM(
                    CASE 
                        WHEN "Luv2_invoice_detail".consignment = \'Y\' THEN
                            ("Luv2_invoice_detail".total_price - ("Luv2_item".avgprice * "Luv2_invoice_detail".qty)) * 
                            ("Luv2_invoice_detail".shared_margin / 100)
                        ELSE 
                            "Luv2_invoice_detail".total_price - ("Luv2_item".avgprice * "Luv2_invoice_detail".qty)
                    END
                ) as margin')
            ])
            ->join('Luv2_invoice_detail', 'Luv2_invoice.no', '=', 'Luv2_invoice_detail.no')
            ->leftJoin('Luv2_item', function($join) {
                $join->on('Luv2_invoice_detail.code', '=', 'Luv2_item.code')
                    ->where('Luv2_item.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_supplier', function($join) {
                $join->on('Luv2_item.supp_code', '=', 'Luv2_supplier.code')
                    ->where('Luv2_supplier.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_brand', function($join) {
                $join->on('Luv2_item.brand_code', '=', 'Luv2_brand.code')
                    ->where('Luv2_brand.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_salesemp', function($join) {
                $join->on('Luv2_invoice.sales_code', '=', 'Luv2_salesemp.code')
                    ->where('Luv2_salesemp.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_user', function($join) {
                $join->on('Luv2_invoice.user_code', '=', 'Luv2_user.code')
                    ->where('Luv2_user.company_code', '=', auth()->user()->company_code);
            })
            ->where('Luv2_invoice.whs_code', auth()->user()->whs_code)
            ->where('Luv2_invoice.company_code', auth()->user()->company_code)
            ->whereBetween('Luv2_invoice.created_at', [
                $this->startDate . ' 00:00:00',
                $this->endDate . ' 23:59:59'
            ]);

        // Apply search filter if search term exists
        if ($this->search) {
            $query->where(function($q) {
                $q->where('Luv2_invoice.no', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.name', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.code', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.barcode', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_brand.name', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_supplier.name', 'ILIKE', '%' . $this->search . '%');
            });
        }

        $query->groupBy([
            'Luv2_invoice.no',
            'Luv2_invoice.created_at',
            'Luv2_invoice.must_paid',
            'Luv2_invoice.note',
            'Luv2_salesemp.name',
            'Luv2_user.username',
            'Luv2_item.code',
            'Luv2_item.name',
            'Luv2_item.barcode',
            'Luv2_brand.name',
            'Luv2_supplier.name',
            'Luv2_item.avgprice',
            'Luv2_invoice_detail.qty',
            'Luv2_invoice_detail.total_price',
            'Luv2_invoice_detail.price',
            'Luv2_invoice_detail.cost',
            'Luv2_brand.code',
            'Luv2_supplier.code'
        ]);

        // Apply sorting
        $query->orderBy($this->sortColumn, $this->sortDirection);

        return $query->paginate($this->perPage);
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'DESC') ? 'ASC' : 'DESC';
        } else {
            $this->sortColumn = $column;
            $this->sortDirection = 'ASC';
        }
    }

    public function exportExcel()
    {
        $currentDate = Carbon::now();
        $now = $currentDate->format('d-m-Y');
        
        // Get the base query without pagination
        $query = DB::table('Luv2_invoice')
            ->select([
                'Luv2_invoice.no',
                'Luv2_invoice.created_at', 
                'Luv2_invoice.must_paid as grandtotal',
                'Luv2_invoice.note',
                'Luv2_salesemp.name as sales_name',
                'Luv2_user.username as users_name',
                'Luv2_item.code as item_code',
                'Luv2_item.name as item_name',
                'Luv2_item.barcode as item_barcode',
                'Luv2_brand.name as item_brand',
                'Luv2_supplier.name as item_supp',
                'Luv2_item.avgprice',
                'Luv2_invoice_detail.price',
                'Luv2_invoice_detail.cost',
                'Luv2_invoice_detail.qty as total_qty',
                'Luv2_invoice_detail.total_price',
                'Luv2_brand.code as brand_code',
                'Luv2_supplier.code as supp_code',
                DB::raw('SUM(
                    CASE 
                        WHEN "Luv2_invoice_detail".consignment = \'Y\' THEN
                            ("Luv2_invoice_detail".total_price - ("Luv2_item".avgprice * "Luv2_invoice_detail".qty)) * 
                            ("Luv2_invoice_detail".shared_margin / 100)
                        ELSE 
                            "Luv2_invoice_detail".total_price - ("Luv2_item".avgprice * "Luv2_invoice_detail".qty)
                    END
                ) as margin')
            ])
            ->join('Luv2_invoice_detail', 'Luv2_invoice.no', '=', 'Luv2_invoice_detail.no')
            ->leftJoin('Luv2_item', function($join) {
                $join->on('Luv2_invoice_detail.code', '=', 'Luv2_item.code')
                    ->where('Luv2_item.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_supplier', function($join) {
                $join->on('Luv2_item.supp_code', '=', 'Luv2_supplier.code')
                    ->where('Luv2_supplier.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_brand', function($join) {
                $join->on('Luv2_item.brand_code', '=', 'Luv2_brand.code')
                    ->where('Luv2_brand.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_salesemp', function($join) {
                $join->on('Luv2_invoice.sales_code', '=', 'Luv2_salesemp.code')
                    ->where('Luv2_salesemp.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_user', function($join) {
                $join->on('Luv2_invoice.user_code', '=', 'Luv2_user.code')
                    ->where('Luv2_user.company_code', '=', auth()->user()->company_code);
            })
            ->where('Luv2_invoice.whs_code', auth()->user()->whs_code)
            ->where('Luv2_invoice.company_code', auth()->user()->company_code)
            ->whereBetween('Luv2_invoice.created_at', [
                $this->startDate . ' 00:00:00',
                $this->endDate . ' 23:59:59'
            ]);

        if ($this->search) {
            $query->where(function($q) {
                $q->where('Luv2_invoice.no', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.name', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.code', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.barcode', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_brand.name', 'ILIKE', '%' . $this->search . '%')
                  ->orWhere('Luv2_supplier.name', 'ILIKE', '%' . $this->search . '%');
            });
        }

        $query->groupBy([
            'Luv2_invoice.no',
            'Luv2_invoice.created_at',
            'Luv2_invoice.must_paid',
            'Luv2_invoice.note',
            'Luv2_salesemp.name',
            'Luv2_user.username',
            'Luv2_item.code',
            'Luv2_item.name',
            'Luv2_item.barcode',
            'Luv2_brand.name',
            'Luv2_supplier.name',
            'Luv2_item.avgprice',
            'Luv2_invoice_detail.qty',
            'Luv2_invoice_detail.total_price',
            'Luv2_invoice_detail.price',
            'Luv2_invoice_detail.cost',
            'Luv2_brand.code',
            'Luv2_supplier.code'
        ]);

        $query->orderBy($this->sortColumn, $this->sortDirection);
        
        $data = [
            'invoices' => $query->get(),
        ];
        
        if ($this->startDate === null) {
            $filename = "Margin Report - {$now}.xlsx";
        } else {
            $filename = "Margin Report - {$this->startDate} - {$this->endDate}.xlsx";
        }

        return Excel::download(new ExportMarginReport($data), $filename);
    }

    public function updatedWhsCode()
    {
        $user = auth()->user();
        DB::table('Luv2_user')
            ->where('id', $user->id)
            ->update(['whs_code' => $this->whs_code]);
            
        $this->resetPage();
    }

    #[On('refreshData')]
    public function refresh()
    {
        $this->resetPage();
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }
}
