<?php

namespace App\Livewire\Report\Analysis;

use App\Exports\ReportLossSales;
use App\Exports\ReportStockAlert;
use Carbon\Carbon;
use DateTime;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;

class StockAlertTable extends Component
{
    use LivewireAlert;
    use WithPagination;

    public $page = 1;
    public $perPage = 10;
    public $search = '';
    public $sortDirection = 'ASC';
    public $sortColumn = 'whs_name';
    public $id = '';
    public $sortedData = [];
    public $isLoading = false;
    public $transfersCreated = false;

    public $tableName = '';
    public $period;

    // Filter properties
    public $filters = [];
    public $filters_min = [];
    public $filters_max = [];
    public $filterOptions = [];
    public $selectAll = [];
    public $search_filters = [];

    public function mount()
    {
        $this->initializeFilters();
    }

    protected function initializeFilters()
    {
        $columns = [
            'whs_name' => 'Warehouse',
            'code' => 'Code',
            'name' => 'Name',
            'brand' => 'Brand',
            'category' => 'Category',
            'size' => 'Size',
            'onhand' => 'Stock',
            'sales' => 'Sales',
            'recommend' => 'Recommend',
            'info' => 'Info'
        ];

        foreach ($columns as $column => $label) {
            $this->filters[$column] = [];
            $this->selectAll[$column] = false;
            $this->search_filters[$column] = '';

            if (in_array($column, ['onhand', 'sales', 'recommend'])) {
                $this->filters_min[$column] = null;
                $this->filters_max[$column] = null;
            }
        }
    }

    public function updatedTableName()
    {
        $this->loadFilterOptions();
    }

    protected function loadFilterOptions()
    {
        if (!$this->tableName) return;

        try {
            // Load text column options
            $textColumns = ['whs_name', 'code', 'name', 'brand', 'category', 'size'];
            foreach ($textColumns as $column) {
                $this->filterOptions[$column] = DB::table($this->tableName)
                    ->select($column)
                    ->distinct()
                    ->whereNotNull($column)
                    ->where($column, '<>', '')
                    ->orderBy($column)
                    ->pluck($column)
                    ->toArray();
            }

            // Set info options
            $this->filterOptions['info'] = ['Transfer', 'Return'];

        } catch (Exception $e) {
            $this->alert('error', $e->getMessage());
        }
    }

    public function toggleSelectAll($column)
    {
        if ($this->selectAll[$column]) {
            $this->filters[$column] = [];
        } else {
            $this->filters[$column] = $this->filterOptions[$column] ?? [];
        }
        $this->selectAll[$column] = !$this->selectAll[$column];
    }

    public function clearFilter($column)
    {
        if (isset($this->filters_min[$column])) {
            $this->filters_min[$column] = null;
            $this->filters_max[$column] = null;
        } else {
            $this->filters[$column] = [];
            $this->selectAll[$column] = false;
        }
    }

    public function getFilteredData()
    {
        if (!$this->tableName) return [];

        $query = DB::table($this->tableName)
            ->where(function ($query) {
                $query->where('whs_code', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('whs_name', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('brand', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('category', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('size', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('code', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('barcode', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('name', 'ILIKE', '%' . $this->search . '%');
            })
            ->where('recommend', '<>', 0);

        // Apply text filters
        foreach (['whs_name', 'code', 'name', 'brand', 'category', 'size'] as $column) {
            if (!empty($this->filters[$column])) {
                $query->whereIn($column, $this->filters[$column]);
            }
        }

        // Apply numeric filters
        foreach (['onhand', 'sales', 'recommend'] as $column) {
            if (!is_null($this->filters_min[$column])) {
                $query->where($column, '>=', $this->filters_min[$column]);
            }
            if (!is_null($this->filters_max[$column])) {
                $query->where($column, '<=', $this->filters_max[$column]);
            }
        }

        // Apply info filter
        if (!empty($this->filters['info'])) {
            $query->where(function ($q) {
                foreach ($this->filters['info'] as $info) {
                    if ($info === 'Transfer') {
                        $q->orWhere('recommend', '>', 0);
                    } else if ($info === 'Return') {
                        $q->orWhere('recommend', '<', 0);
                    }
                }
            });
        }

        return $query->orderBy($this->sortColumn, $this->sortDirection)
            ->paginate($this->perPage);
    }

    public function getFilteredData2()
    {
        if (!$this->tableName) return [];

        $query = DB::table($this->tableName)
            ->where(function ($query) {
                $query->where('whs_code', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('whs_name', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('brand', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('category', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('size', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('code', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('barcode', 'ILIKE', '%' . $this->search . '%')
                    ->orWhere('name', 'ILIKE', '%' . $this->search . '%');
            })
            ->where('recommend', '<>', 0);

        // Apply text filters
        foreach (['whs_name', 'code', 'name', 'brand', 'category', 'size'] as $column) {
            if (!empty($this->filters[$column])) {
                $query->whereIn($column, $this->filters[$column]);
            }
        }

        // Apply numeric filters
        foreach (['onhand', 'sales', 'recommend'] as $column) {
            if (!is_null($this->filters_min[$column])) {
                $query->where($column, '>=', $this->filters_min[$column]);
            }
            if (!is_null($this->filters_max[$column])) {
                $query->where($column, '<=', $this->filters_max[$column]);
            }
        }

        // Apply info filter
        if (!empty($this->filters['info'])) {
            $query->where(function ($q) {
                foreach ($this->filters['info'] as $info) {
                    if ($info === 'Transfer') {
                        $q->orWhere('recommend', '>', 0);
                    } else if ($info === 'Return') {
                        $q->orWhere('recommend', '<', 0);
                    }
                }
            });
        }

        return $query->orderBy($this->sortColumn, $this->sortDirection)
            ->get();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    #[On('showTable')]
    public function showTable($tableName, $period)
    {
        $this->tableName = $tableName;
        $this->period = $period;
        $this->loadFilterOptions();
    }

    public function createTransferReturn()
    {
        $this->isLoading = true;

        try {
            if ($this->tableName != '') {
                $user = Auth::user();
                // $data = DB::table($this->tableName)
                //     ->where(function ($query) {
                //         $query->where('whs_code', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('whs_name', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('brand', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('category', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('size', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('code', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('barcode', 'ILIKE', '%' . $this->search . '%')
                //             ->orWhere('name', 'ILIKE', '%' . $this->search . '%');
                //     })
                //     ->where('recommend', '<>', 0)
                //     ->orderBy($this->sortColumn, $this->sortDirection)
                //     ->get();

                $data = $this->getFilteredData2();

                // Get initial counter value
                $baseCount = DB::table('Luv2_return')
                    ->where('user_code', $user->code)
                    ->whereDate('created_at', Carbon::today())
                    ->count();
                $currentCount = $baseCount;

                // Process return items
                $returnGroups = [];
                foreach ($data as $item) {
                    if ($item->recommend < 0) {
                        $whsCode = $item->whs_code;
                        if (!isset($returnGroups[$whsCode])) {
                            // Increment counter for each new header
                            $currentCount++;
                            $countgr = min($currentCount, 999);
                            $countgr = str_pad($countgr, 3, '0', STR_PAD_LEFT);
                            
                            // Generate unique document number
                            $huruf = 'IT';
                            $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');
                            $no = $huruf . $tanggal . $countgr;

                            $returnGroups[$whsCode] = [
                                'header' => [
                                    'company_code' => auth()->user()->company_code,
                                    'no' => $no,
                                    'date' => date('Y-m-d'),
                                    'whs_code' => $item->whs_code,
                                    'whs_code_to' => $item->whs_central,
                                    'remark' => 'Automatic Return from ARS',
                                    'qty' => 0,
                                    'qty_request' => 0,
                                    'qty_open' => 0,
                                    'total' => 0,
                                    'user_code' => auth()->user()->code,
                                    'status' => 5
                                ],
                                'details' => []
                            ];
                        }
                        $returnGroups[$whsCode]['details'][] = [
                            'company_code' => auth()->user()->company_code,
                            'no' => $returnGroups[$whsCode]['header']['no'],
                            'item_code' => $item->code,
                            'qty' => (int)abs($item->recommend),
                            'actualqty' => (int)abs($item->recommend),
                            'price' => (float)0,
                            'qty_unit' => (int)abs($item->recommend),
                            'unit' => 0,
                            'total' => (float)0,
                            'linenum' => count($returnGroups[$whsCode]['details']) + 1,
                            'qty_request' => (int)abs($item->recommend),
                            'created_at' => now(),
                            'updated_at' => now()
                        ];
                        $returnGroups[$whsCode]['header']['qty'] += abs($item->recommend);
                        $returnGroups[$whsCode]['header']['qty_request'] += abs($item->recommend);
                    }
                }
                $return = array_values($returnGroups);

                // Process transfer items
                $transferGroups = [];
                foreach ($data as $item) {
                    if ($item->recommend > 0) {
                        $whsCode = $item->whs_code;
                        if (!isset($transferGroups[$whsCode])) {
                            // Increment counter for each new header
                            $currentCount++;
                            $countgr = min($currentCount, 999);
                            $countgr = str_pad($countgr, 3, '0', STR_PAD_LEFT);
                            
                            // Generate unique document number
                            $huruf = 'IT';
                            $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');
                            $no = $huruf . $tanggal . $countgr;

                            $transferGroups[$whsCode] = [
                                'header' => [
                                    'company_code' => auth()->user()->company_code,
                                    'no' => $no,
                                    'date' => date('Y-m-d'),
                                    'whs_code' => $item->whs_central,
                                    'whs_code_to' => $item->whs_code,
                                    'remark' => 'Automatic Transfer from ARS',
                                    'qty' => 0,
                                    'qty_request' => 0,
                                    'qty_open' => 0,
                                    'total' => 0,
                                    'user_code' => auth()->user()->code,
                                    'status' => 1
                                ],
                                'details' => []
                            ];
                        }
                        $transferGroups[$whsCode]['details'][] = [
                            'company_code' => auth()->user()->company_code,
                            'no' => $transferGroups[$whsCode]['header']['no'],
                            'item_code' => $item->code,
                            'qty' => (int)$item->recommend,
                            'actualqty' => (int)$item->recommend,
                            'price' => (float)0,
                            'qty_unit' => (int)abs($item->recommend),
                            'unit' => 0,
                            'total' => (float)0,
                            'linenum' => count($transferGroups[$whsCode]['details']) + 1,
                            'qty_open' => (int)$item->recommend,
                            'created_at' => now(),
                            'updated_at' => now()
                        ];
                        $transferGroups[$whsCode]['header']['qty'] += $item->recommend;
                        $transferGroups[$whsCode]['header']['qty_open'] += $item->recommend;
                    }
                }
                $transfer = array_values($transferGroups);

                $this->sortedData = [
                    'return' => $return,
                    'transfer' => $transfer
                ];

                // Save the sorted data to database
                $this->saveSortedDataToDatabase();
            }
        } catch (Exception $ex) {
            Log::error('Error in createTransferReturn: ' . $ex->getMessage(), [
                'exception' => [
                    'message' => $ex->getMessage(),
                    'file' => $ex->getFile(),
                    'line' => $ex->getLine(),
                    'trace' => $ex->getTraceAsString()
                ],
                'context' => [
                    'user_id' => auth()->id(),
                    'company_code' => auth()->user()->company_code,
                    'table_name' => $this->tableName,
                    'search' => $this->search,
                    'sort_column' => $this->sortColumn,
                    'sort_direction' => $this->sortDirection
                ]
            ]);

            $message = $this->getReadableErrorMessage($ex->getMessage());
            $this->alert('error', $message, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }

        $this->isLoading = false;
    }

    private function getReadableErrorMessage($message)
    {
        if (strpos($message, 'reset') !== false) {
            return $message . ", silakan coba lagi";
        }

        if (strpos($message, '22P02') !== false) {
            return "Terjadi kesalahan format data. Silakan hubungi administrator.";
        }

        if (strpos($message, 'duplicate key') !== false) {
            return "Dokumen sudah pernah dibuat sebelumnya.";
        }

        // If no specific pattern matches, return a generic message
        return "Terjadi kesalahan: " . $message;
    }

    public function render()
    {
        $data = $this->getFilteredData();
        return view('livewire.report.analysis.stock-alert-table', compact('data'));
    }

    public function exportData()
    {
        try {
            // $data = DB::table($this->tableName)
            //     ->where(function ($query) {
            //         $query->where('whs_code', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('whs_name', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('brand', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('category', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('size', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('code', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('barcode', 'ILIKE', '%' . $this->search . '%')
            //             ->orWhere('name', 'ILIKE', '%' . $this->search . '%');
            //     })
            //     ->where('recommend', '<>', 0)
            //     ->orderBy('whs_name', 'ASC')
            //     ->orderBy('recommend', 'DESC')
            //     ->get();
            $data = $this->getFilteredData2();
            $filename = "Automatic Restocking System (ARS).xlsx";
            $period = DateTime::createFromFormat('Y-m-d', $this->period)->format('d M Y');

            return Excel::download(new ReportStockAlert($data, $period), $filename);
        } catch (Exception $ex) {
            $message = strpos($ex->getMessage(), 'reset') !== false ? $ex->getMessage() . ", try again" : $ex->getMessage();
            $this->alert('error', $message, [
                'position' => 'top-end',
                'timer' => 2500,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function saveSortedDataToDatabase()
    {
        try {
            DB::beginTransaction();

            if (empty($this->sortedData['return']) && empty($this->sortedData['transfer'])) {
                Log::warning('No data to save', [
                    'user_id' => auth()->id(),
                    'company_code' => auth()->user()->company_code
                ]);
                throw new Exception('Tidak ada data untuk disimpan');
            }

            Log::info('Starting database transaction');

            // Process returns
            foreach ($this->sortedData['return'] as $returnGroup) {
                // Insert header
                DB::table('Luv2_transfer')->insert($returnGroup['header']);
                Log::info('Inserted return header', [
                    'no' => $returnGroup['header']['no'],
                    'whs_code' => $returnGroup['header']['whs_code']
                ]);

                // Insert details
                DB::table('Luv2_transfer_detail')->insert($returnGroup['details']);
                Log::info('Inserted return details', [
                    'no' => $returnGroup['header']['no'],
                    'detail_count' => count($returnGroup['details'])
                ]);
            }

            // Process transfers
            foreach ($this->sortedData['transfer'] as $transferGroup) {
                // Insert header
                DB::table('Luv2_transfer')->insert($transferGroup['header']);
                Log::info('Inserted transfer header', [
                    'no' => $transferGroup['header']['no'],
                    'whs_code' => $transferGroup['header']['whs_code']
                ]);

                // Insert details
                DB::table('Luv2_transfer_detail')->insert($transferGroup['details']);
                Log::info('Inserted transfer details', [
                    'no' => $transferGroup['header']['no'],
                    'detail_count' => count($transferGroup['details'])
                ]);
            }

            DB::commit();
            Log::info('Database transaction committed successfully');

            $this->transfersCreated = true;

            $this->alert('success', 'Transfer and Return created successfully. will redirect to transfer page in 2 seconds', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->dispatch('returnToTransferPage');

        } catch (Exception $e) {
            DB::rollback();
            Log::error('Error in saveSortedDataToDatabase: ' . $e->getMessage(), [
                'exception' => [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'trace' => $e->getTraceAsString()
                ],
                'context' => [
                    'user_id' => auth()->id(),
                    'company_code' => auth()->user()->company_code,
                    'data_summary' => [
                        'return_count' => count($this->sortedData['return'] ?? []),
                        'transfer_count' => count($this->sortedData['transfer'] ?? [])
                    ]
                ]
            ]);
            Log::info('Database transaction rolled back');
            throw $e;
        }
    }

    #[On('returnToTransferPage')]
    public function returnToTransferPage()
    {
        sleep(2);
        return redirect()->route('item-transfer.index');
        
    }

    public function updatedSelectAll($value, $column)
    {
        if ($value) {
            if ($column === 'info') {
                $this->filters[$column] = ['Transfer', 'Return'];
            } else {
                $this->filters[$column] = $this->filterOptions[$column] ?? [];
            }
        } else {
            $this->filters[$column] = [];
        }
    }
}
