<?php

namespace App\Livewire\Report\Analysis;

use App\Exports\ReportLossSales;
use DateTime;
use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;

class LossSalesTable extends Component
{
    use LivewireAlert;
    use WithPagination;

    public $page = 1;
    public $perPage = 10;
    public $search = '';
    public $sortDirection = 'DESC';
    public $sortColumn = 'losssales';
    public $id = '';

    public $tableName = '';
    public $total_losssales;
    public $dates = [];
    public $from_date;
    public $to_date;

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    #[On('showTable')]
    public function showTable($tableName, $total_losssales, $dates, $from_date, $to_date)
    {
        $this->tableName = $tableName;
        $this->total_losssales = $total_losssales;
        $this->dates = $dates;
        $this->from_date = $from_date;
        $this->to_date = $to_date;

        $this->render();
    }

    public function render()
    {
        $data = [];

        if ($this->tableName != '') {
            try {
                $data = DB::table($this->tableName)
                    
                    ->where(function ($query) {
                        $query->where('whs', 'ILIKE', '%' . $this->search . '%')
                            ->orWhere('code', 'ILIKE', '%' . $this->search . '%')
                            ->orWhere('barcode', 'ILIKE', '%' . $this->search . '%')
                            ->orWhere('name', 'ILIKE', '%' . $this->search . '%');
                    })
                    ->orderBy('whs', 'asc')
                    ->orderBy($this->sortColumn, $this->sortDirection)
                    ->paginate($this->perPage);
            } catch (Exception $ex) {
                $message = strpos($ex->getMessage(), 'reset') !== false ? $ex->getMessage() . ", try again" : $ex->getMessage();
                $this->alert('error', $message, [
                    'position' => 'top-end',
                    'timer' => 2500,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }
        }

        return view('livewire.report.analysis.loss-sales-table', compact('data'));
    }

    public function exportData()
    {
        try {
            $data = DB::table($this->tableName)
                ->where('losssales', '>', 0)
                ->where(function ($query) {
                    $query->where('whs', 'ILIKE', '%' . $this->search . '%')
                        ->orWhere('code', 'ILIKE', '%' . $this->search . '%')
                        ->orWhere('barcode', 'ILIKE', '%' . $this->search . '%')
                        ->orWhere('name', 'ILIKE', '%' . $this->search . '%');
                })
                ->orderBy('whs', 'asc')
                ->orderBy($this->sortColumn, $this->sortDirection)
                ->get();


            $fromDate = DateTime::createFromFormat('Y-m-d', $this->from_date)->format('d M Y');
            $toDate = DateTime::createFromFormat('Y-m-d', $this->to_date)->format('d M Y');

            $period = $this->from_date == $this->to_date ? $fromDate : $fromDate . ' - ' . $toDate;
            $filename = "Loss Sales Report {$period}.xlsx";

            return Excel::download(new ReportLossSales($data, $this->dates, $period), $filename);
        } catch (Exception $ex) {
            $message = strpos($ex->getMessage(), 'reset') !== false ? $ex->getMessage() . ", try again" : $ex->getMessage();
            $this->alert('error', $message, [
                'position' => 'top-end',
                'timer' => 2500,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }
}
