<?php

namespace App\Livewire\Pos\Item;

use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\WithPagination;

class InsertUnitItem extends Component
{
    use LivewireAlert;
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $warehouses;
    public $whs_code;
    public $params;
    public $item_code = '';
    public $qty = '';
    public $pricesell = '';
    public $selectedUnitCode = '';
    public $selectedUnitBarcode = '';
    public $selectedUnitName = '';
    public $selectedUnitId = '';

    public $perPage = 5;
    public $search = '';
    public $sortColumn = 'item_code';
    public $sortDirection = 'ASC';

    protected $listeners = [
        'closeModal' => 'clearModal'
    ];

    protected $queryString = [
        'search' => ['except' => ''],
        'sortColumn' => ['except' => 'item_code'],
        'sortDirection' => ['except' => 'ASC']
    ];

    public function clearModal()
    {
        $this->reset([
            'selectedUnitCode',
            'selectedUnitBarcode',
            'selectedUnitName',
            'selectedUnitId',
            'qty',
            'pricesell',
            'item_code'
        ]);
        $this->resetValidation();
    }

    public function closeModal()
    {
        $this->clearModal();
    }

    protected function rules()
    {
        return [
            'item_code' => [
                'required',
                'string',
                'max:100',
                function ($attribute, $value, $fail) {
                    $existsInItemUnit = DB::table('Luv2_item_unit')
                        ->where('item_code', $value)
                        ->where('id_unit', $this->params)
                        ->where('company_code', auth()->user()->company_code)
                        ->where('whs_code', auth()->user()->whs_code)
                        ->exists();

                    if ($existsInItemUnit) {
                        $fail("Unit with this item code already exists in this warehouse.");
                    }

                    $existsInItem = DB::table('Luv2_item')
                        ->where('code', $value)
                        ->where('company_code', auth()->user()->company_code)
                        ->doesntExist();

                    if ($existsInItem) {
                        $fail("The item code does not exist.");
                    }
                },
            ],
            'qty' => 'required|numeric|min:1',
            'pricesell' => 'required|numeric|min:0',
        ];
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function render()
    {
        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)
            ->get();

        if (Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        }

        $query = DB::table('Luv2_item_unit')
            ->where('Luv2_item_unit.whs_code', auth()->user()->whs_code)
            ->join('Luv2_item', 'Luv2_item_unit.item_code', '=', 'Luv2_item.code')
            ->where('Luv2_item_unit.id_unit', $this->params)
            ->where('Luv2_item_unit.company_code', auth()->user()->company_code)
            ->select('Luv2_item_unit.*', 'Luv2_item.name', 'Luv2_item.barcode');

        // Apply search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('Luv2_item_unit.item_code', 'like', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.name', 'like', '%' . $this->search . '%')
                  ->orWhere('Luv2_item.barcode', 'like', '%' . $this->search . '%');
            });
        }

        // Apply sorting
        $query->orderBy($this->sortColumn, $this->sortDirection);
        
        return view('livewire.pos.item.insert-unit-item', [
            'itemUnit' => $query->paginate($this->perPage)
        ]);
    }

    #[On('updatedWhsUnit')]
    public function mount($param)
    {
        $this->params = $param;
    }

    public function saveUnitItem()
    {
        $this->validate();

        DB::table('Luv2_item_unit')->insert([
            'company_code' => Auth::user()->company_code,
            'whs_code' => auth()->user()->whs_code,
            'id_unit' => $this->params,
            'item_code' => $this->item_code,
            'qty' => $this->qty,
            'pricesell' => $this->pricesell,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        $this->alert('success', 'Data unit item successfully saved', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
        
        $this->clearModal();
        $this->dispatch('closemodalCreate');
        $this->dispatch('updatedWhsUnit', $this->params);
    }

    #[On('editUnitItem')]
    public function editUnitItem($code, $unitcode)
    {
        $item = DB::table('Luv2_item_unit')
            ->where('Luv2_item_unit.whs_code', auth()->user()->whs_code)
            ->where('Luv2_item_unit.item_code', $code)
            ->join('Luv2_item', 'Luv2_item_unit.item_code', '=', 'Luv2_item.code')
            ->where('Luv2_item_unit.id_unit', $unitcode)
            ->where('Luv2_item_unit.company_code', auth()->user()->company_code)
            ->select('Luv2_item_unit.*', 'Luv2_item.name', 'Luv2_item.barcode')
            ->first();

        $this->selectedUnitCode = $item->item_code;
        $this->selectedUnitBarcode = $item->barcode;
        $this->selectedUnitName = $item->name;
        $this->selectedUnitId = $item->id_unit;
        $this->qty = $item->qty;
        $this->pricesell = $item->pricesell;

        $this->dispatch('openModalEdit');
    }

    public function saveUpdate()
    {
        DB::table('Luv2_item_unit')
            ->where('company_code', auth()->user()->company_code)
            ->where('whs_code', auth()->user()->whs_code)
            ->where('item_code', $this->selectedUnitCode)
            ->where('id_unit', $this->selectedUnitId) 
            ->update([
                'qty' => $this->qty,
                'pricesell' => $this->pricesell,
                'updated_at' => now(),
            ]);

        $this->clearModal();

        $this->alert('success', 'Data unit item successfully updated', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
        
        $this->dispatch('closeModalEdit'); 
        $this->dispatch('updatedWhsUnit', $this->params);
    }
}
