<?php

namespace App\Livewire\Pos\Item;

use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\WithPagination;

class InsertUnit extends Component
{
    use LivewireAlert;
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $perPage = 5;
    public $search = '';
    public $sortColumn = 'name';
    public $sortDirection = 'ASC';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortColumn' => ['except' => 'name'],
        'sortDirection' => ['except' => 'ASC']
    ];

    #[Rule('required', message: 'Name field is required.')]
    #[Rule('min:4', message: 'Name field must be at least 4 characters.')]
    public string $name = '';

    public $warehouses;
    public $whs_code;

    private function getNextId()
    {
        $lastUnit = DB::table('Luv2_unit')
            ->where('company_code', Auth::user()->company_code)
            ->orderBy('id', 'desc')
            ->first();

        return $lastUnit ? $lastUnit->id + 1 : 1;
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function render()
    {
        $query = DB::table('Luv2_unit')
            ->where('company_code', Auth::user()->company_code);

        // Apply search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%');
            });
        }

        // Apply sorting
        $query->orderBy($this->sortColumn, $this->sortDirection);
        
        // Get paginated results
        $units = $query->paginate($this->perPage);

        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        if(Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        }

        return view('livewire.pos.item.insert-unit', [
            'units' => $units
        ]);
    }

    public function save()
    {
        $this->validate();

        $nextId = $this->getNextId();

        DB::table('Luv2_unit')->insert([
            'id' => $nextId,
            'company_code' => Auth::user()->company_code,
            'name' => $this->name,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        // $this->alert('success', 'Data Unit berhasil disimpan', [
        //     'position' => 'top-end',
        //     'timer' => 3000,
        //     'toast' => true,
        //     'timerProgressBar' => true
        // ]);
        
        $this->dispatch('closeModal');
        $this->dispatch('afterSave');
    }

    #[On('closeModal')]
    public function closeModal()
    {
        $this->name = '';
        $this->dispatch('closemodalCreate');
        
        $this->alert('success', 'Data Unit berhasil disimpan', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
    }

    #[On('afterSave')]
    public function dataUnitafterSave()
    {
        $this->resetPage();
        $this->dispatch('refreshDatatable');
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }
}
