<?php

namespace App\Livewire\Pos\Item;

use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;

class InsertItemPricelist extends Component
{

    use WithFileUploads;
    use LivewireAlert;
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $perPage = 10;
    public $search = '';
    public $sortColumn = 'item_code';
    public $sortDirection = 'ASC';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortColumn' => ['except' => 'item_code'],
        'sortDirection' => ['except' => 'ASC']
    ];

    #[Rule('required', message: 'Item Code field is required.')]
    #[Rule('min:3', message: 'Item Code field must be at least 3 characters.')]
    public string $item_code;

    #[Rule('required', message: 'Pricelist Code field is required.')]
    #[Rule('min:3', message: 'Pricelist Code field must be at least 3 characters.')]
    public string $pricelist_code;


    #[Rule('required', message: 'Price field is required.')]
    public $price;

    public $file;
    public $chunks;
    public $editChunks = false;
    public $errMsg = '';
    public $isTable = true;
    public $itemPricelist;
    public $Pricelist;

    public $selectedItemCode = '';
    public $selectedItemPricelistCode = '';
    public $selectedItemPrice = '';

    // #[Rule('numeric', message: 'Price field must be a numeric value.')]
    public $pricenew;

    public $warehouses;
    #[Rule('required', message: 'Warehouse field is required.')]
    public $whs_code;
    public $param;
    public $searchitem = "";
    public $item;
    public $selectedIndexItem = 0;
    public $selectedCodeItem = '';
    public $selectedItem;



    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function render()
    {
        $resultitem = [];

        if (strlen($this->searchitem) >= 1) {
            $resultitem = DB::table('Luv2_item')
                ->where(function ($query) {
                    $query->where('barcode', 'LIKE', '%' . $this->searchitem . '%')
                        ->orWhere('barcode', 'ILIKE', '%' . $this->searchitem . '%')
                        ->orWhere('code', 'LIKE', '%' . $this->searchitem . '%')
                        ->orWhere('code', 'ILIKE', '%' . $this->searchitem . '%');
                })
                ->limit(10)
                ->where('company_code', auth()->user()->company_code)
                ->get();
            $this->item = $resultitem;
        }

        $query = DB::table('Luv2_item_pricelist as ip')
            ->join('Luv2_item as i', function ($join) {
                $join->on('ip.item_code', '=', 'i.code')
                    ->where('ip.company_code', '=', Auth::user()->company_code)
                    ->where('i.company_code', '=', Auth::user()->company_code);
            })
            ->select('ip.*', 'i.name as item_name')
            ->where('ip.company_code', Auth::user()->company_code)
            ->where('ip.pricelist_code', $this->pricelist_code);
            

        // Apply search filter
        if ($this->search) {
            $query->where(function ($q) {
                $q->where('ip.item_code', 'like', '%' . $this->search . '%')
                    ->orWhere('i.name', 'like', '%' . $this->search . '%');
            });
        }

        // Apply sorting
        $query->orderBy($this->sortColumn, $this->sortDirection);

        // Get paginated results
        $itemPricelists = $query->paginate($this->perPage);

        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        return view('livewire.pos.item.insert-item-pricelist', [
            'resultitem' => $resultitem,
            'itemPricelists' => $itemPricelists
        ]);
    }

    public function selectItem($index)
    {
        $this->selectedIndexItem = $index;
        $this->selectedCodeItem = $this->item[$index]->code;

        $item = DB::table('Luv2_item')->where('code', $this->selectedCodeItem)
            ->where('company_code', auth()->user()->company_code)
            ->first();
        if ($item) {
            $this->searchitem = $item->name;
            $this->item_code = $item->code;
            $this->selectedItem = $item;
        }

        $this->alert('info', 'Item: ' . $item->name, [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
    }

    public function selectItemByClick($itemCode)
    {
        $item = DB::table('Luv2_item')->where('code', $itemCode)
            ->where('company_code', auth()->user()->company_code)
            ->first();
        if ($item) {
            $this->searchitem = $item->name;
            $this->item_code = $item->code;
            $this->selectedItem = $item;
        }
        $this->alert('info', 'Item: ' . $item->name, [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
    }

    public function updatedSearchitem()
    {
        $this->selectedItem = null;
        $this->item_code = '';
    }


    // public function importItem()
    // {
    //     $data = $this->validate([
    //         'file' => 'required|mimes:csv,xlsx',
    //     ]);

    //     try {
    //         // Determine if the file is CSV or XLSX
    //         $extension = $this->file->getClientOriginalExtension();

    //         if ($extension === 'csv') {
    //             // Handle CSV file
    //             $file = fopen($this->file->getRealPath(), 'r');
    //             $skipFirstRow = true;
    //             $chunks = [];

    //             while (($data = fgetcsv($file)) !== false) {
    //                 if ($skipFirstRow) {
    //                     $skipFirstRow = false;
    //                     continue;
    //                 }

    //                 $chunks[] = [
    //                     'company_code' => Auth::user()->company_code,
    //                     'item_code' => $data[0],
    //                     'pricelist_code' => $data[1],
    //                     'price' => $data[2],
    //                     'id_user' => Auth::user()->id,
    //                     'created_at' => Carbon::now(),
    //                     'updated_at' => Carbon::now(),
    //                     'whs_code' => Auth::user()->whs_code,
    //                 ];
    //             }
    //             fclose($file);
    //         } elseif ($extension === 'xlsx') {
    //             $chunks = Excel::toArray([], $this->file)[0]; // Extract the first sheet

    //             unset($chunks[0]);

    //             // Transform the data
    //             $chunks = array_map(function ($row) {
    //                 return [
    //                     'company_code' => Auth::user()->company_code,
    //                     'item_code' => $row[0],
    //                     'pricelist_code' => $row[1],
    //                     'price' => $row[2],
    //                     'id_user' => Auth::user()->id,
    //                     'created_at' => Carbon::now(),
    //                     'updated_at' => Carbon::now(),
    //                     'whs_code' => Auth::user()->whs_code,

    //                 ];
    //             }, $chunks);
    //         }

    //         $this->chunks = $chunks;

    //         $this->dispatch('refreshjs');
    //     } catch (\Throwable $th) {
    //         dd($th);

    //         $this->alert('warning', 'Error, check the template!', [
    //             'position' => 'top-end',
    //             'timer' => 3000,
    //             'toast' => true,
    //             'timerProgressBar' => true,
    //         ]);
    //     }
    // }

    public function importItem()
    {
        $data = $this->validate([
            'file' => 'required|file|mimes:csv,xlsx,txt',
        ]);

        try {
            $fileExtension = $this->file->getClientOriginalExtension();

            if ($fileExtension === 'csv' || $fileExtension === 'txt') {
                $this->importCSV();
            } elseif ($fileExtension === 'xlsx') {
                $this->importXLSX();
            } else {
                $this->alert('warning', 'Format file tidak didukung', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }
        } catch (\Throwable $th) {
            $this->alert('warning', 'Error, Format template salah!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function importCSV()
    {
        try {
            $file = fopen($this->file->getRealPath(), 'r');
            $skipFirstRow = true;
            $chunks = [];
            $rowNumber = 1;

            while (($data = fgetcsv($file)) !== false) {
                // Filter out empty values and whitespace
                $data = array_map('trim', $data);
                $data = array_filter($data, function ($value) {
                    return $value !== '' && $value !== null;
                });

                if ($skipFirstRow) {
                    // Validate header row
                    if (
                        count($data) < 3 ||
                        !in_array(strtolower(trim($data[0])), ['item code', 'itemcode', 'item_code']) ||
                        !in_array(strtolower(trim($data[1])), ['pricelist code', 'pricelistcode', 'pricelist_code']) ||
                        !in_array(strtolower(trim($data[2])), ['price'])
                    ) {
                        fclose($file);
                        throw new \Exception('Invalid header format. Expected: Item Code, Pricelist Code, Price');
                    }
                    $skipFirstRow = false;
                    continue;
                }

                $rowNumber++;

                // Skip completely empty rows
                if (empty($data)) {
                    continue;
                }

                // Validate row data
                if (count($data) < 3) {
                    $actualColumns = count($data);
                    fclose($file);
                    throw new \Exception("Row $rowNumber has only $actualColumns column(s) but needs 3 columns (Item Code, Pricelist Code, Price). Please check for missing data.");
                }

                // Validate price format
                $price = trim($data[2]);
                if (!is_numeric($price)) {
                    fclose($file);
                    throw new \Exception("Invalid price format in row $rowNumber. Found '$price' but expected a number.");
                }

                // Validate required fields are not empty
                if (empty($data[0]) || empty($data[1])) {
                    fclose($file);
                    throw new \Exception("Row $rowNumber has empty Item Code or Pricelist Code. Both fields are required.");
                }

                $chunks[] = [
                    'company_code' => Auth::user()->company_code,
                    'item_code' => $data[0],
                    'pricelist_code' => $data[1],
                    'price' => (float) $price,
                    'id_user' => Auth::user()->id,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ];
            }

            fclose($file);

            if (empty($chunks)) {
                throw new \Exception('No valid data found in the file. Please check the file format and content.');
            }

            $this->chunks = $chunks;

            // Check for duplicates after import
            $duplicateInfo = $this->checkForDuplicates($chunks);
            if ($duplicateInfo['hasDuplicates']) {
                $message = $this->buildDuplicateMessage($duplicateInfo);
                $this->alert('warning', $message, [
                    'position' => 'top-end',
                    'timer' => 7000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }
        } catch (\Exception $e) {
            $this->alert('error', 'Error: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 7000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->chunks = [];
        }
    }

    private function importXLSX()
    {
        try {
            $rows = Excel::toArray([], $this->file->getRealPath())[0];
            if (empty($rows)) {
                throw new \Exception('File is empty. Please add data to import.');
            }

            // Validate header row
            $header = array_map('strtolower', array_map('trim', $rows[0]));
            if (
                count($header) < 3 ||
                !in_array('item code', $header) && !in_array('itemcode', $header) && !in_array('item_code', $header) ||
                !in_array('pricelist code', $header) && !in_array('pricelistcode', $header) && !in_array('pricelist_code', $header) ||
                !in_array('price', $header)
            ) {
                throw new \Exception('Invalid header format. Expected: Item Code, Pricelist Code, Price');
            }

            $chunks = [];
            $rowNumber = 1;

            foreach (array_slice($rows, 1) as $row) {
                $rowNumber++;

                // Filter out empty values and whitespace
                $row = array_map('trim', $row);
                $row = array_filter($row, function ($value) {
                    return $value !== '' && $value !== null;
                });

                // Skip completely empty rows
                if (empty($row)) {
                    continue;
                }

                // Validate row data
                if (count($row) < 3) {
                    $actualColumns = count($row);
                    throw new \Exception("Row $rowNumber has only $actualColumns column(s) but needs 3 columns (Item Code, Pricelist Code, Price). Please check for missing data.");
                }

                // Reset array keys after filtering
                $row = array_values($row);

                // Validate price format
                $price = trim($row[2]);
                if (!is_numeric($price)) {
                    throw new \Exception("Invalid price format in row $rowNumber. Found '$price' but expected a number.");
                }

                // Validate required fields are not empty
                if (empty($row[0]) || empty($row[1])) {
                    throw new \Exception("Row $rowNumber has empty Item Code or Pricelist Code. Both fields are required.");
                }

                $chunks[] = [
                    'company_code' => Auth::user()->company_code,
                    'item_code' => $row[0],
                    'pricelist_code' => $row[1],
                    'price' => (float) $price,
                    'id_user' => Auth::user()->id,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ];
            }

            if (empty($chunks)) {
                throw new \Exception('No valid data found in the file. Please check the file format and content.');
            }

            $this->chunks = $chunks;

            // Check for duplicates after import
            $duplicateInfo = $this->checkForDuplicates($chunks);
            if ($duplicateInfo['hasDuplicates']) {
                $message = $this->buildDuplicateMessage($duplicateInfo);
                $this->alert('warning', $message, [
                    'position' => 'top-end',
                    'timer' => 7000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }
        } catch (\Exception $e) {
            $this->alert('error', 'Error: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 7000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->chunks = [];
        }
    }

    private function checkForDuplicates($data)
    {
        // Check for duplicates within the imported data
        $compositeKeys = array_map(function ($item) {
            return $item['item_code'] . '|' . $item['pricelist_code'];
        }, $data);
        $duplicateKeys = array_filter(array_count_values($compositeKeys), function ($count) {
            return $count > 1;
        });

        // Check for duplicates in the database
        $existingItems = DB::table('Luv2_item_pricelist')
            ->where('company_code', Auth::user()->company_code)
            ->whereIn('item_code', array_column($data, 'item_code'))
            ->whereIn('pricelist_code', array_column($data, 'pricelist_code'))
            ->get();

        $existingKeys = [];
        foreach ($existingItems as $item) {
            $existingKeys[] = $item->item_code . '|' . $item->pricelist_code;
        }

        return [
            'hasDuplicates' => !empty($duplicateKeys) || !empty($existingKeys),
            'importDuplicates' => array_keys($duplicateKeys),
            'databaseDuplicates' => $existingKeys
        ];
    }

    private function buildDuplicateMessage($duplicateInfo)
    {
        $message = 'Duplicate entries found:';

        if (!empty($duplicateInfo['importDuplicates'])) {
            $message .= "\nDuplicate entries in import file: " . implode(', ', array_map(function ($key) {
                list($itemCode, $pricelistCode) = explode('|', $key);
                return "Item: $itemCode, Pricelist: $pricelistCode";
            }, $duplicateInfo['importDuplicates']));
        }

        if (!empty($duplicateInfo['databaseDuplicates'])) {
            $message .= "\nEntries already in database: " . implode(', ', array_map(function ($key) {
                list($itemCode, $pricelistCode) = explode('|', $key);
                return "Item: $itemCode, Pricelist: $pricelistCode";
            }, $duplicateInfo['databaseDuplicates']));
        }

        $message .= "\nYou can export the duplicates for review.";

        return $message;
    }

    public function getHasDuplicatesProperty()
    {
        if (empty($this->chunks)) {
            return false;
        }
        $duplicateInfo = $this->checkForDuplicates($this->chunks);
        return $duplicateInfo['hasDuplicates'];
    }

    public function downloadDuplicates($format = 'xlsx')
    {
        try {
            if (empty($this->chunks)) {
                $this->alert('warning', 'No data available for export', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            $duplicateInfo = $this->checkForDuplicates($this->chunks);

            if (!$duplicateInfo['hasDuplicates']) {
                $this->alert('info', 'No duplicates found to export', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            // Collect all duplicate items
            $duplicateItems = [];

            // Get items with duplicate codes in import file
            foreach ($duplicateInfo['importDuplicates'] as $compositeKey) {
                list($itemCode, $pricelistCode) = explode('|', $compositeKey);
                foreach ($this->chunks as $item) {
                    if ($item['item_code'] === $itemCode && $item['pricelist_code'] === $pricelistCode) {
                        $duplicateItems[] = [
                            'item_code' => $item['item_code'],
                            'pricelist_code' => $item['pricelist_code'],
                            'price' => $item['price'],
                            'duplicate_type' => 'Import File Duplicate'
                        ];
                    }
                }
            }

            // Get existing items from database
            if (!empty($duplicateInfo['databaseDuplicates'])) {
                foreach ($duplicateInfo['databaseDuplicates'] as $compositeKey) {
                    list($itemCode, $pricelistCode) = explode('|', $compositeKey);
                    $existingItems = DB::table('Luv2_item_pricelist')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('item_code', $itemCode)
                        ->where('pricelist_code', $pricelistCode)
                        ->get();

                    foreach ($existingItems as $item) {
                        $duplicateItems[] = [
                            'item_code' => $item->item_code,
                            'pricelist_code' => $item->pricelist_code,
                            'price' => $item->price,
                            'duplicate_type' => 'Existing in Database'
                        ];
                    }
                }
            }

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "duplicate_item_pricelists_{$timestamp}.xlsx";

            return Excel::download(new class($duplicateItems) implements
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return [
                        'Item Code',
                        'Pricelist Code',
                        'Price',
                        'Duplicate Type'
                    ];
                }
            }, $filename);
        } catch (\Exception $e) {
            $this->alert('error', 'Error exporting duplicates: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return null;
        }
    }

    public function saveChunks()
    {
        try {
            $userId = Auth::user()->id;
            DB::table('Luv2_item_pricelist_temp')->where('id_user', $userId)->delete();

            // Get list of existing items in database
            $existingItems = DB::table('Luv2_item_pricelist')
                ->where('company_code', Auth::user()->company_code)
                ->get();
            $existingKeys = [];
            foreach ($existingItems as $item) {
                $existingKeys[$item->item_code . '|' . $item->pricelist_code] = true;
            }

            // Get list of items in current chunks
            $chunkKeys = [];
            $duplicateKeysInChunks = [];
            foreach ($this->chunks as $data) {
                $key = $data['item_code'] . '|' . $data['pricelist_code'];
                if (isset($chunkKeys[$key])) {
                    $duplicateKeysInChunks[$key] = true;
                }
                $chunkKeys[$key] = true;
            }

            // Filter out duplicates
            $filteredChunks = [];
            $seenKeys = [];
            foreach ($this->chunks as $data) {
                $key = $data['item_code'] . '|' . $data['pricelist_code'];
                // Skip if combination already exists in database or is a duplicate in chunks
                if (isset($existingKeys[$key]) || isset($duplicateKeysInChunks[$key]) || isset($seenKeys[$key])) {
                    continue;
                }
                $seenKeys[$key] = true;
                $filteredChunks[] = $data;
            }

            // Process filtered chunks in batches
            $chunksItem = array_chunk($filteredChunks, 1000);
            foreach ($chunksItem as $chunk) {
                DB::table('Luv2_item_pricelist_temp')->insert($chunk);
            }

            // Insert non-duplicate items
            DB::statement("
                INSERT INTO \"Luv2_item_pricelist\" (\"company_code\", \"item_code\", \"pricelist_code\", \"price\", \"created_at\", \"updated_at\") 
                SELECT DISTINCT \"company_code\", \"item_code\", \"pricelist_code\", \"price\", \"created_at\", \"updated_at\"
                FROM \"Luv2_item_pricelist_temp\"
                WHERE \"id_user\"='" . $userId . "'
            ");

            $skippedCount = count($this->chunks) - count($filteredChunks);
            $successMessage = 'Data Item Pricelist successfully saved.';
            if ($skippedCount > 0) {
                $successMessage .= " ($skippedCount duplicate items were skipped)";
            }

            $this->alert('success', $successMessage, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->chunks = [];
            $this->file = '';
            $this->dispatch('afterSave');
        } catch (\Exception $e) {
            $this->alert('error', 'There was an error: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }

    #[On('editItemPricelist')]
    public function editPricelistItem($code, $pricelistcode)
    {
        // dd($code, $pricelistcode);
        $item = DB::table('Luv2_item_pricelist')->where('item_code', $code)->where('pricelist_code', $pricelistcode)->first();
        $this->selectedItemCode = $item->item_code;
        $this->selectedItemPricelistCode = $item->pricelist_code;
        $this->selectedItemPrice = $item->price;
        $this->dispatch('openModalEdit');
    }

    public function update()
    {
        $cleanPrice = str_replace('.', '', $this->pricenew);
        $this->pricenew = (int) $cleanPrice;
        DB::table('Luv2_item_pricelist')
            ->where('item_code', $this->selectedItemCode)
            ->where('pricelist_code', $this->selectedItemPricelistCode)
            ->update(['price' => $this->pricenew]);

        $this->alert('success', 'Data Item Pricelist successfully updated', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);

        $this->dispatch('closeModalEdit');
        $this->dispatch('redirect');
    }

    #[On('redirect')]
    public function toRedirect()
    {
        sleep(2);
        return redirect()->route('item.pricelistitem', $this->pricelist_code);
    }

    public function cancelImport()
    {
        $this->chunks = [];
    }

    public function updateChunk($index, $field, $value)
    {
        $this->chunks[$index][$field] = $value;
    }

    public function modeEdit()
    {
        $this->editChunks = !$this->editChunks;

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    #[On('updatedWhs')]
    public function mount($param)
    {
        $this->itemPricelist = DB::table('Luv2_item_pricelist')
            ->where('company_code', Auth::user()->company_code)
            ->where('pricelist_code', $param)
            ->get();

        $this->Pricelist = DB::table('Luv2_pricelist')->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->get();
        $this->pricelist_code = $param;
        $this->whs_code = Auth::user()->whs_code;
        $this->dispatch('refreshjs');
    }

    public function save()
    {

        // dd($this->pricelist_code);
        // $this->validate();
        // dd($this->whs_code, $this->item_code, $this->pricelist_code, $this->price);
        $cleanPrice = str_replace('.', '', $this->price);
        $this->price = (int) $cleanPrice;
        $item = DB::table('Luv2_item')
            ->where('company_code', auth()->user()->company_code)
            ->where('code', $this->item_code)->first();
        if ($item) {
            $checkitem = DB::table('Luv2_item_pricelist')->where('company_code', Auth::user()->company_code)->where('item_code', $this->item_code)->where('pricelist_code', $this->pricelist_code)->first();
            if ($checkitem) {
                $this->errMsg = 'Pricelist with Item Code ' . $this->item_code . ' and Pricelist Code ' . $this->pricelist_code . ' already exists !';
            } else {
                $getitem = DB::table('Luv2_item')->where('company_code', Auth::user()->company_code)
                    // ->where('whs_code', auth()->user()->whs_code)
                    ->where('code', $this->item_code)->first();
                if ($getitem) {
                    DB::table('Luv2_item')->where('company_code', Auth::user()->company_code)
                        ->where('code', $this->item_code)
                        //    ->where('whs_code', auth()->user()->whs_code)
                        ->update([
                            'price' => $this->price,
                        ]);
                }
                DB::table('Luv2_item_pricelist')->insert([
                    'company_code' => Auth::user()->company_code,
                    // 'whs_code' => $this->whs_code,
                    'item_code' => $this->item_code,
                    'pricelist_code' => $this->pricelist_code,
                    'price' => $this->price,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
                $this->dispatch('closeModal');
                $this->dispatch('afterSave');
            }
        } else {
            $this->errMsg = 'Item Code ' . $this->item_code . ' not found! ';
        }
    }

    public function updated()
    {
        $this->errMsg = '';
    }

    #[On('closeModal')]
    public function closeModal()
    {
        $this->item_code = '';
        $this->price = '';
        $this->searchitem = '';

        // $this->dispatch('closemodalCreate');
        $this->alert('success', 'Data Item Pricelist successfully saved', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
    }

    public function changeTable()
    {
        $this->isTable = !$this->isTable;

        $this->chunks = [];
        $this->file = '';

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    #[On('afterSave')]
    public function dataItemPricelistafterSave()
    {
        $this->itemPricelist = DB::table('Luv2_item_pricelist')->where('company_code', Auth::user()->company_code)
            ->where('pricelist_code', $this->param)
            ->get();
    }

    public function gotoPage($page)
    {
        $this->setPage($page);
    }

    public function previousPage()
    {
        $this->previousPage();
    }

    public function nextPage()
    {
        $this->nextPage();
    }
}
