<?php

namespace App\Livewire\Pos\Item;

use App\Models\Item;
use App\Models\ItemTemp;
use App\Models\Warehouse;
use App\Rules\UniqueCompanyCode;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Illuminate\Validation\Rule as ValRule;

use Livewire\Component;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;
use Livewire\WithPagination;


class InsertItem extends Component
{
    use WithFileUploads;
    use LivewireAlert;
    use WithPagination;

    public $inventory = true;
    public $sales = true;
    public $purchase = true;
    public $isConsignment = false;
    public $itemCategory;
    public $itemBrand;
    public $itemSize;
    public $itemUnit;
    public $item;
    public $removedDuplicates = [];

    public $isEdit = false;

    public string $code = '';
    public ?string $barcode = null;
    public string $name = '';
    public string $brand_code = '';
    public string $category_code = '';
    public string $size_code = '';

    public string $unit_code = '';
    public $file;
    public $chunks;
    public $editChunks = false;
    public $isTable = true;
    public $isCreateNewBrand = false;
    public $isCreateNewCategory = false;
    public $isCreateNewSize = false;
    public $isCreateNewUnit = false;
    public $isCreateNewSupplier = false;
    public $name_brand = '';
    public $name_unit = '';
    public $unit_label = '';
    public $name_size = '';
    public $name_category = '';
    public $code_brand = '';
    public $code_size = '';
    public $code_category = '';
    public $warehouses;
    public $whs_code;
    public $qtyunit;

    public $suppliers;

    public $istutorials;

    public $canEditSupplier = true;
    public $supplier_code = '';
    public $infosupplier_name = '';
    public $infosupplier_code = '';
    public $infoitem_name = '';
    public $infoitem_code = '';
    public $infoitem_barcode = '';
    public $suppliernew_name = '';
    public $suppliernew_address = '';
    public $suppliernew_phone = '';
    public $suppliernew_email = '';
    public $shared_margin = 0;
    public $activeUDFs;
    public $udfFields = [];
    public $autoGenerateCode = false;

    public $page = 1;
    public $perPage = 10;
    public $search = '';
    public $sortDirection = 'ASC';
    public $sortColumn = 'name';

    private $requiredColumns = [
        0 => 'Code',
        1 => 'Barcode',
        2 => 'Item Name',
        3 => 'Brand Code',
        4 => 'Category Code',
        5 => 'Size Code'
    ];

    public function rules()
    {
        return [
            'code' => [
                'required',
                'min:3',
                ValRule::unique('Luv2_item', 'code')->ignore($this->code, 'code'),
            ],
            'barcode' => [
                'required',
                'min:3',
                ValRule::unique('Luv2_item', 'barcode')->ignore($this->barcode, 'barcode'),
            ],
            'name' => [
                'required',
                'min:4',
            ],
            'brand_code' => [
                'required',
            ],
            'category_code' => [
                'required',
            ],
            'size_code' => [
                'required',
            ],
        ];
    }


    public function updatedAutoGenerateCode($value)
    {
        if ($value) {
            $this->code = $this->generateAutoCode('PRD');
        } else {
            $this->code = '';
        }
        $this->resetErrorBag('code');

        $this->dispatch('getGeneratedCode', $this->code);
    }

    public function generateAutoCode($prefix)
    {
        $random = strtoupper(Str::random(5));
        return $prefix . '-' . $random;
    }



    public function render()
    {
        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        $items = Item::where('company_code', Auth::user()->company_code)
            ->with([
                'brand' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
                'size' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
                'category' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
            ])
            ->when($this->search, function($query) {
                $query->where(function($q) {
                    $q->where('code', 'like', '%' . $this->search . '%')
                      ->orWhere('barcode', 'like', '%' . $this->search . '%')
                      ->orWhere('name', 'like', '%' . $this->search . '%')
                      ->orWhereHas('brand', function($q) {
                          $q->where('name', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('category', function($q) {
                          $q->where('name', 'like', '%' . $this->search . '%');
                      })
                      ->orWhereHas('size', function($q) {
                          $q->where('name', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->orderBy($this->sortColumn, $this->sortDirection)
            ->paginate($this->perPage);

        if ($items->isNotEmpty()) {
            $items->getCollection()->transform(function ($item) {
                if ($this->activeUDFs->isNotEmpty()) {
                    $itemUDFs = DB::table('Luv2_udf_item')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('item_code', $item->code)
                        ->whereIn('code_udf', $this->activeUDFs->pluck('code'))
                        ->pluck('value', 'code_udf');

                    foreach ($this->activeUDFs as $udf) {
                        $item->{$udf->field} = $itemUDFs[$udf->code] ?? null;
                    }
                }
                return $item;
            });
        }

        return view('livewire.pos.item.insert-item', [
            'items' => $items
        ]);
    }

    private function checkForDuplicates($data)
    {
        try {
            // Check for duplicates within the imported data
            $codes = array_column($data, 'code');
            $barcodes = array_filter(array_column($data, 'barcode')); // Get non-empty barcodes
            
            $duplicateCodesCount = array_count_values($codes);
            $duplicateBarcodesCount = array_count_values($barcodes);
            
            // Get list of codes and barcodes that exist in database
            $existingItems = DB::table('Luv2_item')
                ->where('company_code', Auth::user()->company_code)
                ->where(function($query) use ($codes, $barcodes) {
                    $query->whereIn('code', $codes)
                          ->orWhereIn('barcode', $barcodes);
                })
                ->get(['code', 'name', 'barcode', 'brand_code', 'category_code', 'size_code', 'inventory', 'sales', 'purchase', 'consignment', 'supp_code as supplier_code', 'price'])
                ->keyBy('code');

            // Separate data into updates and inserts
            $updatesData = [];
            $insertsData = [];
            $removedDuplicates = [];
            $seenCodes = [];
            $seenBarcodes = [];

            foreach ($data as $index => $item) {
                if (!isset($item['code']) || empty($item['code'])) {
                    continue;
                }

                $code = $item['code'];
                $barcode = !empty($item['barcode']) ? $item['barcode'] : null;
                
                $isDuplicate = false;
                $duplicateType = '';

                // Check for duplicate code in chunks
                if (isset($seenCodes[$code])) {
                    $isDuplicate = true;
                    $duplicateType = 'Duplikat Kode dalam File Import';
                }
                // Check for duplicate barcode in chunks if barcode exists
                else if ($barcode && isset($seenBarcodes[$barcode])) {
                    $isDuplicate = true;
                    $duplicateType = 'Duplikat Barcode dalam File Import';
                }
                // Check if exists in database by code
                else if (isset($existingItems[$code])) {
                    $isDuplicate = true;
                    $duplicateType = 'Kode Sudah Ada di Database';
                }
                // Check if exists in database by barcode
                else if ($barcode && $existingItems->contains('barcode', $barcode)) {
                    $isDuplicate = true;
                    $duplicateType = 'Barcode Sudah Ada di Database';
                }

                if ($isDuplicate) {
                    $item['duplicate_type'] = $duplicateType;
                    $removedDuplicates[] = $item;
                    continue;
                }
                
                // For first occurrence, add to inserts and mark as seen
                $insertsData[] = $item;
                $seenCodes[$code] = true;
                if ($barcode) {
                    $seenBarcodes[$barcode] = true;
                }

                // If this code appears multiple times in import, add all occurrences after first to removedDuplicates
                if ($duplicateCodesCount[$code] > 1) {
                    foreach ($data as $dupIndex => $dupItem) {
                        if ($dupIndex > $index && $dupItem['code'] === $code) {
                            $dupItem['duplicate_type'] = 'Duplikat Kode dalam File Import';
                            $removedDuplicates[] = $dupItem;
                        }
                    }
                }

                // If this barcode appears multiple times in import, add all occurrences after first to removedDuplicates
                if ($barcode && isset($duplicateBarcodesCount[$barcode]) && $duplicateBarcodesCount[$barcode] > 1) {
                    foreach ($data as $dupIndex => $dupItem) {
                        if ($dupIndex > $index && !empty($dupItem['barcode']) && $dupItem['barcode'] === $barcode) {
                            $dupItem['duplicate_type'] = 'Duplikat Barcode dalam File Import';
                            $removedDuplicates[] = $dupItem;
                        }
                    }
                }
            }

            return [
                'hasDuplicates' => !empty($removedDuplicates),
                'updates' => $updatesData,
                'inserts' => $insertsData,
                'removedDuplicates' => $removedDuplicates,
                'existingItems' => $existingItems
            ];
        } catch (\Exception $e) {
            Log::error('Error in checkForDuplicates: ' . $e->getMessage());
            throw new \Exception('Error checking for duplicates: ' . $e->getMessage());
        }
    }

    private function processUpdates($updates)
    {
        try {
            foreach ($updates as $item) {
                if (!isset($item['code']) || empty($item['code'])) {
                    continue;
                }

                DB::table('Luv2_item')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('code', $item['code'])
                    ->update([
                        'barcode' => $item['barcode'] ?? null,
                        'name' => $item['name'] ?? '',
                        'brand_code' => $item['brand_code'] ?? '',
                        'category_code' => $item['category_code'] ?? '',
                        'size_code' => $item['size_code'] ?? '',
                        'inventory' => $item['inventory'] ?? 'Y',
                        'sales' => $item['sales'] ?? 'Y',
                        'purchase' => $item['purchase'] ?? 'Y',
                        'consignment' => $item['consignment'] ?? 'N',
                        'supp_code' => $item['supplier_code'] ?? null,
                        'price' => is_numeric($item['price'] ?? null) ? $item['price'] : 0,
                        'updated_at' => Carbon::now()
                    ]);
            }
        } catch (\Exception $e) {
            Log::error('Error in processUpdates: ' . $e->getMessage());
            throw new \Exception('Error updating records: ' . $e->getMessage());
        }
    }

    public function saveChunks()
    {
        if ($this->hasConsignmentWithoutSupplier()) {
            $this->alert('error', 'Beberapa item adalah konsinyasi tetapi tidak memiliki supplier.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return;
        }

        // Check for invalid supplier codes
        $invalidSuppliers = [];
        foreach ($this->chunks as $chunk) {
            if (($chunk['consignment'] ?? 'N') === 'Y' && !empty($chunk['supplier_code'])) {
                $supplierExists = DB::table('Luv2_supplier')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('code', $chunk['supplier_code'])
                    ->exists();

                if (!$supplierExists) {
                    $invalidSuppliers[] = [
                        'item_code' => $chunk['code'],
                        'item_name' => $chunk['name'],
                        'supplier_code' => $chunk['supplier_code']
                    ];
                }
            }
        }

        if (!empty($invalidSuppliers)) {
            $errorMessage = "Beberapa item konsinyasi memiliki kode supplier yang tidak valid:\n";
            foreach ($invalidSuppliers as $invalid) {
                $errorMessage .= "- Item {$invalid['item_name']} (Kode: {$invalid['item_code']}) dengan Supplier: {$invalid['supplier_code']}\n";
            }
            
            $this->alert('error', $errorMessage, [
                'position' => 'top-end',
                'timer' => 7000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return;
        }

        DB::beginTransaction();
        try {
            $userId = Auth::user()->id;
            
            // Get list of codes that exist in database
            $codes = array_column($this->chunks, 'code');
            $existingItems = DB::table('Luv2_item')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $codes)
                ->pluck('code')
                ->toArray();

            $updateCount = 0;
            $insertCount = 0;

            foreach ($this->chunks as $chunk) {
                $data = [
                    'company_code' => Auth::user()->company_code,
                    'code' => $chunk['code'],
                    'barcode' => $chunk['barcode'],
                    'name' => $chunk['name'],
                    'brand_code' => $chunk['brand_code'],
                    'category_code' => $chunk['category_code'],
                    'size_code' => $chunk['size_code'],
                    'status' => 'Y',
                    'inventory' => $chunk['inventory'],
                    'sales' => $chunk['sales'],
                    'purchase' => $chunk['purchase'],
                    'consignment' => $chunk['consignment'],
                    'supp_code' => $chunk['supplier_code'],
                    'active' => 'Y',
                    'avgprice' => 0,
                    'price' => $chunk['price'],
                    'updated_at' => Carbon::now(),
                ];

                if (in_array($chunk['code'], $existingItems)) {
                    // Update existing record
                    DB::table('Luv2_item')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $chunk['code'])
                        ->update($data);
                    $updateCount++;
                } else {
                    // Insert new record
                    $data['created_at'] = Carbon::now();
                    DB::table('Luv2_item')->insert($data);
                    $insertCount++;
                }
            }

            DB::commit();

            $message = "Data berhasil disimpan ke database. ";
            if ($insertCount > 0) {
                $message .= "Menambahkan $insertCount data baru. ";
            }
            if ($updateCount > 0) {
                $message .= "Memperbarui $updateCount data yang sudah ada.";
            }

            $this->alert('success', $message, [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => false,
            ]);

            $this->dispatch('successImport');
            $this->dispatch('refreshDatatable');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving chunks: ' . $e->getMessage());
            $this->alert('error', 'Error saat menyimpan data: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function exportDuplicates($duplicateItems, $fileType)
    {
        try {
            // Ensure all items have the required fields
            $processedItems = array_map(function($item) {
                return [
                    'code' => $item['code'] ?? '',
                    'barcode' => $item['barcode'] ?? '',
                    'name' => $item['name'] ?? '',
                    'brand_code' => $item['brand_code'] ?? '',
                    'category_code' => $item['category_code'] ?? '',
                    'size_code' => $item['size_code'] ?? '',
                    'inventory' => $item['inventory'] ?? 'Y',
                    'sales' => $item['sales'] ?? 'Y',
                    'purchase' => $item['purchase'] ?? 'Y',
                    'consignment' => $item['consignment'] ?? 'N',
                    'supplier_code' => $item['supplier_code'] ?? '',
                    'price' => $item['price'] ?? 0,
                    'duplicate_type' => $item['duplicate_type'] ?? 'Import File Duplicate'
                ];
            }, $duplicateItems);

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "duplicate_items_{$timestamp}";

            if ($fileType === 'csv') {
                $filename .= '.csv';
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => "attachment; filename={$filename}",
                ];

                $handle = fopen('php://temp', 'r+');
                
                // Add headers
                fputcsv($handle, [
                    'Code', 'Barcode', 'Item Name', 'Brand Code', 'Category Code', 
                    'Size Code', 'Inventory', 'Sales', 'Purchase', 'Consignment',
                    'Supplier Code', 'Price', 'Duplicate Type'
                ]);
                
                // Add data
                foreach ($processedItems as $item) {
                    fputcsv($handle, array_values($item));
                }

                rewind($handle);
                $content = stream_get_contents($handle);
                fclose($handle);

                return response($content, 200, $headers);
            } 
            
            // Excel export
            $filename .= '.xlsx';
            return Excel::download(new class($processedItems) implements 
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return [
                        'Code', 'Barcode', 'Item Name', 'Brand Code', 'Category Code', 
                        'Size Code', 'Inventory', 'Sales', 'Purchase', 'Consignment',
                        'Supplier Code', 'Price', 'Duplicate Type'
                    ];
                }
            }, $filename);

        } catch (\Exception $e) {
            Log::error('Error in exportDuplicates: ' . $e->getMessage());
            throw new \Exception('Error exporting duplicates: ' . $e->getMessage());
        }
    }

    public function downloadDuplicates($format = 'xlsx')
    {
        try {
            if (empty($this->removedDuplicates)) {
                $this->alert('error', 'Tidak ada data duplikat yang tersedia untuk diekspor', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => false,
                ]);
                return null;
            }

            return $this->exportDuplicates($this->removedDuplicates, $format);
        } catch (\Exception $e) {
            Log::error('Error in downloadDuplicates: ' . $e->getMessage());
            $this->alert('error', 'Error saat mengunduh data duplikat: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => false,
            ]);
            return null;
        }
    }

    public function toImportItem()
    {
        $this->isTable = false;
        $this->dispatch('goToimportItem');
    }

    #[On('goToimportItem')]
    public function importItem()
    {
        try {
            $this->validate([
                'file' => 'required|file|mimes:csv,xlsx,txt',
            ]);

            $fileExtension = $this->file->getClientOriginalExtension();

            switch ($fileExtension) {
                case 'csv':
                case 'txt':
                    $this->importCSV();
                    break;
                case 'xlsx':
                    $this->importXLSX();
                    break;
                default:
                    throw new \Exception('Unsupported file format. Please use CSV, TXT, or XLSX files.');
            }

            $this->dispatch('refreshjs');
        } catch (\Exception $e) {
            $this->alert('error', $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function validateImportData($data, $rowNumber)
    {
        $errors = [];
        
        // Check if we have data
        if (empty($data) || !is_array($data)) {
            return [
                'error' => true,
                'message' => 'Row ' . $rowNumber . ': Invalid data format'
            ];
        }

        // Validate required fields with trimmed values
        $data = array_map('trim', $data);

        if (empty($data[0])) { // Code
            $errors[] = "Code is required";
        } elseif (!preg_match('/^[a-zA-Z0-9-_]+$/', $data[0])) {
            $errors[] = "Code must contain only letters, numbers, hyphens and underscores";
        }

        if (empty($data[2])) { // Item Name
            $errors[] = "Item Name is required";
        }

        if (empty($data[3])) { // Brand Code
            $errors[] = "Brand Code is required";
        }

        if (empty($data[4])) { // Category Code
            $errors[] = "Category Code is required";
        }

        if (empty($data[5])) { // Size Code
            $errors[] = "Size Code is required";
        }

        if (!empty($errors)) {
            return [
                'error' => true,
                'message' => "Row {$rowNumber}: " . implode(', ', $errors)
            ];
        }

        return ['error' => false];
    }

    private function processImportData($data)
    {
        // Trim whitespace from all values and create structured array
        $processedData = [
            'code' => strval(trim($data[0])),
            'barcode' => !empty($data[1]) ? trim($data[1]) : null,
            'name' => trim($data[2]),
            'brand_code' => trim($data[3]),
            'category_code' => trim($data[4]),
            'size_code' => trim($data[5]),
            'inventory' => isset($data[6]) && strtoupper(trim($data[6])) === 'Y' ? 'Y' : 'N',
            'sales' => isset($data[7]) && strtoupper(trim($data[7])) === 'Y' ? 'Y' : 'N',
            'purchase' => isset($data[8]) && strtoupper(trim($data[8])) === 'Y' ? 'Y' : 'N',
            'consignment' => isset($data[9]) && strtoupper(trim($data[9])) === 'Y' ? 'Y' : 'N',
            'supplier_code' => isset($data[10]) && !empty(trim($data[10])) ? trim($data[10]) : null,
            'price' => isset($data[11]) && is_numeric(trim($data[11])) ? trim($data[11]) : 0,
        ];

        // Check if consignment item has supplier code
        if ($processedData['consignment'] === 'Y' && empty($processedData['supplier_code'])) {
            $processedData['duplicate_type'] = 'Item konsinyasi harus memiliki kode supplier';
            return ['error' => true, 'data' => $processedData];
        }

        // If item is consignment and has supplier code, check if supplier exists
        if ($processedData['consignment'] === 'Y' && !empty($processedData['supplier_code'])) {
            $supplierExists = DB::table('Luv2_supplier')
                ->where('company_code', Auth::user()->company_code)
                ->where('code', $processedData['supplier_code'])
                ->exists();

            if (!$supplierExists) {
                $processedData['duplicate_type'] = 'Supplier dengan kode ' . $processedData['supplier_code'] . ' tidak ditemukan';
                return ['error' => true, 'data' => $processedData];
            }
        }

        return ['error' => false, 'data' => $processedData];
    }

    private function importCSV()
    {
        try {
            $file = fopen($this->file->getRealPath(), 'r');
            if ($file === false) {
                throw new \Exception('Tidak dapat membuka file');
            }

            $skipFirstRow = true;
            $chunks = [];
            $this->removedDuplicates = [];
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];
            $seenBarcodes = [];

            // Try to detect the delimiter from the first line
            $firstLine = fgets($file);
            rewind($file);
            
            // Count occurrences of potential delimiters
            $commaCount = substr_count($firstLine, ',');
            $tabCount = substr_count($firstLine, "\t");
            $semicolonCount = substr_count($firstLine, ';');
            
            // Choose the delimiter with the highest count
            $delimiter = ',';  // default
            $maxCount = $commaCount;
            
            if ($tabCount > $maxCount) {
                $delimiter = "\t";
                $maxCount = $tabCount;
            }
            if ($semicolonCount > $maxCount) {
                $delimiter = ";";
            }

            while (($data = fgetcsv($file, 0, $delimiter)) !== false) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($data))) {
                    continue;
                }

                $validationResult = $this->validateImportData($data, $rowNumber);
                if ($validationResult['error']) {
                    $errors[] = $validationResult['message'];
                    continue;
                }

                $processedResult = $this->processImportData($data);
                $processedData = $processedResult['data'];
                
                if ($processedResult['error']) {
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                $code = $processedData['code'];
                $barcode = $processedData['barcode'];

                // Check for duplicates
                $isDuplicate = false;
                $duplicateType = '';

                // Check if code exists in seen codes
                if (isset($seenCodes[$code])) {
                    $isDuplicate = true;
                    $duplicateType = 'Duplikat Kode dalam File Import';
                }
                // Check if barcode exists in seen barcodes (only if barcode is not empty)
                else if (!empty($barcode) && isset($seenBarcodes[$barcode]) && $seenBarcodes[$barcode] !== $code) {
                    $isDuplicate = true;
                    $duplicateType = 'Duplikat Barcode dalam File Import';
                }

                if ($isDuplicate) {
                    $processedData['duplicate_type'] = $duplicateType;
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                // Store both code and barcode
                $seenCodes[$code] = true;
                if (!empty($barcode)) {
                    $seenBarcodes[$barcode] = $code; // Store the code that this barcode belongs to
                }
                
                $chunks[] = $processedData;
            }
            
            fclose($file);

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            $this->chunks = $chunks;

            if (empty($chunks)) {
                $this->alert('error', 'Tidak ada data valid yang dapat diimpor. Semua data duplikat atau tidak valid.', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data yang tidak valid atau duplikat yang telah dihapus. Anda dapat mengekspor data tersebut menggunakan tombol 'Ekspor Data Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                
                // Dispatch event to show success message after 5 seconds
                $this->dispatch('showSuccessImport', 5);
            } else {
                $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true
                ]);
            }
            
            $this->dispatch('refreshjs');
            
        } catch (\Exception $e) {
            $this->alert('error', 'Error saat mengimpor CSV: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 10000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => true
            ]);
        }
    }

    public function importXLSX()
    {
        try {
            $rows = Excel::toArray([], $this->file->getRealPath())[0];
            if (count($rows) <= 1) {
                throw new \Exception('Tidak ada data dalam file Excel.');
            }

            $chunks = [];
            $this->removedDuplicates = [];
            $skipFirstRow = true;
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];
            $seenBarcodes = [];

            foreach ($rows as $row) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($row))) {
                    continue;
                }

                $validationResult = $this->validateImportData($row, $rowNumber);
                if ($validationResult['error']) {
                    $errors[] = $validationResult['message'];
                    continue;
                }

                $processedResult = $this->processImportData($row);
                $processedData = $processedResult['data'];
                
                if ($processedResult['error']) {
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                $code = $processedData['code'];
                $barcode = $processedData['barcode'];

                // Check for duplicates
                $isDuplicate = false;
                $duplicateType = '';

                // Check if code exists in seen codes
                if (isset($seenCodes[$code])) {
                    $isDuplicate = true;
                    $duplicateType = 'Duplikat Kode dalam File Import';
                }
                // Check if barcode exists in seen barcodes (only if barcode is not empty)
                else if (!empty($barcode) && isset($seenBarcodes[$barcode]) && $seenBarcodes[$barcode] !== $code) {
                    $isDuplicate = true;
                    $duplicateType = 'Duplikat Barcode dalam File Import';
                }

                if ($isDuplicate) {
                    $processedData['duplicate_type'] = $duplicateType;
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                // Store both code and barcode
                $seenCodes[$code] = true;
                if (!empty($barcode)) {
                    $seenBarcodes[$barcode] = $code; // Store the code that this barcode belongs to
                }
                
                $chunks[] = $processedData;
            }

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            $this->chunks = $chunks;

            if (empty($chunks)) {
                $this->alert('error', 'Tidak ada data valid yang dapat diimpor. Semua data duplikat atau tidak valid.', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data yang tidak valid atau duplikat yang telah dihapus. Anda dapat mengekspor data tersebut menggunakan tombol 'Ekspor Data Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                
                // Dispatch event to show success message after 5 seconds
                $this->dispatch('showSuccessImport', 5);
            } else {
                $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true
                ]);
            }

            $this->dispatch('refreshjs');

        } catch (\Exception $e) {
            Log::error('Error in importXLSX: ' . $e->getMessage());
            $this->alert('error', 'Error saat mengimpor Excel: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
        }
    }

    #[On('showSuccessImport')] 
    public function showSuccessImport($delay = 0)
    {
        sleep($delay);
        $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
            'position' => 'top-end',
            'timer' => 5000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
    }

    public function changeTable()
    {
        $this->isTable = !$this->isTable;

        if ($this->isTable) {
            $this->dispatch('reGet');
        }

        $this->chunks = [];
        $this->file = '';

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }

    public function cancelImport()
    {
        $this->dispatch('cancelImport');
        $this->chunks = [];
    }

    public function handleDuplicates($chunks)
    {
        $duplicateInfo = $this->checkForDuplicates($chunks);
        if ($duplicateInfo['hasDuplicates']) {
            $message = $this->buildDuplicateMessage($duplicateInfo);
            
            $this->alert('warning', $message, [
                'position' => 'top-end',
                'timer' => 7000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->chunks = $chunks; // Store chunks for potential export
            return true;
        }
        return false;
    }

    public function exportCurrentTable($format = 'xlsx')
    {
        try {
            // Get current filtered/searched items
            $items = Item::where('company_code', Auth::user()->company_code)
                ->with([
                    'brand' => function ($query) {
                        $query->where('company_code', Auth::user()->company_code);
                    },
                    'size' => function ($query) {
                        $query->where('company_code', Auth::user()->company_code);
                    },
                    'category' => function ($query) {
                        $query->where('company_code', Auth::user()->company_code);
                    },
                ])
                ->when($this->search, function($query) {
                    $query->where(function($q) {
                        $q->where('code', 'like', '%' . $this->search . '%')
                          ->orWhere('barcode', 'like', '%' . $this->search . '%')
                          ->orWhere('name', 'like', '%' . $this->search . '%')
                          ->orWhereHas('brand', function($q) {
                              $q->where('name', 'like', '%' . $this->search . '%');
                          })
                          ->orWhereHas('category', function($q) {
                              $q->where('name', 'like', '%' . $this->search . '%');
                          })
                          ->orWhereHas('size', function($q) {
                              $q->where('name', 'like', '%' . $this->search . '%');
                          });
                    });
                })
                ->orderBy($this->sortColumn, $this->sortDirection)
                ->get();

            if ($items->isEmpty()) {
                $this->alert('warning', 'No data to export', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            // Transform items for export
            $exportData = $items->map(function ($item) {
                return [
                    'code' => $item->code,
                    'barcode' => $item->barcode ?? '',
                    'name' => $item->name,
                    'brand_code' => $item->brand_code,
                    'brand_name' => $item->brand->name ?? '',
                    'category_code' => $item->category_code,
                    'category_name' => $item->category->name ?? '',
                    'size_code' => $item->size_code,
                    'size_name' => $item->size->name ?? '',
                    'inventory' => $item->inventory,
                    'sales' => $item->sales,
                    'purchase' => $item->purchase,
                    'consignment' => $item->consignment,
                    'supplier_code' => $item->supp_code ?? '',
                    'price' => $item->price ?? 0,
                ];
            })->toArray();

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "items_export_{$timestamp}";

            if ($format === 'csv') {
                $filename .= '.csv';
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => "attachment; filename={$filename}",
                ];

                $handle = fopen('php://temp', 'r+');
                
                // Add headers
                fputcsv($handle, [
                    'Code', 'Barcode', 'Item Name', 'Brand Code', 'Brand Name',
                    'Category Code', 'Category Name', 'Size Code', 'Size Name',
                    'Inventory', 'Sales', 'Purchase', 'Consignment',
                    'Supplier Code', 'Price'
                ]);
                
                // Add data
                foreach ($exportData as $row) {
                    fputcsv($handle, $row);
                }

                rewind($handle);
                $content = stream_get_contents($handle);
                fclose($handle);

                return response($content, 200, $headers);
            }

            // Excel export
            $filename .= '.xlsx';
            return Excel::download(new class($exportData) implements 
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return [
                        'Code', 'Barcode', 'Item Name', 'Brand Code', 'Brand Name',
                        'Category Code', 'Category Name', 'Size Code', 'Size Name',
                        'Inventory', 'Sales', 'Purchase', 'Consignment',
                        'Supplier Code', 'Price'
                    ];
                }
            }, $filename);

        } catch (\Exception $e) {
            $this->alert('error', 'Error exporting data: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return null;
        }
    }

    #[On('downloadDuplicatesXLSX')]
    public function downloadDuplicatesXLSX()
    {
        return $this->downloadDuplicates('xlsx');
    }

    #[On('dismissDuplicatesAlert')]
    public function dismissDuplicatesAlert()
    {
        // Just dismiss the alert
        return;
    }

    private function buildDuplicateMessage($duplicateInfo)
    {
        $message = '';
        if (!empty($duplicateInfo['importDuplicates'])) {
            $message .= 'Duplicate codes in import file: ' . implode(', ', $duplicateInfo['importDuplicates']) . '. ';
        }
        if (!empty($duplicateInfo['databaseDuplicates'])) {
            $message .= 'Codes already exist in database: ' . implode(', ', $duplicateInfo['databaseDuplicates']);
        }
        return $message;
    }

    public function updateChunk($index, $field, $value)
    {
        $this->chunks[$index][$field] = $value;
    }

    public function modeEditTrue()
    {
        $this->editChunks = true;
    }

    public function modeEditFalse()
    {
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }

    #[On('createNewItem')]
    public function createNewItem()
    {
        $this->isEdit = false;
    }


    #[On('reGet')]
    public function mount()
    {
        $activeUDFs = DB::table('Luv2_udf')
            ->where('company_code', Auth::user()->company_code)
            ->where('active', 'Y')
            ->get();


        if ($activeUDFs->isNotEmpty()) {
            $this->activeUDFs = $activeUDFs;
            $this->udfFields = [];

            foreach ($activeUDFs as $udf) {
                $this->udfFields[$udf->field] = '';
            }
        } else {
            $this->activeUDFs = collect();
            $this->udfFields = [];
        }

        $this->itemSize = DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)->orderBy('name', 'ASC')
            ->get();
        $this->itemBrand = DB::table('Luv2_brand')->where('company_code', Auth::user()->company_code)->orderBy('name', 'ASC')
            ->get();
        $this->itemCategory = DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)->orderBy('name', 'ASC')
            ->get();
        $this->itemUnit = DB::table('Luv2_unit')->where('company_code', Auth::user()->company_code)->orderBy('name', 'ASC')
            ->get();

        $this->size_code = '';
        $this->brand_code = '';
        $this->category_code = '';
        $this->item = Item::where('company_code', Auth::user()->company_code)
            ->with([
                'brand' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
                'size' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
                'category' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
            ])
            ->get()
            ->map(function ($item) use ($activeUDFs) {
                if ($activeUDFs->isNotEmpty()) {
                    $itemUDFs = DB::table('Luv2_udf_item')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('item_code', $item->code)
                        ->whereIn('code_udf', $activeUDFs->pluck('code'))
                        ->pluck('value', 'code_udf');

                    foreach ($activeUDFs as $udf) {
                        $item->{$udf->field} = $itemUDFs[$udf->code] ?? null;
                    }
                }

                return $item;
            });


        if ($this->item->isEmpty()) {
            $this->item = [];
        }

        if (Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        } else {
            $this->whs_code = Warehouse::where('company_code', auth()->user()->company_code)->first()->code;
        }


        $tutorials = DB::table('Luv2_tutorial')
            ->where('id_user', auth()->user()->id)
            ->where('menu', 'Product')
            ->first();
        if ($tutorials && $tutorials->active === 'Y') {
            $this->istutorials = true;
        } else {
            $this->istutorials = false;
        }

        $this->suppliers = DB::table('Luv2_supplier')
            ->where('company_code', Auth::user()->company_code)
            ->where('consignment', 'Y')
            ->get();

        if (session()->has('fromCashier') && session('fromCashier') == auth()->id()) {
            $this->dispatch('openModalBuat');
        }
    }

    public function updatedSupplierCode()
    {
        if ($this->supplier_code == 'SupplierCN') {
            $this->isCreateNewSupplier = true;
        } else {
            $this->isCreateNewSupplier = false;
        }
    }

    public function updatedBrandCode()
    {
        if ($this->brand_code == 'BrandCN') {
            $this->isCreateNewBrand = true;
        } else {
            $this->isCreateNewBrand = false;
            $this->name_brand = '';
        }
        $this->resetErrorBag('code_brand');
    }

    public function updatedCategoryCode()
    {
        if ($this->category_code == 'CategoryCN') {
            $this->isCreateNewCategory = true;
        } else {
            $this->isCreateNewCategory = false;
            $this->name_category = '';
        }
        $this->resetErrorBag('code_category');
    }

    public function updatedSizeCode()
    {
        if ($this->size_code == 'SizeCN') {
            $this->isCreateNewSize = true;
        } else {
            $this->isCreateNewSize = false;
            $this->name_size = '';
        }
        $this->resetErrorBag('code_size');
    }
    public function save()
    {
        try {
            $hasError = false;
           
            if (empty($this->code)) {
                $this->addError('code', 'Kode item tidak boleh kosong.');
                $hasError = true;
            } else if (strlen($this->code) < 3) {
                $this->addError('code', 'Kode item minimal 3 karakter.');
                $hasError = true;
            }
            
            if (empty($this->barcode)) {
                $this->addError('barcode', 'Barcode tidak boleh kosong.');
                $hasError = true;
            } else if (strlen($this->barcode) < 3) {
                $this->addError('barcode', 'Barcode minimal 3 karakter.');
                $hasError = true;
            }
            if (empty($this->name)) {
                $this->addError('name', 'Nama item tidak boleh kosong.');
                $hasError = true;
            } else if (strlen($this->name) < 4) {
                $this->addError('name', 'Nama item minimal 4 karakter.');
                $hasError = true;
            }

            if ($this->isConsignment) {
                if (empty($this->supplier_code)) {
                    $this->addError('supplier_code', 'Supplier harus dipilih untuk item konsinyasi.');
                    $hasError = true;
                }

                if ($this->supplier_code === 'SupplierCN') {
                    if (empty($this->suppliernew_name)) {
                        $this->addError('suppliernew_name', 'Nama supplier baru tidak boleh kosong.');
                        $hasError = true;
                    }
                    if (empty($this->suppliernew_address)) {
                        $this->addError('suppliernew_address', 'Alamat supplier baru tidak boleh kosong.');
                        $hasError = true;
                    }
                    if (empty($this->suppliernew_phone)) {
                        $this->addError('suppliernew_phone', 'Nomor telepon supplier baru tidak boleh kosong.');
                        $hasError = true;
                    }
                    if (empty($this->suppliernew_email)) {
                        $this->addError('suppliernew_email', 'Email supplier baru tidak boleh kosong.');
                        $hasError = true;
                    }
                }
            }

            if ($hasError) {
                return;
            }

            $this->validate();

            // Handle brand creation
            if ($this->isCreateNewBrand) {
                $hasError = $this->createNewBrand();
            }

            // Handle category creation 
            if ($this->isCreateNewCategory) {
                $hasError = $this->createNewCategory();
            }

            // Handle size creation
            if ($this->isCreateNewSize) {
                $hasError = $this->createNewSize(); 
            }

            if ($hasError) {
                return;
            }

            // Check for duplicate code
            $existingItem = DB::table('Luv2_item')
                ->where('company_code', Auth::user()->company_code)
                ->where('code', $this->code)
                ->first();

            if ($existingItem) {
                $this->addError('code', 'Kode item sudah digunakan.');
                return;
            }

            // Check for duplicate barcode if provided
            if ($this->barcode) {
                $existingBarcode = DB::table('Luv2_item')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('barcode', $this->barcode)
                    ->first();

                if ($existingBarcode) {
                    $this->addError('barcode', 'Barcode sudah digunakan.');
                    return;
                }
            }

            // Handle UDF fields
            if ($this->udfFields) {
                $this->saveUdfFields();
            }

            // Handle supplier creation
            if ($this->isCreateNewSupplier) {
                $this->createNewSupplier();
            }

            // Create main item record
            DB::table('Luv2_item')->insert([
                'company_code' => Auth::user()->company_code,
                'code' => $this->code,
                'barcode' => $this->barcode,
                'name' => $this->name,
                'brand_code' => $this->code_brand !== '' ? $this->code_brand : $this->brand_code,
                'category_code' => $this->code_category !== '' ? $this->code_category : $this->category_code,
                'size_code' => $this->code_size !== '' ? $this->code_size : $this->size_code,
                'status' => 'Y',
                'inventory' => $this->inventory ? 'Y' : 'N',
                'sales' => $this->sales ? 'Y' : 'N', 
                'purchase' => $this->purchase ? 'Y' : 'N',
                'consignment' => $this->isConsignment ? 'Y' : 'N',
                'supp_code' => $this->isConsignment ? $this->supplier_code : '',
                'active' => 'Y',
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);

            $this->dispatch('closeModal');
            $this->dispatch('afterSave');

            if (session()->has('toCashier') && session('toCashier') == auth()->id()) {
                session()->forget('toCashier');
                return redirect()->route('welcome.dashboard');
            }

        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('Database error in save(): ' . $e->getMessage());
            $this->alert('error', 'Terjadi kesalahan pada database. Silahkan hubungi softcomp support.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        } catch (\Exception $e) {
            Log::error('Error in save(): ' . $e->getMessage());
            $this->alert('error', 'Terjadi kesalahan yang tidak terduga. Silahkan hubungi softcomp support.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function createNewBrand()
    {
        $checkBrand = DB::table('Luv2_brand')
            ->where('company_code', Auth::user()->company_code)
            ->where('code', $this->code_brand)
            ->first();

        if ($checkBrand) {
            $this->addError('code_brand', 'Code brand sudah digunakan.');
            return true;
        }

        DB::table('Luv2_brand')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => $this->code_brand,
            'name' => $this->name_brand,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        return false;
    }

    private function createNewCategory()
    {
        $checkCategory = DB::table('Luv2_category')
            ->where('company_code', Auth::user()->company_code)
            ->where('code', $this->code_category)
            ->first();

        if ($checkCategory) {
            $this->addError('code_category', 'Code category sudah digunakan.');
            return true;
        }

        DB::table('Luv2_category')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => $this->code_category,
            'name' => $this->name_category,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        return false;
    }

    private function createNewSize()
    {
        $checkSize = DB::table('Luv2_size')
            ->where('company_code', Auth::user()->company_code)
            ->where('code', $this->code_size)
            ->first();

        if ($checkSize) {
            $this->addError('code_size', 'Code size sudah digunakan.');
            return true;
        }

        $lastFourDigits = substr(Auth::user()->company_code, -4);
        $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');

        DB::table('Luv2_size')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => $this->code_size,
            'name' => $this->name_size,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
            'unique_code' => $lastFourDigits . $this->code_size . $tanggal,
        ]);

        return false;
    }

    private function saveUdfFields()
    {
        foreach ($this->udfFields as $field => $value) {
            DB::table('Luv2_udf_item')->updateOrInsert(
                [
                    'item_code' => $this->code,
                    'code_udf' => $this->activeUDFs->firstWhere('field', $field)->code ?? null,
                    'company_code' => Auth::user()->company_code,
                ],
                [
                    'value' => $value,
                    'updated_at' => Carbon::now(),
                    'created_at' => Carbon::now(),
                ]
            );
        }
    }

    private function createNewSupplier()
    {
        $words = explode(' ', $this->suppliernew_name);
        $firstLetters = '';
        foreach ($words as $word) {
            $firstLetters .= substr($word, 0, 1);
        }
        
        $lastFourDigits = substr(Auth::user()->company_code, -4);
        $seconds = Carbon::now()->format('s');
        $this->supplier_code = $lastFourDigits . $firstLetters . $seconds;

        DB::table('Luv2_supplier')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => $this->supplier_code,
            'name' => $this->suppliernew_name,
            'address' => $this->suppliernew_address,
            'phone' => $this->suppliernew_phone,
            'email' => $this->suppliernew_email,
            'consignment' => 'Y',
            'shared_margin' => $this->shared_margin,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);
    }

    #[On('closeModal')]
    public function closeModal()
    {
        $this->code = '';
        $this->barcode = '';
        $this->name = '';
        $this->brand_code = '';
        $this->category_code = '';
        $this->size_code = '';
        $this->isConsignment = false;
        $this->inventory = '';
        $this->sales = '';
        $this->purchase = '';
        $this->name_brand = '';
        $this->name_size = '';
        $this->name_category = '';
        $this->code_brand = '';
        $this->code_size = '';
        $this->code_category = '';
        $this->whs_code = '';
        $this->unit_code = '';
        $this->name_unit = '';
        $this->unit_label = '';
        $this->canEditSupplier = true;
        $this->isCreateNewBrand = false;
        $this->isCreateNewSize = false;
        $this->isCreateNewCategory = false;

        $this->dispatch('closemodalCreateItem');
        if ($this->isEdit) {
            $msg = 'Data item berhasil dirubah';

        } else {
            $msg = 'Data item berhasil dibuat';

        }

        $this->alert('success', $msg, [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);

        $this->isEdit = false;

        $this->dispatch('reGet');
    }
    


    // public function closeModalEdit()
    // {
    //     $this->code = '';
    //     $this->barcode = '';
    //     $this->name = '';
    //     $this->brand_code = '';
    //     $this->category_code = '';
    //     $this->size_code = '';
    //     $this->inventory = '';
    //     $this->sales = '';
    //     $this->purchase = '';
    //     $this->name_brand = '';
    //     $this->name_size = '';
    //     $this->name_category = '';
    //     $this->code_brand = '';
    //     $this->code_size = '';
    //     $this->code_category = '';
    //     $this->whs_code = '';
    //     $this->unit_code = '';
    //     $this->name_unit = '';
    //     $this->unit_label = '';

    //     $this->dispatch('closemodalCreateItem');

    // }

    #[On('isEdit')]
    public function isEdit($code)
    {
        $item = DB::table('Luv2_item as a')
            ->select([
                'a.*',
                DB::raw('COALESCE((
                    SELECT SUM(qty)
                    FROM "Luv2_item_trans"
                    WHERE item_code = a.code
                    AND company_code = \'' . auth()->user()->company_code . '\'
                    AND whs_code = \'' . auth()->user()->whs_code . '\'
                ), 0) AS trans_qty')
            ])
            ->where('a.company_code', auth()->user()->company_code)
            ->where('a.code', $code)
            ->first();
        $isConsignment = $item->consignment == 'Y' ? true : false;

        if ($isConsignment) {
            $supplier = DB::table('Luv2_supplier')->where('code', $item->supp_code)->first();
            $this->supplier_code = $supplier->code;
        }

        $this->code = $item->code;
        $this->barcode = $item->barcode;
        $this->name = $item->name;
        $this->brand_code = $item->brand_code;
        $this->category_code = $item->category_code;
        $this->size_code = $item->size_code;
        $this->inventory = $item->inventory == 'Y' ? true : false;
        $this->isConsignment = $isConsignment;
        $this->supplier_code = $isConsignment ? $supplier->code : '';
        $this->sales = $item->sales == 'Y' ? true : false;
        $this->purchase = $item->purchase == 'Y' ? true : false;

        // Fetch active UDFs for the item
        $activeUDFs = DB::table('Luv2_udf')
            ->where('company_code', Auth::user()->company_code)
            ->where('active', 'Y')
            ->get();

        if ($activeUDFs->isNotEmpty()) {
            foreach ($activeUDFs as $udf) {
                $udfValue = DB::table('Luv2_udf_item')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('code_udf', $udf->code)
                    ->where('item_code', $item->code)
                    ->first();
                $this->udfFields[$udf->field] = $udfValue->value ?? '';
            }
        } else {
            $this->udfFields = [];
        }

        $this->isEdit = true;
        $this->canEditSupplier = ($item->trans_qty == 0) ? true : false;
        $this->dispatch('toOpenModal');
    }

    #[On('toOpenModal')]
    public function openModal()
    {
        $this->dispatch('openModalforEdit', [
            'code' => $this->code,
            'barcode' => $this->barcode,
            'name' => $this->name,
            'unit_code' => $this->unit_code,
            'qtyunit' => $this->qtyunit,
            'brand_code' => $this->brand_code,
            'category_code' => $this->category_code,
            'size_code' => $this->size_code,
            'inventory' => $this->inventory,
            'consignment' => $this->isConsignment,
            'supp_code' => $this->supplier_code,
            'sales' => $this->sales,
            'purchase' => $this->purchase,
        ]);
        $this->dispatch('refreshjs');
    }

    public function saveUpdate()
    {
        try {
            $this->validate($this->rules());

            // Check for duplicate barcode if changed
            if ($this->barcode) {
                $existingBarcode = DB::table('Luv2_item')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('barcode', $this->barcode)
                    ->where('code', '!=', $this->code)
                    ->first();

                if ($existingBarcode) {
                    $this->addError('barcode', 'Barcode sudah digunakan.');
                    return;
                }
            }

            DB::beginTransaction();
            try {
                if ($this->isCreateNewBrand) {
                    $checkBrand = DB::table('Luv2_brand')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $this->code_brand)
                        ->first();

                    if ($checkBrand) {
                        $this->addError('code_brand', 'Kode brand sudah digunakan.');
                        DB::rollBack();
                        return;
                    }

                    DB::table('Luv2_brand')->insert([
                        'company_code' => Auth::user()->company_code,
                        'code' => $this->code_brand,
                        'name' => $this->name_brand,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ]);
                }

                if ($this->isCreateNewCategory) {
                    $checkCategory = DB::table('Luv2_category')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $this->code_category)
                        ->first();

                    if ($checkCategory) {
                        $this->addError('code_category', 'Kode kategori sudah digunakan.');
                        DB::rollBack();
                        return;
                    }

                    DB::table('Luv2_category')->insert([
                        'company_code' => Auth::user()->company_code,
                        'code' => $this->code_category,
                        'name' => $this->name_category,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ]);
                }

                if ($this->isCreateNewSize) {
                    $checkSize = DB::table('Luv2_size')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $this->code_size)
                        ->first();

                    if ($checkSize) {
                        $this->addError('code_size', 'Kode ukuran sudah digunakan.');
                        DB::rollBack();
                        return;
                    }

                    $lastFourDigits = substr(Auth::user()->company_code, -4);
                    $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');
                    DB::table('Luv2_size')->insert([
                        'company_code' => Auth::user()->company_code,
                        'code' => $this->code_size,
                        'name' => $this->name_size,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                        'unique_code' => $lastFourDigits . $this->code_size . $tanggal,
                    ]);
                }

                // Handle UDF fields
                if ($this->udfFields) {
                    foreach ($this->udfFields as $field => $value) {
                        DB::table('Luv2_udf_item')->updateOrInsert(
                            [
                                'item_code' => $this->code,
                                'code_udf' => $this->activeUDFs->firstWhere('field', $field)->code ?? null,
                                'company_code' => Auth::user()->company_code,
                            ],
                            [
                                'value' => $value,
                                'updated_at' => Carbon::now(),
                                'created_at' => Carbon::now(),
                            ]
                        );
                    }
                }

                if ($this->isCreateNewSupplier) {
                    $words = explode(' ', $this->suppliernew_name);
                    $firstLetters = '';
                    foreach ($words as $word) {
                        $firstLetters .= substr($word, 0, 1);
                    }
                    $lastFourDigits = substr(Auth::user()->company_code, -4);
                    $seconds = Carbon::now()->format('s');
                    $this->supplier_code = $lastFourDigits . $firstLetters . $seconds;

                    DB::table('Luv2_supplier')->insert([
                        'company_code' => Auth::user()->company_code,
                        'code' => $this->supplier_code,
                        'name' => $this->suppliernew_name,
                        'address' => $this->suppliernew_address,
                        'phone' => $this->suppliernew_phone,
                        'email' => $this->suppliernew_email,
                        'consignment' => 'Y',
                        'shared_margin' => $this->shared_margin,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                    ]);
                }

                $updateData = [
                    'barcode' => $this->barcode,
                    'name' => $this->name,
                    'brand_code' => $this->code_brand !== '' ? $this->code_brand : $this->brand_code,
                    'category_code' => $this->code_category !== '' ? $this->code_category : $this->category_code,
                    'size_code' => $this->code_size !== '' ? $this->code_size : $this->size_code,
                    'status' => 'Y',
                    'inventory' => $this->inventory ? 'Y' : 'N',
                    'sales' => $this->sales ? 'Y' : 'N',
                    'purchase' => $this->purchase ? 'Y' : 'N',
                    'active' => 'Y',
                    'updated_at' => Carbon::now(),
                ];

                if ($this->canEditSupplier) {
                    $updateData['consignment'] = $this->isConsignment ? 'Y' : 'N';
                    $updateData['supp_code'] = $this->isConsignment ? $this->supplier_code : '';
                }

                DB::table('Luv2_item')
                    ->where('company_code', auth()->user()->company_code)
                    ->where('code', $this->code)
                    ->update($updateData);

                DB::commit();

                $this->alert('success', 'Item berhasil diperbarui', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);

                $this->dispatch('closeModal');
                $this->dispatch('afterSave');

            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }

        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('Database error in saveUpdate(): ' . $e->getMessage());
            $this->alert('error', 'Terjadi kesalahan pada database. Silakan coba lagi.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        } catch (\Exception $e) {
            Log::error('Error in saveUpdate(): ' . $e->getMessage());
            $this->alert('error', 'Terjadi kesalahan yang tidak terduga. Silakan coba lagi.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }


    #[On('afterSave')]
    public function dataItemeafterSave()
    {
        $this->item = Item::where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->with(['brand', 'size', 'category'])
            ->get();

        $this->dispatch('refreshjs');
    }

    #[On('openSupplierModal')]
    public function openSupplierModal($code)
    {
        $item = DB::table('Luv2_item')->where('code', $code)->first();
        $supplier = DB::table('Luv2_supplier')->where('code', $item->supp_code)->first();
        $this->infoitem_name = $item->name;
        $this->infoitem_code = $item->code;
        $this->infoitem_barcode = $item->barcode;
        $this->infosupplier_name = $supplier->name;
        $this->infosupplier_code = $supplier->code;

        $this->dispatch('openModalInfoSupplier');
    }

    #[On('resetTheSession')]
    public function clearSes()
    {
        session()->forget('fromCashier');
    }

    #[On('markAsDeleted')]
    public function deleteItem($code)
    {
        $itemtrans = DB::table('Luv2_item_trans')->where('company_code', Auth::user()->company_code)->where('item_code', $code)->first();
        if ($itemtrans) {
            $this->alert('warning', 'Item tidak bisa dihapus karena sudah ada transaksi', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        } else {
            DB::table('Luv2_item')
                ->where('company_code', auth()->user()->company_code)
                ->where('code', $code)
                ->delete();
            $this->alert('success', 'Item berhasil dihapus', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
        $this->dispatch('refreshjs');
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function getHasDuplicatesProperty()
    {
        if (empty($this->removedDuplicates)) {
            return false;
        }
        $duplicateInfo = $this->checkForDuplicates($this->removedDuplicates);
        return $duplicateInfo['hasDuplicates'];
    }

    private function hasConsignmentWithoutSupplier()
    {
        if (empty($this->chunks)) {
            return false;
        }

        foreach ($this->chunks as $chunk) {
            if (($chunk['consignment'] ?? 'N') === 'Y' && empty($chunk['supplier_code'])) {
                return true;
            }
        }
        return false;
    }

    // Add methods to clear error bags when updating fields
    public function updated($field)
    {
        $this->resetErrorBag($field);
        $this->resetValidation($field);
    }

    public function updatedCode()
    {
        $this->resetErrorBag('code');
        $this->resetValidation('code');
    }

    public function updatedBarcode()
    {
        $this->resetErrorBag('barcode');
        $this->resetValidation('barcode');
    }

    public function updatedName()
    {
        $this->resetErrorBag('name');
        $this->resetValidation('name');
    }
}
