<?php

namespace App\Livewire\Pos\Item;

use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Maatwebsite\Excel\Facades\Excel;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Log;

class InsertCategory extends Component
{
    use WithFileUploads;
    use LivewireAlert;
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $perPage = 10;
    public $search = '';
    public $sortColumn = 'name';
    public $sortDirection = 'ASC';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortColumn' => ['except' => 'name'],
        'sortDirection' => ['except' => 'ASC']
    ];

    #[Rule('required', message: 'Code field is required.')]
    #[Rule('min:3', message: 'Code field must be at least 3 characters.')]
    #[Rule('unique:Luv2_category,code', message: 'Code has been taken and must be unique.')]
    public string $code;


    #[Rule('required', message: 'Name field is required.')]
    #[Rule('min:4', message: 'Name field must be at least 4 characters.')]
    public string $name = '';

    public $file;
    public $chunks;
    public $editChunks = false;
    public $isTable = true;
    public $itemCategory;
    public $warehouses;
    public $whs_code;

    public $isEdit = false;
    public $removedDuplicates = [];

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function render()
    {
        $query = DB::table('Luv2_category')
            ->where('company_code', Auth::user()->company_code);

        // Apply search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('code', 'like', '%' . $this->search . '%')
                  ->orWhere('name', 'like', '%' . $this->search . '%');
            });
        }

        // Apply sorting
        $query->orderBy($this->sortColumn, $this->sortDirection);
        
        // Get paginated results
        $categories = $query->paginate($this->perPage);

        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        if(Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        }

        return view('livewire.pos.item.insert-category', [
            'categories' => $categories
        ]);
    }

    public function loadCategories()
    {
        // This method is now only used to refresh the data
        $this->dispatch('$refresh');
    }

    public function importItem()
    {
        $data = $this->validate([
            'file' => 'required|file|mimes:csv,xlsx,txt',
        ]);
        try {
            $fileExtension = $this->file->getClientOriginalExtension();

            if ($fileExtension === 'csv' || $fileExtension === 'txt') {
                $this->importCSV();
            } elseif ($fileExtension === 'xlsx') {
                $this->importXLSX();
            } else {
                $this->alert('warning', 'Format file tidak didukung', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true
                ]);
            }
        } catch (\Throwable $th) {
            $this->alert('warning', 'Error, Format template salah!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
        }
    }

    #[On('isEdit')]
    public function isEdit($code)
    {
        $this->isEdit = true;
        $category = DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)->where('code', $code)->first();
        if ($category) {
            $this->code = $category->code;
            $this->name = $category->name;
        }
        $this->dispatch('openModalEdit', $this->code, $this->name);
    }

    public function saveUpdate()
    {
        DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)->where('code', $this->code)->update([
            'name' => $this->name,
            'updated_at' => Carbon::now(),
        ]);
        
        $this->alert('success', 'Data Category berhasil dirubah', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
        
        $this->dispatch('clearAfterCloseModalEdit');
        $this->dispatch('refreshjs');
        $this->isEdit = false;
    }
    

    #[On('deleteCategory')]
    public function deleteCategory($code)
    {
        $item = DB::table('Luv2_item')->where('company_code', Auth::user()->company_code)->where('category_code', $code)->first();
        if ($item) {
            $this->alert('error', 'Data Category tidak dapat dihapus, karena masih digunakan pada item', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
        } else {
            DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)->where('code', $code)->delete();
            $this->alert('success', 'Data Category berhasil dihapus', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
        }
        $this->dispatch('refreshjs');
    }
    

    public function saveChunks()
    {
        DB::beginTransaction();
        try {
            $userId = Auth::user()->id;
            
            // Get list of codes that exist in database
            $codes = array_column($this->chunks, 'code');
            $existingItems = DB::table('Luv2_category')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $codes)
                ->pluck('code')
                ->toArray();

            $updateCount = 0;
            $insertCount = 0;

            foreach ($this->chunks as $chunk) {
                $data = [
                    'company_code' => Auth::user()->company_code,
                    'code' => $chunk['code'],
                    'name' => $chunk['name'],
                    'updated_at' => Carbon::now(),
                ];

                if (in_array($chunk['code'], $existingItems)) {
                    // Update existing record
                    DB::table('Luv2_category')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $chunk['code'])
                        ->update($data);
                    $updateCount++;
                } else {
                    // Insert new record
                    $data['created_at'] = Carbon::now();
                    DB::table('Luv2_category')->insert($data);
                    $insertCount++;
                }
            }

            DB::commit();

            $message = "Data berhasil disimpan ke database. ";
            if ($insertCount > 0) {
                $message .= "Menambahkan $insertCount data baru. ";
            }
            if ($updateCount > 0) {
                $message .= "Memperbarui $updateCount data yang sudah ada.";
            }

            $this->alert('success', $message, [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => false,
            ]);

            // Reset the import state
            $this->chunks = [];
            $this->removedDuplicates = [];
            $this->file = null;
            $this->editChunks = false;
            $this->isTable = true;

            $this->dispatch('successImport');
            $this->dispatch('refreshDatatable');
            $this->dispatch('success-import');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving chunks: ' . $e->getMessage());
            $this->alert('error', 'Error saat menyimpan data: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
        }
    }

    private function importCSV()
    {
        try {
            $file = fopen($this->file->getRealPath(), 'r');
            if ($file === false) {
                throw new \Exception('Tidak dapat membuka file');
            }

            $skipFirstRow = true;
            $chunks = [];
            $this->removedDuplicates = [];
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];

            // Try to detect the delimiter from the first line
            $firstLine = fgets($file);
            rewind($file);
            
            // Count occurrences of potential delimiters
            $commaCount = substr_count($firstLine, ',');
            $tabCount = substr_count($firstLine, "\t");
            $semicolonCount = substr_count($firstLine, ';');
            
            // Choose the delimiter with the highest count
            $delimiter = ',';  // default
            $maxCount = $commaCount;
            
            if ($tabCount > $maxCount) {
                $delimiter = "\t";
                $maxCount = $tabCount;
            }
            if ($semicolonCount > $maxCount) {
                $delimiter = ";";
            }

            while (($data = fgetcsv($file, 0, $delimiter)) !== false) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($data))) {
                    continue;
                }

                $processedData = [
                    'code' => strval(trim($data[0])),
                    'name' => trim($data[1])
                ];

                // Validate required fields
                if (empty($processedData['code'])) {
                    $errors[] = "Row {$rowNumber}: Code is required";
                    continue;
                }
                if (empty($processedData['name'])) {
                    $errors[] = "Row {$rowNumber}: Name is required";
                    continue;
                }

                if (isset($seenCodes[$processedData['code']])) {
                    $processedData['duplicate_type'] = 'Duplikat dalam File Import';
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                $seenCodes[$processedData['code']] = true;
                $chunks[] = $processedData;
            }
            
            fclose($file);

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            if (empty($chunks)) {
                throw new \Exception('Tidak ada data valid dalam file CSV. Mohon periksa format template.');
            }

            $this->chunks = $chunks;

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data duplikat yang telah dihapus. Anda dapat mengekspor data duplikat menggunakan tombol 'Export Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => true
                ]);
                
                // Dispatch event to show success message after 5 seconds
                $this->dispatch('showSuccessImport');
            } else {
                $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true
                ]);
            }
            
            $this->dispatch('refreshjs');
            
        } catch (\Exception $e) {
            $this->alert('error', 'Error saat mengimpor CSV: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 10000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => true
            ]);
        }
    }

    private function importXLSX()
    {
        try {
            $rows = Excel::toArray([], $this->file->getRealPath())[0];
            if (count($rows) <= 1) {
                throw new \Exception('Tidak ada data dalam file Excel.');
            }

            $chunks = [];
            $this->removedDuplicates = [];
            $skipFirstRow = true;
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];

            foreach ($rows as $row) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($row))) {
                    continue;
                }

                $processedData = [
                    'code' => strval(trim($row[0])),
                    'name' => trim($row[1])
                ];

                // Validate required fields
                if (empty($processedData['code'])) {
                    $errors[] = "Row {$rowNumber}: Code is required";
                    continue;
                }
                if (empty($processedData['name'])) {
                    $errors[] = "Row {$rowNumber}: Name is required";
                    continue;
                }

                if (isset($seenCodes[$processedData['code']])) {
                    $processedData['duplicate_type'] = 'Duplikat dalam File Import';
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                $seenCodes[$processedData['code']] = true;
                $chunks[] = $processedData;
            }

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            if (empty($chunks)) {
                throw new \Exception('Tidak ada data valid dalam file Excel. Mohon periksa format template.');
            }

            $this->chunks = $chunks;

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data duplikat yang telah dihapus. Anda dapat mengekspor data duplikat menggunakan tombol 'Export Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => true
                ]);
                
                // Dispatch event to show success message after 5 seconds
                $this->dispatch('showSuccessImport');
            } else {
                $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true
                ]);
            }

            $this->dispatch('refreshjs');

        } catch (\Exception $e) {
            $this->alert('error', 'Error saat mengimpor Excel: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
        }
    }

    #[On('showSuccessImport')] 
    public function showSuccessImport()
    {
        $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
            'position' => 'top-end',
            'timer' => 5000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
    }

    public function modeEditTrue()
    {
        $this->editChunks = true;
    }

    public function modeEditFalse()
    {
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }

    public function cancelImport()
    {
        $this->chunks = [];
        $this->removedDuplicates = [];
        $this->file = null;
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }

    public function modeEdit()
    {
        $this->editChunks = !$this->editChunks;

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    public function save()
    {
        $this->validate();

        DB::table('Luv2_category')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => $this->code,
            'name' => $this->name,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        $this->alert('success', 'Data Category berhasil disimpan', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
        
        // Clear the inputs
        $this->code = '';
        $this->name = '';
        
        $this->dispatch('closemodalCreate');
        $this->dispatch('afterSave');
    }

    #[On('closeModal')]
    public function closeModal()
    {
        $this->code = '';
        $this->name = '';
        $this->dispatch('closemodalCreate');
        
        $this->alert('success', 'Data Category berhasil disimpan', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
    }

    public function changeTable()
    {
        $this->isTable = !$this->isTable;

        $this->chunks = [];
        $this->file = '';

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    #[On('afterSave')]
    public function dataCategoryafterSave()
    {
        $this->itemCategory = DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->get();
    }

    public function downloadDuplicates($format = 'xlsx')
    {
        try {
            if (empty($this->removedDuplicates)) {
                $this->alert('error', 'Tidak ada data duplikat yang tersedia untuk diekspor', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true
                ]);
                return null;
            }

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "duplicate_categories_{$timestamp}";

            if ($format === 'csv') {
                $filename .= '.csv';
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => "attachment; filename={$filename}",
                ];

                $handle = fopen('php://temp', 'r+');
                
                // Add headers
                fputcsv($handle, ['Code', 'Name', 'Duplicate Type']);
                
                // Add data
                foreach ($this->removedDuplicates as $item) {
                    fputcsv($handle, [
                        $item['code'] ?? '',
                        $item['name'] ?? '',
                        $item['duplicate_type'] ?? 'Import File Duplicate'
                    ]);
                }

                rewind($handle);
                $content = stream_get_contents($handle);
                fclose($handle);

                return response($content, 200, $headers);
            } 
            
            // Excel export
            $filename .= '.xlsx';
            return Excel::download(new class($this->removedDuplicates) implements 
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return ['Code', 'Name', 'Duplicate Type'];
                }
            }, $filename);

        } catch (\Exception $e) {
            Log::error('Error in downloadDuplicates: ' . $e->getMessage());
            $this->alert('error', 'Error saat mengunduh data duplikat: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
            return null;
        }
    }

    public function getHasDuplicatesProperty()
    {
        return !empty($this->removedDuplicates);
    }
}
