<?php

namespace App\Livewire\Pos\It\ItReqOut;

use App\Exports\ExportITReqOut;
use App\Models\ItReqOut;
use App\Models\Settings;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;

class ItReqOutTable extends Component
{

    use LivewireAlert;

    public $disableButton = false;
    public $its;
    public $selectedOption = 'Default';
    public $totalSumQty;

    public function render()
    {
        return view('livewire.pos.it.it-req-out.it-req-out-table');
    }

    #[On('successSync')]
    public function mount()
    {
        $its = ItReqOut::with(['whsTo', 'details'])
            ->where('canceled', 'N')
            ->whereNull('entry')
            ->withSum('details', 'qty_send')
            ->orderBy('Luv2_it_req_out.date', 'desc')
            ->orderBy('Luv2_it_req_out.no', 'desc')
            ->get();

        if ($its->isEmpty()) {
            $this->its = [];
            $this->totalSumQty = 0;
        } else {
            $totalSumQty = $its->sum('details_sum_qty_send');
            $this->its = $its;
            $this->totalSumQty = $totalSumQty;
        }
    }

    public function detail($it)
    {
        return redirect()->route('it_in_detail', ['it' => $it]);
    }

    #[On('updateSelectedOption')]
    public function updatedSelect($selectedOption)
    {
        $this->selectedOption = $selectedOption;

        $this->dispatch('updateData');
    }

    #[On('updateData')]
    public function updatedIts()
    {
        $query = ItReqOut::with(['whsTo', 'details'])
            ->where('canceled', 'N')
            ->withSum('details', 'qty_send')
            ->orderBy('Luv2_it_req_out.date', 'desc')
            ->orderBy('Luv2_it_req_out.no', 'desc');

        // Apply conditions based on selected option
        if ($this->selectedOption === 'Updated') {
            $query->whereNotNull('entry');
        } elseif ($this->selectedOption === 'Default') {
            $query->whereNull('entry');
        }

        $its = $query->get();

        if ($its->isEmpty()) {
            $this->its = [];
            $this->totalSumQty = 0;
        } else {
            $totalSumQty = $its->sum('details_sum_qty_send');
            $this->its = $its;
            $this->totalSumQty = $totalSumQty;
        }

        $this->dispatch('refreshjs');
    }

    #[On('refresh')]
    public function updated() {}

    #[On('toSyncITReqOut')]
    public function toSyncITReqOut()
    {
        $this->disableButton = true;
        $this->dispatch('syncITReq');
    }

    #[On('toSyncITReqOut')]
    public function syncITReqOut($no)
    {
        try {
            $setting = Settings::first();

            if (!$setting) {
                throw new \Exception('Settings not found. Please check your configuration.');
            }

            $itreqout = ItReqOut::with(['whsTo', 'details'])
                ->where('canceled', 'N')
                ->where('no', $no)
                ->withSum('details', 'qty_send')
                ->orderBy('Luv2_it_req_out.date', 'desc')
                ->orderBy('Luv2_it_req_out.no', 'desc')
                ->first();

            if (!$itreqout) {
                throw new \Exception('IT Request Out data not found for no: ' . $no);
            }

            $itreqoutArray = $itreqout->toArray();

            $itreqoutdetail = DB::table('Luv2_it_req_out_detail')
                ->join('Luv2_it_req_out', 'Luv2_it_req_out_detail.no', '=', 'Luv2_it_req_out.no')
                ->join('Luv2_item', 'Luv2_it_req_out_detail.item_code', '=', 'Luv2_item.code')
                ->select(
                    'Luv2_item.code',
                    'Luv2_item.barcode',
                    'Luv2_item.name',
                    DB::raw('SUM("Luv2_it_req_out_detail"."qty") as "qty"'),
                    DB::raw('SUM("Luv2_it_req_out_detail"."qty_send") as "qty_send"')
                )
                ->where('Luv2_it_req_out.no', $no)
                ->groupBy('Luv2_item.code', 'Luv2_item.barcode', 'Luv2_item.name')
                ->get();

            if ($itreqoutdetail->isEmpty()) {
                throw new \Exception('No details found for IT Request Out no: ' . $no);
            }

            $syncItems = [];
            $linenum = 0; // Initialize the line number

            foreach ($itreqoutdetail as $it) {
                $syncItems[] = [
                    'no' => $no,
                    'id_company' => $setting->id_company,
                    'linenum' => $linenum++,
                    'qty' => $it->qty_send,
                    'qty_receive' => $it->qty_send,
                    'item_name' => $it->name,
                    'item_barcode' => $it->barcode,
                    'item_code' => $it->code,
                ];
            }

            $formattedData = [
                'id_company' => $setting->id_company,
                'no' => $itreqoutArray['no'],
                'id_whs' => $itreqoutArray['whs_code'],
                'id_whs_to' => $itreqoutArray['whs_code_to'],
                'date' => $itreqoutArray['date'],
                'id_user' => $itreqoutArray['id_user'],
                'comments' => $itreqoutArray['comments'],
                'details' => $syncItems,
            ];

            $urladmin = $setting->url_admin;

            $response = Http::post($urladmin . 'itreqout', $formattedData);
            Log::info('API Response: ', [
                'status' => $response->status(),
                'body' => $response->body(),
            ]);

            if ($response->successful() && $response->json('success') === true) {
                $itreqoutModel = ItReqOut::where('no', $no)->first();
                $itreqoutModel->sync = 'Y';
                $itreqoutModel->save();

                $this->alert('success', 'IT Request Out Sync successfully', [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                $this->dispatch('successSync');
            } else {
                $errorMessage = $response->json('message') . ' ' . $response->json('error');
                throw new \Exception('Sync failed: ' . $errorMessage);
            }
        } catch (\Throwable $e) {
            // Log the exception
            Log::error('Error during IT Request Out Sync: ', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            // Display error message to the user
            $this->alert('error', 'An error occurred: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->dispatch('refreshjs');
        } finally {
            $this->disableButton = false;
        }
    }

    #[On('exportITReqOut')]
    public function exportExcelITReqout($no)
    {
        $filename = "IT Request Out - {$no}.xlsx";
        $this->dispatch('updateData');
        return Excel::download(new ExportITReqOut($no), $filename);
    }
}
