<?php

namespace App\Http\Controllers;

use App\Jobs\CreateDataInFirstJob;
use App\Mail\VerifyEmail;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\ValidationException;

class RegisterAPIController extends Controller
{
    public function register(Request $request)
    {
        $token = $request->header('Authorization');

        if ($token) {
            $token = str_replace('Bearer ', '', $token);
        }

        $storedToken = env('API_TOKEN');

        if ($token !== $storedToken) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        try {
            // Validate request
            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'address' => 'required|string',
                'phone' => 'required|string|max:15',
                'email' => 'required|email|unique:Luv2_user,email',
                'username' => 'required|string|max:255|unique:Luv2_user,username',
                'password' => 'required|string|min:8|confirmed',
                'plan' => 'required|exists:Luv2_paket,name',
            ]);

            $code = Carbon::now()->format('ymdHis');

            DB::beginTransaction();

            DB::table('Luv2_company')->insert([
                'code' => $code,
                'name' => $validatedData['name'],
                'address' => $validatedData['address'],
                'phone' => $validatedData['phone'],
                'email' => $validatedData['email'],
                'plan' => $validatedData['plan'],
                'updated_at' => Carbon::now(),
            ]);

            $maxId = DB::table('Luv2_user')->max('id');
            DB::table('Luv2_user')->insert([
                'id' => $maxId + 1,
                'company_code' => $code,
                'whs_code' => '-',
                'code' => $code,
                'email' => $validatedData['email'],
                'username' => $validatedData['username'],
                'password' => Hash::make($validatedData['password']),
                'active' => 'Y',
                'role' => 'Admin',
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);

            DB::commit();

            CreateDataInFirstJob::dispatch($code, $validatedData, $maxId);

            $this->sendEmailVerification($validatedData['email'], $validatedData['username'], $validatedData['name'], $code);
            $this->sendEmailLog($request->all(), 'Success');

            return response()->json(['message' => 'Registration successful, please check your email to verify your account!'], 201);
        } catch (ValidationException $e) {
            Log::error('Gagal Register Data: ' . json_encode($request->all(), JSON_PRETTY_PRINT));
            $this->sendEmailLog($request->all(), 'Error', $e->errors());
            return response()->json([
                'errors' => $e->errors(),
                'message' => 'Validation failed',
            ], 422);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to register new user!', 'error' => $e], 500);
        }
    }


    public function register2(Request $request)
    {
        $token = $request->header('Authorization');

        if ($token) {
            $token = str_replace('Bearer ', '', $token);
        }

        $storedToken = env('API_TOKEN');

        if ($token !== $storedToken) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        try {
            // Validate request
            $validatedData = $request->validate([
                'email' => 'required|email|unique:Luv2_user,email',
            ]);

            $validatedData['plan'] = 'Pro';
            $validatedData['name'] = explode('@', $validatedData['email'])[0];
            $validatedData['address'] = '-';
            $validatedData['phone'] = '-';
            $validatedData['username'] = explode('@', $validatedData['email'])[0];
            $validatedData['password'] = explode('@', $validatedData['email'])[0];

            $code = Carbon::now()->format('ymdHis');

            DB::beginTransaction();

            DB::table('Luv2_company')->insert([
                'code' => $code,
                'name' => $validatedData['name'],
                'address' => $validatedData['address'],
                'phone' => $validatedData['phone'],
                'email' => $validatedData['email'],
                'plan' => $validatedData['plan'],
                'updated_at' => Carbon::now(),
            ]);

            $maxId = DB::table('Luv2_user')->max('id');
            DB::table('Luv2_user')->insert([
                'id' => $maxId + 1,
                'company_code' => $code,
                'whs_code' => '-',
                'code' => $code,
                'email' => $validatedData['email'],
                'username' => $validatedData['username'],
                'password' => Hash::make($validatedData['password']),
                'active' => 'Y',
                'role' => 'Admin',
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);

            DB::commit();

            CreateDataInFirstJob::dispatch($code, $validatedData, $maxId);

            $this->sendEmailVerification($validatedData['email'], $validatedData['username'], $validatedData['name'], $code);
            $this->sendEmailLog($request->all(), 'Success');

            return response()->json(['message' => 'Registration successful, please check your email to verify your account!'], 201);
        } catch (ValidationException $e) {
            Log::error('Gagal Register Data: ' . json_encode($request->all(), JSON_PRETTY_PRINT));
            $this->sendEmailLog($request->all(), 'Error', $e->errors());
            return response()->json([
                'errors' => $e->errors(),
                'message' => 'Validation failed',
            ], 422);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to register new user!', 'error' => $e], 500);
        }
    }


    public function sendEmailLog($data, $status, $errors = null)
    {
        try {
            $emailContent = 'Register Data: ' . json_encode($data, JSON_PRETTY_PRINT);
            if ($errors) {
                $emailContent .= "\n\nValidation Errors: " . json_encode($errors, JSON_PRETTY_PRINT);
            }

            Mail::raw($emailContent, function ($message) use ($status) {
                $message->to('itm@softcomp.io')
                    ->subject('Registration ' . $status . ' Report');
            });
        } catch (\Exception $e) {
            Log::error('Failed to send error report email: ' . $e->getMessage());
        }
    }

    public function sendEmailVerification($email, $username, $name, $companyCode)
    {
        try {
            $payload = [
                'company_code' => $companyCode,
                'timestamp' => now()->timestamp,
            ];

            $encryptedId = Crypt::encryptString(json_encode($payload));
            Mail::to($email)->queue(new VerifyEmail($name, $username, $encryptedId));
        } catch (\Exception $e) {
        }
    }

    public function getUserBySocialToken(Request $request)
    {
        $token = $request->header('Authorization');

        if ($token) {
            $token = str_replace('Bearer ', '', $token);
        }

        $storedToken = env('API_TOKEN');

        if ($token !== $storedToken) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        try {
            $user_token = $request->input('social_token');

            $user = DB::table('Luv2_user')
                ->where('social_token', $user_token)
                ->select('company_code', 'code', 'username')
                ->first();

            if ($user) {
                return response()->json(['user' => $user], 200);
            } else {
                return response()->json(['message' => 'User not found'], 404);
            }
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to get user by session token'], 500);
        }
    }

    public function updateAfterRegisterBySocial(Request $request)
    {
        $token = $request->header('Authorization');

        if ($token) {
            $token = str_replace('Bearer ', '', $token);
        }

        $storedToken = env('API_TOKEN');

        if ($token !== $storedToken) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized access.'
            ], 401);
        }

        $validatedData = $request->validate([
            'company_code' => 'required|string|exists:Luv2_company,code',
            'code' => 'required|string|exists:Luv2_user,code',
            'company_name' => 'required|string|max:255',
            'address' => 'required|string|max:255',
            'phone' => 'required|numeric|max:20',
            'username' => 'required|string|max:50|unique:Luv2_user,username',
            'password' => 'required|string|min:6',
        ]);

        $company = DB::table('Luv2_company')->where('code', $validatedData['company_code'])->first();
        $user = DB::table('Luv2_user')->where('code', $validatedData['code'])->first();

        if (!$company || !$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'Company or user not found.'
            ], 404);
        }
        $validatedData['email'] = $user->email;
        $validatedData['name'] = $validatedData['company_name'];
        $validatedData['plan'] = 'Pro';

        DB::beginTransaction();

        try {
            DB::table('Luv2_company')->where('code', $validatedData['company_code'])->update([
                'name' => $validatedData['company_name'],
                'address' => $validatedData['address'],
                'phone' => $validatedData['phone'],
                'plan' => $validatedData['plan'],
                'updated_at' => now()
            ]);

            $updateData = [
                'password' => Hash::make($validatedData['password']),
                'updated_at' => now(),
                'social_token' => null
            ];

            if ($validatedData['username'] !== $user->username) {
                $updateData['username'] = $validatedData['username'];
            }

            DB::table('Luv2_user')->where('code', $validatedData['code'])->update($updateData);

            $user = DB::table('Luv2_user')->where('code', $validatedData['code'])->first();

            $maxId = $user->id - 1;
            DB::commit();

            CreateDataInFirstJob::dispatch($user->code, $validatedData, $maxId);

            return response()->json([
                'status' => 'success',
                'message' => 'Registration updated successfully.',
                'redirect_url' => route('login')
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Update After Register Failed: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update registration. Please try again later.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    public function updateAfterRegisterBySocial2(Request $request)
    {
        $token = $request->header('Authorization');

        if ($token) {
            $token = str_replace('Bearer ', '', $token);
        }

        $storedToken = env('API_TOKEN');

        if ($token !== $storedToken) {
            return response()->json([
                'status' => 'error',
                'message' => 'Unauthorized access.'
            ], 401);
        }

        $validatedData = $request->validate([
            'plan' => 'required|exists:Luv2_paket,name',
            'company_code' => 'required|string|exists:Luv2_company,code',
            'code' => 'required|string|exists:Luv2_user,code',
        ]);

        $company = DB::table('Luv2_company')->where('code', $validatedData['company_code'])->first();
        $user = DB::table('Luv2_user')->where('code', $validatedData['code'])->first();

        if (!$company || !$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'Company or user not found.'
            ], 404);
        }
        $validatedData['name'] = explode('@', $validatedData['email'])[0];
        $validatedData['company_name'] = $company->name;
        $validatedData['address'] = '-';
        $validatedData['phone'] = '-';
        $validatedData['email'] = $user->email;
        $validatedData['username'] = explode('@', $validatedData['email'])[0];
        $validatedData['password'] = explode('@', $validatedData['email'])[0];

        DB::beginTransaction();

        try {
            DB::table('Luv2_company')->where('code', $validatedData['company_code'])->update([
                'plan' => $validatedData['plan'],
                'updated_at' => now()
            ]);

            $updateData = [
                'password' => Hash::make($validatedData['password']),
                'updated_at' => now(),
                'social_token' => null
            ];

            if ($validatedData['username'] !== $user->username) {
                $updateData['username'] = $validatedData['username'];
                DB::table('Luv2_user')->where('code', $validatedData['code'])->update($updateData);
            }

            $user = DB::table('Luv2_user')->where('code', $validatedData['code'])->first();

            $maxId = $user->id - 1;
            DB::commit();

            CreateDataInFirstJob::dispatch($user->code, $validatedData, $maxId);

            return response()->json([
                'status' => 'success',
                'message' => 'Registration updated successfully.',
                'redirect_url' => route('login')
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Update After Register Failed: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to update registration. Please try again later.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }
}
