<?php

namespace App\Http\Controllers;

use App\Mail\PaymentStatusUpdated;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class PaymentSubscriptionController extends Controller
{

    public function pagePay()
    {
        $userAccountCount = User::where('company_code', auth()->user()->company_code)->count();
        $accountLabel = $userAccountCount . ' Account' . ($userAccountCount > 1 ? 's' : '');
        $amount = 250000;
        $totalAmount = $amount * $userAccountCount;
        $company = DB::table('Luv2_company')->where('code', auth()->user()->company_code)->first();

        return view('payment.payment', [
            'company' => $company,
            'userCount' => $accountLabel,
            'amount' => $amount,
            'totalAmount' => $totalAmount,
        ]);
    }

    public function pageUpgrade()
    {
        $company = DB::table('Luv2_company')->where('code', auth()->user()->company_code)->first();

        return view('payment.upgrade', [
            'company' => $company,
        ]);
    }

    public function createDataPay()
    {
        \Midtrans\Config::$serverKey = config('services.midtrans.serverKey');
        \Midtrans\Config::$isProduction = config('services.midtrans.isProduction');
        \Midtrans\Config::$isSanitized = config('services.midtrans.isSanitized');
        \Midtrans\Config::$is3ds = config('services.midtrans.is3ds');

        $user = Auth::user();
        $companyCode = $user->company_code;

        $existingRenewal = DB::table('Luv2_renewal')
            ->where('company_code', $companyCode)
            ->where('status', 'pending')
            ->first();

        if ($existingRenewal) {
            return response()->json([
                'success' => false,
                'message' => 'A pending payment exists. Please complete that payment first.',
                'redirectUrl' => route('paymentable'),
            ]);
        }

        $company = DB::table('Luv2_company')->where('code', $companyCode)->first();
        $orderId = '1MNTH-' . $companyCode . '-' . \Carbon\Carbon::now()->format('YmdHis');
        $userAccountCount = User::where('company_code', $companyCode)->count();
        $accountLabel = $userAccountCount . ' Account' . ($userAccountCount > 1 ? 's' : '');
        $amount = 250000;
        $totalAmount = $amount * $userAccountCount;

        $params = [
            'transaction_details' => [
                'order_id' => $orderId,
                'gross_amount' => $totalAmount,
            ],
            'customer_details' => [
                'first_name' => $user->username,
                'last_name' => $company->name,
                'email' => $user->email,
                'phone' => $company->phone,
            ],
            'item_details' => [
                [
                    'id' => $orderId,
                    'price' => $totalAmount,
                    'quantity' => 1,
                    'name' => 'Subscription 1 Month with ' . $accountLabel,
                ]
            ]
        ];

        $detail = [
            'customer_details' => [
                'first_name' => $user->username,
                'last_name' => $company->name,
                'email' => $user->email,
                'phone' => $company->phone,
            ],
            'item_details' => [
                [
                    'id' => $orderId,
                    'price' => $totalAmount,
                    'quantity' => 1,
                    'name' => 'Subscription 1 Month with ' . $accountLabel,
                ]
            ]
        ];

        $snapToken = \Midtrans\Snap::getSnapToken($params);

        DB::table('Luv2_renewal')->insert([
            'order_id' => $orderId,
            'snap_token' => $snapToken,
            'company_code' => $companyCode,
            'detail' => json_encode($detail),
            'amount' => $totalAmount,
            'status' => 'pending',
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Payment created successfully.',
            'redirectUrl' => route('paymentable'),
        ]);
    }


    public function handleWebhook(Request $request)
    {
        if (!$this->isValidSignature($request)) {
            return response()->json(['message' => 'Invalid signature key'], 403);
        }

        $transactionStatus = $request->transaction_status;
        $orderId = $request->order_id;

        $trf = $this->getTransaction($orderId);

        if ($trf) {
            $emailStatus = $this->processTransactionStatus($transactionStatus, $orderId);

            if ($emailStatus !== 'pending') {
                $this->processCompanyPlan($trf, $emailStatus);
                $this->sendEmailNotification($trf, $orderId, $emailStatus);
            }
        }

        return response()->json(['status' => 'success']);
    }

    private function isValidSignature(Request $request)
    {
        $serverKey = config('services.midtrans.serverKey');
        $hashedKey = hash('sha512', $request->order_id . $request->status_code . $request->gross_amount . $serverKey);
        return $hashedKey === $request->signature_key;
    }

    private function getTransaction($orderId)
    {
        return DB::table('Luv2_renewal')
            ->where('order_id', $orderId)
            ->first();
    }

    private function processTransactionStatus($transactionStatus, $orderId)
    {
        switch ($transactionStatus) {
            case 'capture':
            case 'settlement':
                $this->updatePaymentStatus($orderId, 'success');
                return 'success';
            case 'pending':
                $this->updatePaymentStatus($orderId, 'pending');
                return 'pending';
            case 'expire':
            case 'cancel':
                $this->updatePaymentStatus($orderId, 'failed');
                return 'failed';
            default:
                return response()->json(['status' => 'unknown status'], 400);
        }
    }

    private function processCompanyPlan($trf, $emailStatus)
    {
        if ($emailStatus === 'success') {
            $company = DB::table('Luv2_company')->where('code', $trf->company_code)->first();
            if ($trf->plan === $company->plan) {
                $this->updateCompanyPlan($trf, $company);
            } else {
                $this->upgradeOrDowngradeCompanyPlan($trf, $company);
            }
            $this->handleExcludedUsers($trf);
        }
    }

    private function upgradeOrDowngradeCompanyPlan($trf, $company)
    {
        $isUpgrade = ($company->plan === 'Basic' && $trf->plan === 'Pro');
        $isDowngrade = ($company->plan !== 'Basic' && $trf->plan === 'Basic');

        $currentExpiration = Carbon::parse($company->expired_at);
        $newExpirationDate = $currentExpiration->isFuture()
            ? ($trf->month > 1 ? $currentExpiration->addMonths($trf->month) : $currentExpiration->addMonth())
            : ($trf->month > 1 ? now()->addMonths($trf->month) : now()->addMonth());

        $newPlan = $isUpgrade ? 'Pro' : ($isDowngrade ? 'Basic' : null);

        if ($newPlan) {
            DB::table('Luv2_company')
                ->where('code', $trf->company_code)
                ->update([
                    'plan' => $newPlan,
                    'expired_at' => $newExpirationDate
                ]);
        }
    }

    private function updateCompanyPlan($trf, $company)
    {
        $currentExpiration = Carbon::parse($company->expired_at);
        $expirationDate = $this->calculateNewExpirationDate($currentExpiration, $trf->month);

        DB::table('Luv2_company')
            ->where('code', $trf->company_code)
            ->update([
                'plan' => $trf->plan,
                'expired_at' => $expirationDate
            ]);
    }

    private function calculateNewExpirationDate($currentExpiration, $month)
    {
        if ($month > 1) {
            return $currentExpiration->isFuture()
                ? $currentExpiration->addMonths($month)
                : now()->addMonths($month);
        } else {
            return $currentExpiration->isFuture()
                ? $currentExpiration->addMonth()
                : now()->addMonth();
        }
    }

    private function sendEmailNotification($trf, $orderId, $emailStatus)
    {
        $company = DB::table('Luv2_company')->where('code', $trf->company_code)->first();
        Mail::to($company->email)->send(new PaymentStatusUpdated($orderId, $emailStatus, $company));
    }

    private function handleExcludedUsers($trf)
    {
        if (!empty($trf->user_exclude)) {
            $excludedUsers = json_decode($trf->user_exclude, true);

            if (count($excludedUsers) > 0) {
                $excludedUserCodes = collect($excludedUsers)->pluck('code')->toArray();

                DB::table('Luv2_user')
                    ->whereIn('code', $excludedUserCodes)
                    ->update(['active' => 'N']);
            }
        }
    }

    private function updatePaymentStatus($orderId, $status)
    {
        DB::table('Luv2_renewal')
            ->where('order_id', $orderId)
            ->update(['status' => $status, 'updated_at' => now()]);
    }

    public function tablePay()
    {
        return view('payment.paymenttable');
    }
}
