<?php

namespace App\Http\Controllers;

use App\Models\Auths;
use App\Models\Invoice;
use App\Models\ItOut;
use App\Models\ItReqOut;
use App\Models\Member;
use App\Models\Settings;
use App\Models\Warehouse;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class DashboardController extends Controller
{

    public function dashboard()
    {
        $userId = Auth::id();

        // Ambil tanggal dari session jika ada, jika tidak gunakan default
        $defaultStartDate = now()->subDays(7)->format('Y-m-d');
        $defaultEndDate = now()->format('Y-m-d');

        $startDate = session('global_start_date', $defaultStartDate);
        $endDate = session('global_end_date', $defaultEndDate);
        $intervalDays = Carbon::parse($startDate)->diffInDays(Carbon::parse($endDate));

        // Daftar chart
        $charts = [
            'sales_overview',
            'category_chart',
            'shift_chart',
            'itr_chart',
            'price_chart',
        ];

        $settings = [];

        foreach ($charts as $chartName) {
            $setting = DB::table('Luv2_chart')
                ->where('id_user', $userId)
                ->where('chart_name', $chartName)
                ->orderBy('id', 'desc')
                ->first();

            if ($setting) {
                // Update jika tanggal lama sudah kadaluarsa
                if ($setting->end_date < now()->format('Y-m-d')) {
                    DB::table('Luv2_chart')
                        ->where('id', $setting->id)
                        ->update([
                            'start_date' => $startDate,
                            'end_date' => $endDate,
                            'interval_days' => $intervalDays,
                        ]);
                }
            } else {
                // Simpan pertama kali
                DB::table('Luv2_chart')->insert([
                    'id_user' => $userId,
                    'chart_name' => $chartName,
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'interval_days' => $intervalDays,
                ]);
            }

            // Simpan ke array settings
            $settings[$chartName] = (object)[
                'start_date' => $startDate,
                'end_date' => $endDate,
                'interval_days' => $intervalDays,
            ];
        }

        // Ambil data tambahan
        $salesDataResponse = $this->getSalesData();
        $salesData = $salesDataResponse->getData();

        $company = DB::table('Luv2_company')->where('code', Auth::user()->company_code)->first();
        $nps = DB::table('Luv2_nps')->where('company_code', Auth::user()->company_code)->first();

        $alreadyRate = true;
        if ($company->verified_at && Carbon::parse($company->verified_at)->lte(now()->subMonths(3))) {
            $alreadyRate = false;
        }

        if ($nps) {
            $alreadyRate = true;
        }

        return view('dashboard.index', [
            'startDateSales' => $settings['sales_overview']->start_date,
            'endDateSales' => $settings['sales_overview']->end_date,
            'startDateCategory' => $settings['category_chart']->start_date,
            'endDateCategory' => $settings['category_chart']->end_date,
            'startDateShift' => $settings['shift_chart']->start_date,
            'endDateShift' => $settings['shift_chart']->end_date,
            'startDateItr' => $settings['itr_chart']->start_date,
            'endDateItr' => $settings['itr_chart']->end_date,
            'startDatePrice' => $settings['price_chart']->start_date,
            'endDatePrice' => $settings['price_chart']->end_date,
            'itrData' => $this->getItrData()->getData()->itrData,
            'priceData' => $this->getPriceData()->getData()->priceData,
            'totalSales' => $salesData->totalSales ?? [],
            'atv' => $salesData->atv ?? [],
            'upt' => $salesData->upt ?? [],
            'rate' => $alreadyRate,
        ]);
    }


    public function storeNPS(Request $request)
    {

        $user = Auth::user();

        if (!$user) {
            return response()->json(['message' => 'Unauthorized'], 401);
        }

        if ($request->skipped) {
            return response()->json(['message' => 'User skipped'], 200);
        } else {
            DB::table('Luv2_nps')->insert([
                'company_code' => $user->company_code,
                'id_user' => $user->id,
                'rate' => $request->skipped ? null : $request->score,
                'keterangan' => $request->skipped ? 'User skipped' : $request->comment,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            return response()->json(['message' => 'Data saved']);
        }
    }

    public function getSalesData()
    {
        $userId = Auth::id();
        $companyCode = auth()->user()->company_code;

        // Ambil tanggal global dari session
        $startDate = session('global_start_date', now()->subDays(7)->format('Y-m-d'));
        $endDate = session('global_end_date', now()->format('Y-m-d'));

        $sql = "
        SELECT *
        FROM (
            SELECT
                x.*,
                ROW_NUMBER() OVER (ORDER BY code, date ASC) AS row_num
            FROM (
                SELECT
                    code,
                    name,
                    date,
                    SUM(sales) AS sales,
                    AVG(NULLIF(qty, 0)) AS avgqty,
                    AVG(NULLIF(sales, 0)) AS avgval
                FROM (
                    SELECT
                        c.code,
                        c.name,
                        a.date,
                        a.no,
                        b.linenum,
                        SUM(b.qty) AS qty,
                        SUM(b.total_price) AS sales
                    FROM
                        \"Luv2_invoice\" a
                        INNER JOIN \"Luv2_invoice_detail\" b ON a.no = b.no
                        INNER JOIN \"Luv2_warehouse\" c ON a.whs_code = c.code
                    WHERE
                        a.date BETWEEN :start_date AND :end_date
                        AND b.total_price <> 0
                        AND b.qty > 0
                        AND b.code NOT LIKE 'BUY%'
                        AND b.code NOT LIKE 'FG%'
                        AND b.code NOT LIKE 'PP%'
                        AND b.code NOT LIKE 'VOUCHER%'
                        AND c.company_code = :company_code
                    GROUP BY
                        c.code,
                        c.name,
                        a.date,
                        a.no,
                        b.linenum
                ) AS d
                GROUP BY
                    code,
                    name,
                    date
            ) AS x
        ) AS z
        ORDER BY
            code,
            date ASC
        ";

        $results = DB::select($sql, [
            'start_date' => $startDate,
            'end_date' => $endDate,
            'company_code' => $companyCode
        ]);

        if (empty($results)) {
            return response()->json([
                'success' => false,
                'message' => 'No data available for the selected date range.',
                'salesData' => []
            ]);
        }

        $dates = [];
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);

        while ($start->lte($end)) {
            $dates[] = $start->format('Y-m-d');
            $start->addDay();
        }

        $salesData = [];
        foreach ($results as $record) {
            $salesData[$record->name][$record->date] = floatval($record->sales);
        }

        foreach ($salesData as $name => &$sales) {
            foreach ($dates as $date) {
                if (!isset($sales[$date])) {
                    $sales[$date] = 0;
                }
            }
            ksort($sales);
        }

        // Total Sales dan lainnya
        $totalSales = array_sum(array_map(function ($item) {
            return $item->sales;
        }, $results));

        $totalTransactions = count($results);

        $totalAtv = array_sum(array_map(function ($item) {
            return $item->avgval;
        }, $results));

        $totalUpt = array_sum(array_map(function ($item) {
            return $item->avgqty;
        }, $results));

        $atv = ($totalTransactions > 0) ? $totalAtv / $totalTransactions : 0;
        $upt = ($totalTransactions > 0) ? $totalUpt / $totalTransactions : 0;

        return response()->json([
            'success' => true,
            'start_date_sales' => $startDate,
            'end_date_sales' => $endDate,
            'salesData' => $salesData,
            'totalSales' => $totalSales,
            'atv' => $atv,
            'upt' => $upt
        ]);
    }

    // public function updateSalesRange(Request $request)
    // {
    //     $userId = Auth::id();
    //     $validated = $request->validate([
    //         'start_date_sales' => 'required|date',
    //         'end_date_sales' => 'required|date|after_or_equal:start_date_sales',
    //     ]);

    //     $startDate = Carbon::parse($validated['start_date_sales'])->toDateString();
    //     $endDate = Carbon::parse($validated['end_date_sales'])->toDateString();
    //     $intervalDays = Carbon::parse($startDate)->diffInDays(Carbon::parse($endDate));

    //     // Simpan global di session
    //     session([
    //         'global_start_date' => $startDate,
    //         'global_end_date' => $endDate,
    //     ]);

    //     $charts = [
    //         'sales_overview',
    //         'category_chart',
    //         'shift_chart',
    //         'itr_chart',
    //         'price_chart',
    //     ];

    //     foreach ($charts as $chart) {
    //         DB::table('Luv2_chart')
    //             ->updateOrInsert(
    //                 ['id_user' => $userId, 'chart_name' => $chart],
    //                 [
    //                     'start_date' => $startDate,
    //                     'end_date' => $endDate,
    //                     'interval_days' => $intervalDays
    //                 ]
    //             );
    //     }

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Date range updated for all charts.'
    //     ]);
    // }

    public function getCategoryData()
    {
        $userId = Auth::id();
        $companyCode = auth()->user()->company_code;

        // Ambil tanggal global dari session
        $startDate = session('global_start_date', now()->subDays(7)->format('Y-m-d'));
        $endDate = session('global_end_date', now()->format('Y-m-d'));

        // Query data kategori
        $sql = "
        SELECT 
            c.category_code, 
            DATE(a.created_at) AS date, 
            SUM(b.price) AS total_sales 
        FROM \"Luv2_invoice\" a
        INNER JOIN \"Luv2_invoice_detail\" b 
            ON a.company_code = b.company_code
        INNER JOIN \"Luv2_item\" c 
            ON a.company_code = c.company_code
        WHERE a.company_code = :company_code
            AND a.created_at BETWEEN :start_date AND :end_date
        GROUP BY c.category_code, DATE(a.created_at)
        ORDER BY DATE(a.created_at)
        ";

        $results = DB::select($sql, [
            'start_date' => $startDate,
            'end_date' => $endDate,
            'company_code' => $companyCode
        ]);

        // Jika tidak ada data
        if (empty($results)) {
            return response()->json([
                'success' => false,
                'message' => 'No data available for the selected date range.',
                'categorySalesData' => []
            ]);
        }

        // Siapkan data kategori
        $dates = [];
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);

        while ($start->lte($end)) {
            $dates[] = $start->format('Y-m-d');
            $start->addDay();
        }

        $categorySalesData = [];
        foreach ($results as $record) {
            $categorySalesData[$record->category_code][$record->date] = $record->total_sales;
        }

        foreach ($categorySalesData as $category => $sales) {
            foreach ($dates as $date) {
                if (!isset($sales[$date])) {
                    $categorySalesData[$category][$date] = 0;
                }
            }
            ksort($categorySalesData[$category]);
        }

        return response()->json([
            'success' => true,
            'start_date_category' => $startDate,
            'end_date_category' => $endDate,
            'categorySalesData' => $categorySalesData
        ]);
    }

    // public function updateCategoryRange(Request $request)
    // {
    //     $userId = Auth::id();
    //     $validated = $request->validate([
    //         'start_date_category' => 'required|date',
    //         'end_date_category' => 'required|date|after_or_equal:start_date_category',
    //     ]);

    //     // Hitung interval_days
    //     $startDate = Carbon::parse($validated['start_date_category']);
    //     $endDate = Carbon::parse($validated['end_date_category']);
    //     $intervalDays = $endDate->diffInDays($startDate); // Hitung selisih hari

    //     session([
    //         'start_date_category' => $startDate,
    //         'end_date_category' => $endDate,
    //     ]);

    //     DB::table('Luv2_chart')
    //         ->where('id_user', $userId)
    //         ->where('chart_name', 'category_chart')
    //         ->update([
    //             'start_date' => $validated['start_date_category'],
    //             'end_date' => $validated['end_date_category'],
    //             'interval_days' => $intervalDays,
    //         ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Chart updated successfully'
    //     ]);
    // }

    public function getShiftData()
    {
        $userId = Auth::id();

        // Ambil tanggal global dari session
        $startDate = session('global_start_date', now()->subDays(7)->format('Y-m-d'));
        $endDate = session('global_end_date', now()->format('Y-m-d'));

        $sql = "
        SELECT 
            DATE(created_at) AS transaction_date,
            'Pagi' AS shift,
            SUM(total_price) AS total_value
        FROM 
            \"Luv2_invoice_detail\"
        WHERE 
            EXTRACT(HOUR FROM created_at) BETWEEN 6 AND 17
            AND created_at BETWEEN :start_date AND :end_date
        GROUP BY 
            DATE(created_at)
    
        UNION
    
        SELECT 
            DATE(created_at) AS transaction_date,
            'Malam' AS shift,
            SUM(total_price) AS total_value
        FROM 
            \"Luv2_invoice_detail\"
        WHERE 
            EXTRACT(HOUR FROM created_at) NOT BETWEEN 6 AND 17
            AND created_at BETWEEN :start_date AND :end_date
        GROUP BY 
            DATE(created_at)
        ";

        // Eksekusi query untuk shift pagi dan malam
        $results = DB::select($sql, [
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);

        if (empty($results)) {
            return response()->json([
                'success' => false,
                'message' => 'No data available for the selected date range.',
                'shiftSalesData' => []
            ]);
        }

        // Menyiapkan tanggal interval
        $dates = [];
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);

        while ($start->lte($end)) {
            $dates[] = $start->format('Y-m-d');
            $start->addDay();
        }

        // Proses hasil query untuk data shift
        $shiftSalesData = [];
        foreach ($results as $record) {
            $shiftSalesData[$record->transaction_date][$record->shift] = $record->total_value;
        }

        // Pastikan setiap tanggal memiliki data shift pagi dan malam
        foreach ($dates as $date) {
            if (!isset($shiftSalesData[$date])) {
                $shiftSalesData[$date] = [
                    'Pagi' => 0,
                    'Malam' => 0
                ];
            } else {
                $shiftSalesData[$date]['Pagi'] = $shiftSalesData[$date]['Pagi'] ?? 0;
                $shiftSalesData[$date]['Malam'] = $shiftSalesData[$date]['Malam'] ?? 0;
            }
        }
        ksort($shiftSalesData);

        return response()->json([
            'success' => true,
            'start_date_shift' => $startDate,
            'end_date_shift' => $endDate,
            'shiftSalesData' => $shiftSalesData
        ]);
    }

    // public function updateShiftRange(Request $request)
    // {
    //     $userId = Auth::id();
    //     $validated = $request->validate([
    //         'start_date_shift' => 'required|date',
    //         'end_date_shift' => 'required|date|after_or_equal:start_date_shift',
    //     ]);

    //     // Hitung interval_days
    //     $startDate = Carbon::parse($validated['start_date_shift']);
    //     $endDate = Carbon::parse($validated['end_date_shift']);
    //     $intervalDays = $endDate->diffInDays($startDate); // Hitung selisih hari

    //     session([
    //         'start_date_shift' => $startDate,
    //         'end_date_shift' => $endDate,
    //     ]);

    //     DB::table('Luv2_chart')
    //         ->where('id_user', $userId)
    //         ->where('chart_name', 'shift_chart')
    //         ->update([
    //             'start_date' => $validated['start_date_shift'],
    //             'end_date' => $validated['end_date_shift'],
    //             'interval_days' => $intervalDays,
    //         ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Shift Chart updated successfully'
    //     ]);
    // }

    public function getItrData()
    {
        $userId = Auth::id();

        // Ambil tanggal global dari session
        $startDate = session('global_start_date', now()->subDays(7)->format('Y-m-d'));
        $endDate = session('global_end_date', now()->format('Y-m-d'));

        $sql = "
        SELECT 'PENJUALAN' AS TYPE, a.date, a.whs_code, w.name, SUM(total_qty) AS total_qty
        FROM \"Luv2_invoice\" a
        INNER JOIN \"Luv2_invoice_detail\" b 
            ON a.company_code = b.company_code AND a.no = b.no
        LEFT JOIN \"Luv2_warehouse\" w 
            ON a.whs_code = w.code
        WHERE a.date BETWEEN :start_date AND :end_date
        GROUP BY a.date, a.whs_code, w.name

        UNION

        SELECT 'ITR' AS TYPE, date, whs_code_to AS whs_code, w.name, SUM(qty) AS total_qty
        FROM \"Luv2_transfer\" t
        LEFT JOIN \"Luv2_warehouse\" w 
            ON t.whs_code_to = w.code
        WHERE date BETWEEN :start_date AND :end_date
        GROUP BY date, whs_code_to, w.name

        ";

        // Eksekusi query untuk shift pagi dan malam
        $results = DB::select($sql, [
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);

        // Menyiapkan tanggal interval
        $dates = [];
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);

        while ($start->lte($end)) {
            $dates[] = $start->format('Y-m-d');
            $start->addDay();
        }

        // Format data untuk digunakan di frontend
        // $itrData = collect($results)->groupBy('whs_code')->map(function ($group) {
        //     return [
        //         'PENJUALAN' => $group->where('type', 'PENJUALAN')->map(function ($item) {
        //             return [
        //                 'date' => $item->date,
        //                 'total_qty' => $item->total_qty,
        //             ];
        //         })->values()->toArray(), // Konversi ke array
        //         'ITR' => $group->where('type', 'ITR')->map(function ($item) {
        //             return [
        //                 'date' => $item->date,
        //                 'total_qty' => $item->total_qty,
        //             ];
        //         })->values()->toArray(), // Konversi ke array
        //     ];
        // })->toArray();

        // Format data untuk digunakan di frontend
        $itrData = collect($results)->groupBy('whs_code')->map(function ($group) use ($dates) {
            // Inisialisasi data untuk semua tanggal
            $penjualanData = [];
            $itrData = [];
            $whs_name = $group->first()->name;

            foreach ($dates as $date) {
                // Cari data PENJUALAN untuk tanggal ini
                $penjualan = $group->where('type', 'PENJUALAN')->where('date', $date)->first();
                $penjualanData[] = [
                    'date' => $date,
                    'total_qty' => $penjualan ? $penjualan->total_qty : 0,
                ];

                // Cari data ITR untuk tanggal ini
                $itr = $group->where('type', 'ITR')->where('date', $date)->first();
                $itrData[] = [
                    'date' => $date,
                    'total_qty' => $itr ? $itr->total_qty : 0,
                ];
            }

            return [
                'whs_name' => $whs_name,
                'PENJUALAN' => $penjualanData,
                'ITR' => $itrData,
            ];
        })->sortBy('whs_name')->toArray();

        return response()->json([
            'success' => true,
            'start_date_itr' => $startDate,
            'end_date_itr' => $endDate,
            'itrData' => $itrData
        ]);
    }

    // public function updateItrRange(Request $request)
    // {
    //     $userId = Auth::id();
    //     $validated = $request->validate([
    //         'start_date_itr' => 'required|date',
    //         'end_date_itr' => 'required|date|after_or_equal:start_date_itr',
    //     ]);

    //     // Hitung interval_days
    //     $startDate = Carbon::parse($validated['start_date_itr']);
    //     $endDate = Carbon::parse($validated['end_date_itr']);
    //     $intervalDays = $endDate->diffInDays($startDate); // Hitung selisih hari

    //     session([
    //         'start_date_itr' => $startDate,
    //         'end_date_itr' => $endDate,
    //     ]);

    //     DB::table('Luv2_chart')
    //         ->where('id_user', $userId)
    //         ->where('chart_name', 'itr_chart')
    //         ->update([
    //             'start_date' => $validated['start_date_itr'],
    //             'end_date' => $validated['end_date_itr'],
    //             'interval_days' => $intervalDays,
    //         ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'ITR Chart updated successfully'
    //     ]);
    // }

    public function getPriceData()
    {
        $userId = Auth::id();

        // Ambil tanggal global dari session
        $startDate = session('global_start_date', now()->subDays(7)->format('Y-m-d'));
        $endDate = session('global_end_date', now()->format('Y-m-d'));

        $sql = "
        SELECT
            a.date,
            a.whs_code,
            w.name AS whs_name,
            SUM(CASE WHEN a.disc_total = 0 THEN a.must_paid ELSE 0 END) AS harga_normal,
            SUM(CASE WHEN a.disc_total > 0 THEN a.must_paid ELSE 0 END) AS harga_discount
        FROM \"Luv2_invoice\" a
        INNER JOIN \"Luv2_invoice_detail\" b
            ON a.company_code = b.company_code AND a.no = b.no
        LEFT JOIN \"Luv2_warehouse\" w
            ON a.whs_code = w.code
        WHERE
            a.must_paid > 0
            AND a.date BETWEEN :start_date AND :end_date
        GROUP BY a.date, a.whs_code, w.name
        ORDER BY a.date ASC
        ";

        // Eksekusi query
        $results = DB::select($sql, [
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);

        // Inisialisasi struktur data untuk memastikan semua tanggal dan warehouse ada
        $priceData = [];

        // First, populate all possible warehouse codes and their names
        foreach ($results as $record) {
            $whs_code = $record->whs_code ?? "Unknown";
            $whs_name = $record->whs_name ?? $whs_code;

            if (!isset($priceData[$whs_code])) {
                $priceData[$whs_code] = [
                    'name' => $whs_name,
                    // 'data' will hold the date-keyed price data directly, not nested under 'dates'
                ];
            }
        }

        // Prepare all dates in the range for filling missing data points
        $allDatesInRange = [];
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);
        while ($start->lte($end)) {
            $allDatesInRange[] = $start->format('Y-m-d');
            $start->addDay();
        }

        // Initialize all dates for each known warehouse with default zero values
        foreach ($priceData as $whs_code => &$whsEntry) {
            foreach ($allDatesInRange as $date) {
                $whsEntry[$date] = [ // Directly put date as key under warehouse
                    'harga_normal' => 0,
                    'harga_discount' => 0
                ];
            }
        }
        unset($whsEntry); // Unset the reference

        // Populate with actual data from query results
        foreach ($results as $record) {
            $whs_code = $record->whs_code;
            $date = $record->date;

            // Check if the warehouse and date exist, then update
            if (isset($priceData[$whs_code]) && isset($priceData[$whs_code][$date])) {
                $priceData[$whs_code][$date]['harga_normal'] = (float) $record->harga_normal;
                $priceData[$whs_code][$date]['harga_discount'] = (float) $record->harga_discount;
            }
        }

        // The dropdown data uses the 'name' property directly from $priceData
        // The chart data expects date keys directly under the warehouse code
        // We'll prepare a separate structure for the dropdown options if needed,
        // but for the chart, the above $priceData will work directly.

        // We need to reformat $priceData for the dropdown if needed, but it's already
        // structured correctly as per your blade @foreach loop:
        // @foreach (collect($priceData)->sortBy('name') as $whs_code => $data)
        //     <option value="{{ $whs_code }}">{{ $data->name ?? $whs_code }}</option>
        // @endforeach
        // This part already works as long as $priceData[$whs_code]['name'] exists.

        // For the chart, the frontend expects $data->priceData[whs_code][date]...
        // So, we need to ensure $priceData[$whs_code] directly contains date objects.
        // The structure built above ($priceData[$whs_code][$date]) fits this.

        // Final check: the original blade code used `collect($priceData)->sortBy('name') as $whs_code => $data)`
        // This means the `priceData` sent to the frontend view must have the `name` property at the first level
        // under each warehouse code, alongside the date data.
        // The current backend structure does this correctly by adding `name` directly under `priceData[$whs_code]`.

        return response()->json([
            'success' => true,
            // Ensure start_date_price and end_date_price are passed, likely for initial form population
            'start_date_price' => $startDate,
            'end_date_price' => $endDate,
            'priceData' => $priceData
        ]);
    }

    // public function updatePriceRange(Request $request)
    // {
    //     $userId = Auth::id();
    //     $validated = $request->validate([
    //         'start_date_price' => 'required|date',
    //         'end_date_price' => 'required|date|after_or_equal:start_date_price',
    //     ]);

    //     // Hitung interval_days
    //     $startDate = Carbon::parse($validated['start_date_price']);
    //     $endDate = Carbon::parse($validated['end_date_price']);
    //     $intervalDays = $endDate->diffInDays($startDate); // Hitung selisih hari

    //     session([
    //         'start_date_price' => $startDate,
    //         'end_date_price' => $endDate,
    //     ]);

    //     DB::table('Luv2_chart')
    //         ->where('id_user', $userId)
    //         ->where('chart_name', 'price_chart')
    //         ->update([
    //             'start_date' => $validated['start_date_price'],
    //             'end_date' => $validated['end_date_price'],
    //             'interval_days' => $intervalDays,
    //         ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Price Chart updated successfully'
    //     ]);
    // }

    private function formatSalesSeries($salesData) {
        $series = [];
        foreach ($salesData as $name => $data) {
            $series[] = [
                'name' => $name,
                'data' => array_map(function($date, $value) {
                    return [
                        'x' => strtotime($date) * 1000,
                        'y' => $value
                    ];
                }, array_keys((array)$data), array_values((array)$data))
            ];
        }
        return $series;
    }
    private function formatShiftSeries($shiftData) {
        $pagi = [];
        $malam = [];
    
        foreach ($shiftData as $date => $data) {
            // Konversi ke array jika perlu
            if ($data instanceof \stdClass) {
                $data = (array) $data;
            }
    
            $pagi[] = [
                'x' => strtotime($date) * 1000,
                'y' => $data['Pagi'] ?? 0
            ];
            $malam[] = [
                'x' => strtotime($date) * 1000,
                'y' => $data['Malam'] ?? 0
            ];
        }
    
        return [
            [
                'name' => 'Pagi',
                'data' => $pagi
            ],
            [
                'name' => 'Malam',
                'data' => $malam
            ]
        ];
    }    

    private function formatCategorySeries($categoryData) {
        $series = [];
        foreach ($categoryData as $category => $data) {
            $series[] = [
                'name' => $category,
                'data' => array_map(function($date, $value) {
                    return [
                        'x' => strtotime($date) * 1000,
                        'y' => $value
                    ];
                }, array_keys((array)$data), array_values((array)$data))
            ];
        }
        return $series;
    }

    private function formatItrSeries($itrData) {
        $series = [];
        foreach ($itrData as $whs_name => $data) {
            $series[] = [
                'name' => $whs_name,
                'data' => is_array($data) ? $data : (array)$data // ✅ make sure it's array
            ];
        }
        return $series;
    }    

    private function formatPriceSeries($priceData) {
        $series = [];
        foreach ($priceData as $whs_code => $data) {
            $series[] = [
                'name' => $whs_code,
                'data' => is_array($data) ? $data : (array)$data // ✅ prevent error
            ];
        }
        return $series;
    }    

    public function updateGlobalDashboardFilter(Request $request)
    {
        try {
            $userId = Auth::id();

            // Validasi input termasuk warehouse_code
            $validated = $request->validate([
                'start_date' => 'required|date',
                'end_date' => 'required|date|after_or_equal:start_date',
                'warehouse_code' => 'nullable|string'
            ]);

            $startDate = Carbon::parse($validated['start_date'])->toDateString();
            $endDate = Carbon::parse($validated['end_date'])->toDateString();
            $warehouseCode = $validated['warehouse_code'] ?? null;

            // Simpan ke session global
            session([
                'global_start_date' => $startDate,
                'global_end_date' => $endDate,
                'global_warehouse_code' => $warehouseCode
            ]);

            $intervalDays = Carbon::parse($startDate)->diffInDays(Carbon::parse($endDate));

            $charts = [
                'sales_overview',
                'category_chart',
                'shift_chart',
                'itr_chart',
                'price_chart',
            ];

            foreach ($charts as $chart) {
                DB::table('Luv2_chart')->updateOrInsert(
                    ['id_user' => $userId, 'chart_name' => $chart],
                    [
                        'start_date' => $startDate,
                        'end_date' => $endDate,
                        'interval_days' => $intervalDays,
                    ]
                );
            }

            // Ambil data chart
            $salesData = $this->getSalesData()->getData();
            $categoryData = $this->getCategoryData()->getData();
            $shiftData = $this->getShiftData()->getData();
            $itrData = $this->getItrData()->getData();
            $priceData = $this->getPriceData()->getData();

            // Format response
            return response()->json([
                'success' => true,
                'salesData' => isset($salesData->salesData) ? [
                    'series' => $this->formatSalesSeries($salesData->salesData)
                ] : ['series' => []],

                'shiftSalesData' => isset($shiftData->shiftSalesData) ? [
                    'series' => $this->formatShiftSeries($shiftData->shiftSalesData)
                ] : ['series' => []],

                'categorySalesData' => isset($categoryData->categorySalesData) ? [
                    'series' => $this->formatCategorySeries($categoryData->categorySalesData)
                ] : ['series' => []],

                'itrData' => isset($itrData->itrData) ? $itrData->itrData : [],

                'priceData' => isset($priceData->priceData) ? $priceData->priceData : [],
                'totalSales' => $salesData->totalSales ?? 0,
                'atv' => $salesData->atv ?? 0,
                'upt' => $salesData->upt ?? 0,
            ]);

        } catch (\Throwable $e) {
            // Log detail error ke laravel.log
            \Log::error('updateGlobalDashboardFilter error', [
                'message' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan di server. Silakan cek log.'
            ], 500);
        }
    }
    public function home()
    {
        if (Auth::check()) {
            $userId = Auth::id();
            $tutorials = DB::table('Luv2_tutorial')
                ->where('id_user', $userId)
                ->where('menu', 'Home')
                ->first();
            if ($tutorials && $tutorials->active === 'Y') {
                $istutorials = true;
            } else {
                $istutorials = false;
            }
        } else {
            $istutorials = false;
        }

        return view('welcome.welcome', ['istutorials' => $istutorials]);
    }

    public function index()
    {
        if (auth()->user()->role === 'Gudang') {
            return redirect()->route('welcome.awal');
        } else {
            $setting = Warehouse::where('code', auth()->user()->whs_code)->first();
            $warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

            if (Auth::user()->whs_code == '-') {
                return redirect()->route('select-warehouse');
            } else {
                return view('welcome.home', compact('setting', 'warehouses'));
            }
        }
    }

    public function display()
    {
        $countinv = Invoice::where('sandbox', 'Y')->count();
        $setting = Settings::first();
        return view('welcome.display', compact('countinv', 'setting'));
    }

    public function returnsale()
    {
        return view('welcome.returnsale');
    }

    public function item()
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();

        if ($cekwhs) {
            return view('item.item');
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function inputitem()
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();

        if ($cekwhs) {
            if (!session()->has('insertType')) {
                session(['insertType' => 'Item']);
            }
            return view('item.input');
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function pricelistitem($id)
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();

        if ($cekwhs) {
            return view('item.pricelistinput', compact('id'));
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function unititem($id)
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();

        if ($cekwhs) {
            return view('item.unititeminput', compact('id'));
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function opname()
    {
        $userId = Auth::id();
        $tutorials = DB::table('Luv2_tutorial')
            ->where('id_user', $userId)
            ->where('menu', 'Opname')
            ->first();
        if ($tutorials && $tutorials->active === 'Y') {
            $istutorials = true;
        } else {
            $istutorials = false;
        }
        return view('opname.opname', compact('istutorials'));
    }

    public function opnameinput()
    {
        return view('opname.inputopname');
    }

    public function opnamedetail($no)
    {
        return view('opname.detailopname', compact('no'));
    }
    public function resultopname()
    {
        return view('opname.result');
    }

    public function itin()
    {
        return view('it.itin');
    }

    public function itreq()
    {
        return view('it.itreq');
    }


    public function itindetail($it)
    {
        return view('it.itin-detail', compact('it'));
    }

    public function itreqdetail($it)
    {
        return view('it.itreq-detail', compact('it'));
    }

    public function itreqout()
    {
        return view('it.it-req-out.itreqout');
    }
    public function itreqoutcreate()
    {
        return view('it.it-req-out.itreqoutcreate');
    }

    public function itreqoutdetail($it)
    {
        return view('it.it-req-out.itreqout-detail', compact('it'));
    }

    public function itreqoutprint($it)
    {
        $details = DB::table('Luv2_it_req_out_detail')
            ->Join('Luv2_it_req_out', 'Luv2_it_req_out_detail.no', '=', 'Luv2_it_req_out.no')
            ->Join('Luv2_item', 'Luv2_it_req_out_detail.item_code', '=', 'Luv2_item.code')
            ->select('Luv2_item.code', 'Luv2_item.barcode', 'Luv2_item.name', DB::raw('SUM("Luv2_it_req_out_detail"."qty") as "qty"'), DB::raw('SUM("Luv2_it_req_out_detail"."qty_send") as "qty_send"'))
            ->where('Luv2_it_req_out.no', $it)
            ->groupBy('Luv2_item.code', 'Luv2_item.barcode', 'Luv2_item.name')
            ->get();

        $no = ItReqOut::with(['whsTo', 'details'])
            ->where('canceled', 'N')
            ->withSum('details', 'qty_send')
            ->orderBy('Luv2_it_req_out.date', 'desc')
            ->orderBy('Luv2_it_req_out.no', 'desc')
            ->first();
        $sumQty = 0;
        $sumQtySend = 0;

        foreach ($details as $detail) {
            $sumQty += $detail->qty;
            $sumQtySend += $detail->qty_send;
        }

        return view('it.it-req-out.printitreqout', compact('details', 'no', 'sumQty', 'sumQtySend'));
    }


    public function itout()
    {
        $itout = ItOut::get();
        return view('it.itout', compact('itout'));
    }
    public function itoutdetail($it)
    {
        return view('it.itout-detail', compact('it'));
    }
    public function salesemp()
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();
        if ($cekwhs) {
            $tutorials = DB::table('Luv2_tutorial')
                ->where('id_user', auth()->user()->id)
                ->where('menu', 'SalesEmployee')
                ->first();
            if ($tutorials && $tutorials->active === 'Y') {
                $istutorials = true;
            } else {
                $istutorials = false;
            }
            return view('emp-sales.emp', compact('istutorials'));
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }
    public function reportinvoice()
    {
        $user = Auth::user();
        $invoices = Invoice::whereNull('no_inv')->get();
        $authorization = Auths::where('menu', 'Transaction')
            ->where('modul', 'Sales')->first(); // Fetch the specific authorization

        $hasPermission = $user->authorizations
            ->filter(function ($authorization) {
                return strpos($authorization->auth, 'R') !== false;
            })
            ->contains('id_auth', $authorization->id);

        if ($hasPermission) {
            return view('report.reportinvoice', compact('invoices'));
        } else {
            abort(403);
        }
    }

    public function reportinvoicereturnsale()
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();
        if ($cekwhs) {
            $user = Auth::user();
            $invoices = Invoice::whereNull('no_inv')->get();
            return view('report.reportinvoicereturnsale', compact('invoices'));
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function reportinvoicedetail()
    {
        return view('report.reportinvoicedetail');
    }

    public function reportreturnsale()
    {
        $user = Auth::user();

        $authorization = Auths::where('menu', 'Transaction')
            ->where('modul', 'Sales Return')->first(); // Fetch the specific authorization

        $invoices = Invoice::whereNotNull('no_inv')->get();
        $hasPermission = $user->authorizations
            ->filter(function ($authorization) {
                return strpos($authorization->auth, 'R') !== false;
            })
            ->contains('id_auth', $authorization->id);

        if ($hasPermission) {
            return view('report.reportreturnsale', compact('invoices'));
        } else {
            abort(403);
        }
    }
    public function reportvoucher()
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();
        if ($cekwhs) {
            return view('report.reportvoucher');
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function reporttransactionmember()
    {
        $members = Member::withCount([
            'invoices' => function ($query) {
                $query->whereNotNull('code'); // Assuming 'code' is the column representing the member's code in the invoices table
            }
        ])
            ->whereHas('invoices', function ($query) {
                $query->whereNotNull('code'); // Also add a whereHas to filter out members without any invoices
            })
            ->orderBy('invoices_count', 'desc')
            ->get();
        return view('report.reporttransactionmember', compact('members'));
    }

    public function settings()
    {
        if (Auth::user()->whs_code == '-') {
            return redirect()->route('select-warehouse');
        } else {
            return view('auth.settings');
        }
    }
    public function changepass()
    {
        return view('auth.changepassword');
    }

    public function forgotpass()
    {
        return view('auth.forgotpassword');
    }

    /**
     * Show the form for creating a new resource.
     */



    public function welcomeregister()
    {
        return view('welcome.welcomeregister');
    }

    public function synchronization()
    {
        return view('synchronization.index');
    }

    public function print($no)
    {
        $invoice = DB::table('Luv2_invoice')
            ->where('no', $no)
            ->leftJoin('Luv2_salesemp', 'Luv2_invoice.sales_code', '=', 'Luv2_salesemp.code')
            ->leftJoin('Luv2_user', 'Luv2_invoice.user_code', '=', 'Luv2_user.code')
            ->select(
                'Luv2_invoice.*',
                'Luv2_salesemp.name as sales_name',
                'Luv2_user.username as users_name'
            )->first();

        $detail = DB::table('Luv2_invoice_detail')
            ->leftJoin('Luv2_unit', function ($join) {
                $join->on('Luv2_invoice_detail.unit', '=', 'Luv2_unit.id')
                    ->where('Luv2_unit.company_code', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_item_unit', function ($join) {
                $join->on('Luv2_invoice_detail.code', '=', 'Luv2_item_unit.item_code')
                    ->on('Luv2_item_unit.id_unit', '=', 'Luv2_invoice_detail.unit') // Ensure matching unit
                    ->where('Luv2_item_unit.whs_code', auth()->user()->whs_code)
                    ->where('Luv2_item_unit.company_code', auth()->user()->company_code);
            })
            ->where('Luv2_invoice_detail.no', $no)
            ->where('Luv2_invoice_detail.company_code', auth()->user()->company_code)
            ->select('Luv2_invoice_detail.*', 'Luv2_unit.name as unit_name', 'Luv2_item_unit.pricesell')
            ->orderBy('Luv2_invoice_detail.linenum')
            ->get();

        $warehouse = DB::table('Luv2_warehouse')->where('company_code', auth()->user()->company_code)->where('code', auth()->user()->whs_code)->first();
        $footer = str_replace('\n', '<br>', $warehouse->footer);
        $promovoucher = DB::table('Luv2_promo')->where('company_code', auth()->user()->company_code)->where('invoice', $no)->first();
        if($promovoucher){
            $voucher = DB::table('Luv2_promo_voucher')->where('company_code', auth()->user()->company_code)->where('no', $promovoucher->no)->first();
        }else{
            $voucher = null;
        }
        return view('report.print', compact('invoice', 'detail', 'warehouse', 'footer', 'voucher'));
    }

    public function purchasereturnprint()
    {
        $dummyItems = [
            [
                'item_code' => '89548267429766',
                'barcode' => '89548267429766',
                'description' => 'A31305 146 ALL ALANA LOLA SLIM GOLD 555',
                'size' => 'ALL SIZE',
                'price' => 2099000,
                'total' => 2099000,
            ],
            [
                'item_code' => '89548266602.08',
                'barcode' => '89548266602.08',
                'description' => 'CTE1H9 1000 L TAPE PATCH TEE',
                'size' => 'L',
                'price' => 299000,
                'total' => 299000,
            ],
            [
                'item_code' => '89548266668.66',
                'barcode' => '89548266668.66',
                'description' => 'CTE1E2 1000 M WARPED TEE',
                'size' => 'M',
                'price' => 299000,
                'total' => 299000,
            ],
            // [
            //     'item_code' => '89548265979.69',
            //     'barcode' => '89548265979.69',
            //     'description' => 'CTE1B2 94.15 M EMBROID TEE',
            //     'size' => 'M',
            //     'price' => 229000,
            //     'total' => 229000,
            // ],
            // [
            //     'item_code' => '89548268031.79',
            //     'barcode' => '89548268031.79',
            //     'description' => 'VST715 107 ALL STRINGER',
            //     'size' => 'ALL SIZE',
            //     'price' => 1049000,
            //     'total' => 1049000,
            // ],
        ];
        $dummyItems2 = [
            // [
            //     'item_code' => '89548267429766',
            //     'barcode' => '89548267429766',
            //     'description' => 'A31305 146 ALL ALANA LOLA SLIM GOLD 555',
            //     'size' => 'ALL SIZE',
            //     'price' => 2099000,
            //     'total' => 2099000,
            // ],
            // [
            //     'item_code' => '89548266602.08',
            //     'barcode' => '89548266602.08',
            //     'description' => 'CTE1H9 1000 L TAPE PATCH TEE',
            //     'size' => 'L',
            //     'price' => 299000,
            //     'total' => 299000,
            // ],
            // [
            //     'item_code' => '89548267429766',
            //     'barcode' => '89548267429766',
            //     'description' => 'A31305 146 ALL ALANA LOLA SLIM GOLD 555',
            //     'size' => 'ALL SIZE',
            //     'price' => 2099000,
            //     'total' => 2099000,
            // ],
            // [
            //     'item_code' => '89548266602.08',
            //     'barcode' => '89548266602.08',
            //     'description' => 'CTE1H9 1000 L TAPE PATCH TEE',
            //     'size' => 'L',
            //     'price' => 299000,
            //     'total' => 299000,
            // ],
            // [
            //     'item_code' => '89548267429766',
            //     'barcode' => '89548267429766',
            //     'description' => 'A31305 146 ALL ALANA LOLA SLIM GOLD 555',
            //     'size' => 'ALL SIZE',
            //     'price' => 2099000,
            //     'total' => 2099000,
            // ],
            // [
            //     'item_code' => '89548266602.08',
            //     'barcode' => '89548266602.08',
            //     'description' => 'CTE1H9 1000 L TAPE PATCH TEE',
            //     'size' => 'L',
            //     'price' => 299000,
            //     'total' => 299000,
            // ],

        ];

        $dummyItems = array_merge($dummyItems, $dummyItems, $dummyItems, $dummyItems, $dummyItems, $dummyItems, $dummyItems, $dummyItems2);

        // Chunk the items into groups of 25
        $chunks = array_chunk($dummyItems, 34);

        $addedEmptyPage = false;

        // Check the last chunk's count
        if (count($chunks) > 0) {
            $lastChunkIndex = count($chunks) - 1;
            $lastChunkCount = count($chunks[$lastChunkIndex]);

            if ($lastChunkCount >= 28 && $lastChunkCount <= 34) {
                $chunks[] = []; // Add an empty page
                $addedEmptyPage = true; // Set flag
            }
        }

        $totalPages = count($chunks);
        $dataPages = [];

        foreach ($chunks as $pageIndex => $items) {
            $isEmptyPage = $addedEmptyPage && empty($items); // Flag for empty page

            $dataPages[] = [
                'documentNumber' => '280349',
                'documentDate' => '04.01.21',
                'items' => $items,
                'totalPages' => $totalPages,
                'currentPage' => $pageIndex + 1,
                'isLastPage' => ($pageIndex + 1) === $totalPages,
                'isEmptyPage' => $isEmptyPage, // Add flag
                'totalItems' => count($dummyItems),
                'grandTotal' => array_sum(array_column($dummyItems, 'total')),
            ];
        }

        $pdf = Pdf::loadView('export.printpurchasereturn', compact('dataPages'))
            ->setPaper('A4', 'portrait');

        return $pdf->stream('purchase-return.pdf');
    }



    public function updateStatusDriver(Request $request)
    {
        $user = Auth::user();
        if (!$user) {
            return response()->json(['error' => 'User not authenticated'], 401);
        } else {
            $menuValue = $request->input('menu');
            DB::table('Luv2_tutorial')
                ->where('menu', $menuValue)
                ->update([
                    'active' => 'N',
                ]);
        }
    }

    public function changeStatusDriver(Request $request)
    {
        $user = Auth::user();
        if (!$user) {
            return response()->json(['error' => 'User not authenticated'], 401);
        } else {
            $menuValue = $request->input('menu');
            $currentStatus = DB::table('Luv2_tutorial')
                ->where('menu', $menuValue)
                ->value('active');

            $newStatus = $currentStatus === 'Y' ? 'N' : 'Y';

            DB::table('Luv2_tutorial')
                ->where('menu', $menuValue)
                ->update([
                    'active' => $newStatus,
                ]);
        }
    }

    public function usermanagement()
    {
        return view('user-management.index');
    }

    public function chooseWarehouse()
    {
        $previousUrl = session('toUrl', route('welcome.dashboard'));

        $warehouse = Warehouse::where('company_code', auth()->user()->company_code)
            ->where('active', 'Y')
            ->where('code', '!=', auth()->user()->whs_code)
            ->get();
        return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
    }

    public function selectWarehouse(Request $request)
    {
        $user = Auth::user();
        $user->whs_code = $request->whs_code;
        $user->save();
        Session::forget('toUrl');

        return response()->json(['success' => true]);
    }

    public function reportgoodsreceipt()
    {
        $user = Auth::user();
        $purchase = DB::table('Luv2_receipt')
            ->leftJoin('Luv2_supplier', 'Luv2_receipt.supp_code', '=', 'Luv2_supplier.code')
            ->leftJoin('Luv2_warehouse', 'Luv2_receipt.whs_code', '=', 'Luv2_warehouse.code')
            // ->where('Luv2_receipt.user_code', $user->code)
            // ->where('Luv2_receipt.deleted', '<>', 'Y')
            ->select('Luv2_receipt.*', 'Luv2_supplier.*', 'Luv2_warehouse.name as whs_name') // Select all columns from both tables
            ->get();

        $totalSumQty = DB::table('Luv2_receipt')
            ->leftJoin('Luv2_supplier', 'Luv2_receipt.supp_code', '=', 'Luv2_supplier.code')
            ->where('Luv2_receipt.user_code', $user->code)
            ->where('Luv2_receipt.deleted', '<>', 'Y')
            ->sum('Luv2_receipt.qty');

        return view('report.reportgoodsreceipt', compact('purchase', 'totalSumQty'));
    }

    public function reportgoodsreceiptdetail($id)
    {
        return view('report.reportgoodsreceiptdetail', compact('id'));
    }

    public function reportgoodsreturn()
    {
        $user = Auth::user();
        $return = DB::table('Luv2_return')
            ->leftJoin('Luv2_supplier', 'Luv2_return.supp_code', '=', 'Luv2_supplier.code')
            ->leftJoin('Luv2_warehouse', 'Luv2_return.whs_code', '=', 'Luv2_warehouse.code')
            // ->where('Luv2_return.user_code', $user->code)
            // ->where('Luv2_return.deleted', '<>', 'Y')
            ->select('Luv2_return.*', 'Luv2_supplier.*', 'Luv2_warehouse.name as whs_name') // Select all columns from both tables
            ->get();

        $totalSumQty = DB::table('Luv2_return')
            ->leftJoin('Luv2_supplier', 'Luv2_return.supp_code', '=', 'Luv2_supplier.code')
            ->where('Luv2_return.user_code', $user->code)
            ->where('Luv2_return.deleted', '<>', 'Y')
            ->sum('Luv2_return.qty');

        return view('report.reportgoodsreturn', compact('return', 'totalSumQty'));
    }

    public function reportgoodsreturndetail($id)
    {
        return view('report.reportgoodsreturndetail', compact('id'));
    }

    public function reportstock()
    {
        $cekwhs = Warehouse::where('company_code', auth()->user()->company_code)->first();
        if ($cekwhs) {
            $stock = [];
            return view('report.reportstock', compact('stock'));
        } else {
            $previousUrl = session('toUrl', route('welcome.dashboard'));

            $warehouse = Warehouse::where('company_code', auth()->user()->company_code)->get();

            return view('select-warehouse.index', compact('warehouse', 'previousUrl'));
        }
    }

    public function reportstockcard()
    {
        $stockcard = [];

        return view('report.reportstockcard', compact('stockcard'));
    }
    public function reporttransfer()
    {
        $user = Auth::user();
        $transfer = [];

        return view('report.reporttransfer', compact('transfer'));
    }

    public function reporttransferdetail($id)
    {
        return view('report.reporttransferdetail', compact('id'));
    }

    public function reportopname()
    {
        $user = Auth::user();
        $opname = [];


        return view('report.reportopname', compact('opname'));
    }

    public function reportopnamedetail($id)
    {
        return view('report.reportopnamedetail', compact('id'));
    }

    public function reportmargin()
    {
        return view('report.marginreport');
    }

    public function help()
    {
        $tutorial = DB::table('Luv2_tutorial')
            ->where('id_user', auth()->user()->id)
            ->get();

        $menuItems = [
            'Dashboard' => null,
            'Promo' => null,
            'Home' => null,
            'Product' => null,
            'Warehouse' => null,
            'Goods Receipt' => null,
            'Goods Return' => null,
            'Item Transfer' => null,
            'SalesEmployee' => null,
            'Pricelist' => null,
            'Opname' => null,
            'Brand' => null,
            'Cashier' => null,
            'Member' => null,
            'Supplier' => null,
            'Master' => null,
            'Tutorial' => null,
        ];

        foreach ($tutorial as $tut) {
            if (array_key_exists($tut->menu, $menuItems)) {
                $menuItems[$tut->menu] = $tut;
            }
        }
        return view('help.index', compact('menuItems'));
    }

    public function udfCreate()
    {
        return view('udf.create-udf');
    }

    public function waitingData()
    {
        $user = Auth::user();
        if ($user) {
            $warehouse = Warehouse::where('company_code', $user->company_code)->first();
            $user->whs_code = $warehouse->code;
            $user->save();
        }
        return view('waiting');
    }

    public function settingMoh($code = null)
    {
        return view('setting-moh.setting-moh', compact('code'));
    }

    public function indexSettingMoh()
    {
        return view('setting-moh.index');
    }

    public function indexMohUDF($code = null)
    {
        return view('setting-moh.indexmohudf', compact('code'));
    }

    public function createMohUDF($code = null)
    {
        return view('setting-moh.create-moh-udf', compact('code'));
    }

    public function stockAlert()
    {
        return view('stock-alert.index');
    }

    public function lossSales()
    {
        return view('loss-sales.index');
    }

    public function setupPassword($token)
    {
        return view('setup-password', compact('token'));
    }

}
