<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Warehouse;
use App\Providers\RouteServiceProvider;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;


class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    protected function validateLogin(Request $request)
    {
        $request->validate([
            $this->username() => 'required|string',
            'password' => 'required|string',
        ]);
    }

    protected function attemptLogin(Request $request)
    {
        // Check if the user status is 'n'
        $user = $this->guard()->getProvider()->retrieveByCredentials($this->credentials($request));

        if ($user && $user->active === 'N') {
            return false;
        }

        // Proceed with the standard attemptLogin logic
        return $this->guard()->attempt(
            $this->credentials($request),
            $request->boolean('remember')
        );
    }

    protected function credentials(Request $request)
    {
        return $request->only($this->username(), 'password');
    }

    protected function sendLoginResponse(Request $request)
    {
        $user = $this->guard()->user();
        $request->session()->regenerate();

        $this->clearLoginAttempts($request);

        DB::statement("DO $$ 
            DECLARE
                r RECORD;
            BEGIN
                FOR r IN (SELECT tablename FROM pg_tables WHERE tablename LIKE '" . $user->id . "_%') 
                LOOP
                    EXECUTE 'DROP TABLE IF EXISTS \"' || r.tablename || '\" CASCADE';
                END LOOP;
            END $$;");

        $tutorials = DB::table('Luv2_tutorial')
            ->where('id_user', $user->id)
            ->pluck('active');

        if ($tutorials) {
            if ($tutorials->contains('Y')) {
                $user->tutorials = 'Y';
            } else if ($tutorials->every(fn($active) => $active === 'N')) {
                $user->tutorials = 'N';
            }
            $user->save();
        }
        $company = DB::table('Luv2_company')
            ->where('code', $user->company_code)
            ->first();

        if ($company && Carbon::now()->greaterThan($company->expired_at)) {
            return redirect()->route('paymentrenew');
        } else {
            if ($user->role === 'Admin') {
                $warehouse = DB::table('Luv2_warehouse')
                    ->where('company_code', $user->company_code)
                    ->get();

                if ($warehouse->count() === 1) {
                    $user->whs_code = $warehouse->first()->code;
                    $user->save();
                }
                return redirect('/cashier');
            } else {
                return redirect('/');
            }
        }

        if ($response = $this->authenticated($request, $user)) {
            return $response;
        }
    }

    protected function sendFailedLoginResponse(Request $request)
    {
        $credentials = $this->credentials($request);
        $user = $this->guard()->getProvider()->retrieveByCredentials($credentials);

        if (!$user) {
            throw ValidationException::withMessages([
                $this->username() => [trans('Username is incorrect!')],
            ]);
        }
        if ($user->active === 'N') {
            throw ValidationException::withMessages([
                $this->username() => [trans('User is not active, please contact admin!')],
            ]);
        }
        if (!$this->guard()->validate($credentials)) {
            throw ValidationException::withMessages([
                'password' => [trans('Password is incorrect !')],
            ]);
        }

        throw ValidationException::withMessages([
            $this->username() => [trans('auth.failed')],
        ]);
    }

    public function logins(Request $request)
    {
        $this->validateLogin($request);

        $data = $request->all();

        $user = User::where('username', $data['username'])->first();

        if ($user) {

            if (Hash::check($data['password'], $user->password)) {
                $company = DB::table('Luv2_company')->where('code', $user->company_code)->first();
                if ($user->active === 'N') {
                    $message = 'Akun Anda tidak aktif.';
                    Log::channel('login')->info(
                        "\n" .
                            "**LOGIN GAGAL** \n" .
                            "Message: Akun tidak aktif\n" .
                            "Username: {$request->username}\n" .
                            "Password: " . $request->password . "\n" .
                            "Time: " . now()->toDateTimeString() . "\n" .
                            "----------------------------------------------------"
                    );
                    return redirect()->back()->withErrors(['login' => $message]);
                } elseif ($company->verified_at == null) {
                    $message = 'Akun Anda belum aktif, silahkan periksa email anda dan lakukan verifikasi!';
                    Log::channel('login')->info(
                        "\n" .
                            "**LOGIN GAGAL** \n" .
                            "Message: Akun belum melakukan verifikasi\n" .
                            "Username: {$request->username}\n" .
                            "Password: " . $request->password . "\n" .
                            "Time: " . now()->toDateTimeString() . "\n" .
                            "----------------------------------------------------"
                    );
                    return redirect()->back()->withErrors(['login' => $message]);
                }

                // If another session is active, log out that session
                if ($user->session_token) {
                    // Optionally, you can invalidate the session here
                    // and notify the user about the previous session
                }

                // Generate a new unique session token
                $sessionToken = Str::random(60);

                // Store the session token in the database
                $user->session_token = $sessionToken;
                $user->last_login_at = now();
                $user->login_count = $user->login_count + 1;
                $user->save();

                // Store the session token in the current session for verification later
                session(['session_token' => $sessionToken]);

                // Manually log the user in
                Auth::login($user);

                // Handle remember me functionality by setting cookies if needed
                if ($request->has('remember')) {
                    setcookie("name", $data['username'], time() + (7 * 24 * 60 * 60));
                    setcookie("password", $data['password'], time() + (7 * 24 * 60 * 60));
                } else {
                    setcookie("name", "", time() - 3600);
                    setcookie("password", "", time() - 3600);
                }

                // Additional session handling (optional)
                $request->session()->put('auth.password_confirmed_at', time());

                // Regenerate session ID after login
                $request->session()->regenerate();
                Log::channel('login')->info(
                    "\n" .
                        "**LOGIN BERHASIL** \n" .
                        "Message: Berhasil login\n" .
                        "Username: {$request->username}\n" .
                        "Password: " . $request->password . "\n" .
                        "Time: " . now()->toDateTimeString() . "\n" .
                        "----------------------------------------------------"
                );
                // Redirect after successful login
                return $this->sendLoginResponse($request);
            } else {
                // Increment failed login attempts and return failure response
                $this->incrementLoginAttempts($request);
                Log::channel('login')->info(
                    "\n" .
                        "**LOGIN GAGAL** \n" .
                        "Message: Password salah!\n" .
                        "Username: {$request->username}\n" .
                        "Password: " . $request->password . "\n" .
                        "Time: " . now()->toDateTimeString() . "\n" .
                        "----------------------------------------------------"
                );

                return redirect()->back()->withErrors(['login' => 'Invalid username or password.']);
            }
        } else {
            Log::channel('login')->info(
                "\n" .
                    "**LOGIN GAGAL** \n" .
                    "Message: User salah!\n" .
                    "Username: {$request->username}\n" .
                    "Password: " . $request->password . "\n" .
                    "Time: " . now()->toDateTimeString() . "\n" .
                    "----------------------------------------------------"

            );
            return redirect()->back()->withErrors(['login' => 'Invalid username or password.']);
        }
    }


    // public function verifyEmail($encrypted)
    // {
    //     $decryptedId = Crypt::decryptString($encrypted);
    //     $company = DB::table('Luv2_company')->where('code', $decryptedId)->first();
    //     if ($company->verified_at) {
    //         if ($company->verified_at) {
    //             if ($company->verified_at) {
    //                 session()->flash('error', 'This verification link has already been used.');
    //                 return view('verification', compact('company'));
    //             }
    //         }
    //     }
    //     return view('verification', compact('company'));
    // }

    public function verifyEmail($encrypted)
    {
        try {
            // Decrypt the data
            $decrypted = json_decode(Crypt::decryptString($encrypted), true);

            $companyCode = $decrypted['company_code'] ?? null;
            $timestamp = $decrypted['timestamp'] ?? null;

            if (!$companyCode || !$timestamp) {
                abort(400, 'Invalid verification link.');
            }

            $createdAt = Carbon::createFromTimestamp($timestamp);
 
            if (now()->diffInHours($createdAt) >= 24) {
                abort(401);
            }

            $company = DB::table('Luv2_company')->where('code', $companyCode)->first();
            if (!$company) {
                abort(404, 'Company not found.');
            }

            if ($company->verified_at) {
                session()->flash('error', 'This verification link has already been used.');
                return view('verification', compact('company'));
            }

            DB::table('Luv2_company')->where('code', $companyCode)->update([
                'verified_at' => now(),
                'expired_at' => now()->addMonth(),
            ]);

            // Auto-login user and redirect
            $user = \App\Models\User::where('company_code', $companyCode)->first();
            if ($user) {
                $sessionToken = Str::random(60);
                $warehouse = \App\Models\Warehouse::where('company_code', $companyCode)->first();
                $user->whs_code = $warehouse->code;
                $user->session_token = $sessionToken;
                session(['session_token' => $sessionToken]);

                $user->save();
                Auth::login($user);
                return redirect()->route('welcome.dashboard');
            }

            return view('verification', compact('company'));
        } catch (\Throwable $e) {
            abort(401, 'Verification link has expired.');
        }
    }




    public function verifPost($code)
    {
        $company = DB::table('Luv2_company')->where('code', $code)->first();

        if ($company) {
            if (!$company->verified_at) {
                DB::table('Luv2_company')->where('code', $code)->update([
                    'verified_at' => now(),
                    'expired_at' => now()->addMonth(),
                ]);
            }

            $user = \App\Models\User::where('company_code', $code)->first();
            if ($user) {
                $sessionToken = Str::random(60);
                $warehouse = \App\Models\Warehouse::where('company_code', $company->code)->first();
                $user->whs_code = $warehouse->code;
                $user->session_token = $sessionToken;
                session(['session_token' => $sessionToken]);

                $user->save();
                Auth::login($user);
                return redirect()->route('welcome.dashboard');
            }

            return back()->with('success', 'Company verified, but no user found to login.');
        }

        return redirect()->route('login')->withErrors(['error' => 'Company not found.']);
    }
}
