<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\WithCustomStartCell;

class ExportARItemDetail implements FromQuery, WithHeadings, WithMapping, ShouldAutoSize, WithEvents, WithCustomStartCell
{
    use Exportable;

    private $type;
    private $startDate;
    private $endDate;
    private $whs_code;

    public function __construct($data)
    {
        $this->type = $data['type'];
        $this->startDate = $data['startDate'];
        $this->endDate = $data['endDate'];
        $this->whs_code = $data['whs_code'];
    }

    public function query()
    {
        $query = DB::table('Luv2_invoice_detail as d')
            ->select([
                'i.no as invoice_no',
                DB::raw("TO_CHAR(i.created_at, 'YY/MM/DD') as tanggal"),
                DB::raw("TO_CHAR(i.created_at, 'HH24:MI') as jam"),
                'd.barcode as item_barcode',
                'd.name as item_name',
                'i.payment as jenis',
                'u.username as user',
                's.name as staff',
                'd.qty',
                'd.nprice as sub_total',
                'd.disc as discount',
                DB::raw('(d.nprice - d.disc) as grand_total')
            ])
            ->join('Luv2_invoice as i', 'd.no', '=', 'i.no')
            ->leftJoin('Luv2_user as u', 'i.user_code', '=', 'u.code')
            ->leftJoin('Luv2_salesemp as s', 'd.sales_code', '=', 's.code')
            ->where('i.whs_code', $this->whs_code);

        // Date filter
        if ($this->startDate && $this->endDate) {
            if ($this->startDate === $this->endDate) {
                $query->whereDate('i.created_at', '=', $this->startDate);
            } else {
                $endDate = date('Y-m-d 23:59:59', strtotime($this->endDate));
                $query->whereBetween('i.created_at', [$this->startDate, $endDate]);
            }
        }
        // Type filter
        if ($this->type === 'BJ') {
            $query->whereNull('i.no_inv');
        } elseif ($this->type === 'RS') {
            $query->whereNotNull('i.no_inv');
        }
        return $query->orderByDesc('i.created_at');
    }

    public function headings(): array
    {
        return [
            'Invoice',
            'Tanggal',
            'Jam',
            'Item Barcode',
            'Item Name',
            'Jenis',
            'User',
            'Staff',
            'Qty',
            'Sub Total',
            'Discount',
            'Grand Total',
        ];
    }

    public function map($row): array
    {
        return [
            $row->invoice_no,
            $row->tanggal,
            $row->jam,
            $row->item_barcode,
            $row->item_name,
            $row->jenis,
            $row->user,
            $row->staff,
            (int)$row->qty,
            number_format($row->sub_total, 2),
            number_format($row->discount, 2),
            number_format($row->grand_total, 2),
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $lastColumn = $event->sheet->getHighestColumn();
                $lastRow = $event->sheet->getHighestRow();
                $range = 'C3:' . $lastColumn . $lastRow;
                $event->sheet->getStyle($range)->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                            'color' => ['argb' => '#000000'],
                        ],
                    ],
                ]);
            }
        ];
    }

    public function startCell(): string
    {
        return 'C3';
    }
}
