<?php

namespace App\Exports;

use App\Models\Invoice as ModelsInvoice;
use Illuminate\Contracts\View\View;
use Maatwebsite\Excel\Concerns\FromView;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;

class ExportARInvoiceDetail implements FromView, ShouldAutoSize, WithEvents
{
    /**
    * @return \Illuminate\Support\Collection
    */
    public $arinvoices;
    public $type;
    public $startDate;
    public $endDate;
    public $checkboxes = [];

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $lastColumn = $event->sheet->getHighestColumn();
                $lastRow = $event->sheet->getHighestRow();
        
                $rangeb = 'B3:' . $lastColumn . $lastRow;
                $rangec = 'C3:' . $lastColumn . $lastRow;
                $rangee = 'E3:' . $lastColumn . $lastRow;
                $rangeg = 'G3:' . $lastColumn . $lastRow;
                $rangeh = 'H3:' . $lastColumn . $lastRow;
                $rangei = 'I3:' . $lastColumn . $lastRow;

                $event->sheet->getStyle($rangeb)->applyFromArray([
                    'borders' => [
                        'allBorders' => [
                            'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                            'color' => ['argb' => '#000000'],
                        ],
                    ],
                ]);
                $event->sheet->getStyle($rangec)
                    ->getNumberFormat()
                    ->setFormatCode(NumberFormat::FORMAT_TEXT);
                $event->sheet->getStyle($rangee)
                    ->getNumberFormat()
                    ->setFormatCode(NumberFormat::FORMAT_NUMBER_COMMA_SEPARATED1);
                $event->sheet->getStyle($rangeg)
                    ->getNumberFormat()
                    ->setFormatCode(NumberFormat::FORMAT_NUMBER_COMMA_SEPARATED1);
                $event->sheet->getStyle($rangeh)
                    ->getNumberFormat()
                    ->setFormatCode(NumberFormat::FORMAT_NUMBER_COMMA_SEPARATED1);
                $event->sheet->getStyle($rangei)
                    ->getNumberFormat()
                    ->setFormatCode(NumberFormat::FORMAT_NUMBER_COMMA_SEPARATED1);
            }
        ];
    }
    public function __construct($data)
    {
        $this->type = $data['type'];
        $this->startDate = $data['startDate'];
        $this->endDate = $data['endDate'];
        
        // Get invoice IDs first for better performance
        $invoiceQuery = DB::table('Luv2_invoice')
            ->select('no')
            ->where('whs_code', auth()->user()->whs_code);

        // Apply date filter
        if ($this->startDate && $this->endDate) {
            if ($this->startDate === $this->endDate) {
                $invoiceQuery->whereDate('created_at', '=', $this->startDate);
            } else {
                $endDate = date('Y-m-d 23:59:59', strtotime($this->endDate));
                $invoiceQuery->whereBetween('created_at', [$this->startDate, $endDate]);
            }
        }

        // Apply type filter
        if ($this->type === 'BJ') {
            $invoiceQuery->whereNull('no_inv');
        } elseif ($this->type === 'RS') {
            $invoiceQuery->whereNotNull('no_inv');
        }

        $invoiceNos = $invoiceQuery->orderByDesc('created_at')->pluck('no')->toArray();

        // Get all required data in a single query
        $invoices = DB::table('Luv2_invoice as i')
            ->select([
                'i.*',
                'u.username as user_username',
                's.name as sales_name'
            ])
            ->leftJoin('Luv2_user as u', 'i.user_code', '=', 'u.code')
            ->leftJoin('Luv2_salesemp as s', 'i.sales_code', '=', 's.code')
            ->whereIn('i.no', $invoiceNos)
            ->orderByDesc('i.created_at')
            ->get();

        // Get all details in a single query
        $details = DB::table('Luv2_invoice_detail as d')
            ->select([
                'd.*',
                'it.barcode'
            ])
            ->leftJoin('Luv2_item as it', 'd.code', '=', 'it.code')
            ->whereIn('d.no', $invoiceNos)
            ->get()
            ->groupBy('no');

        // Combine the data and create proper objects
        $this->arinvoices = $invoices->map(function ($invoice) use ($details) {
            // Create a proper object with all needed properties
            $invoiceObj = (object)[
                'no' => $invoice->no,
                'created_at' => $invoice->created_at,
                'user' => (object)['username' => $invoice->user_username],
                'detailInvoices' => collect($details[$invoice->no] ?? [])->map(function ($detail) {
                    return (object)[
                        'code' => $detail->code,
                        'name' => $detail->name,
                        'nprice' => $detail->nprice,
                        'price' => $detail->price,
                        'qty' => $detail->qty,
                        'total_price' => $detail->total_price,
                        'item' => (object)['barcode' => $detail->barcode],
                        'sales' => (object)['name' => $detail->sales_name ?? '-']
                    ];
                })
            ];
            return $invoiceObj;
        });
    }

    public function view():View
    {
        return view('export.tablearinvoice', [
            'invoices' => $this->arinvoices,
        ]);
    }
    
}
