import {
  MasterData,
  Package,
  Agent,
  Member,
  Bank,
  Schedule,
} from "./masterDataTypes";
import { API_CONFIG, ASSET_PATHS } from "./config";

// Master Data Store
let masterDataStore: MasterData | null = null;

// Setter untuk master data
export const setMasterData = (data: MasterData) => {
  masterDataStore = data;

  // Console log untuk debugging
  // console.log('masterDataHelpers - setMasterData called with:', data);
  // console.log('masterDataHelpers - packages count:', data.packages?.length || 0);
  // console.log('masterDataHelpers - agents count:', data.agents?.length || 0);
};

// Getter untuk master data
export const getMasterData = (): MasterData | null => {
  return masterDataStore;
};

// Helper functions untuk mengakses data spesifik
export const getPackages = (): Package[] => {
  const packages = masterDataStore?.packages || [];
  // Fallback: if no packages in master data, return empty array
  // This prevents errors when master data hasn't loaded yet
  return packages;
};

export const getAgents = (): Agent[] => {
  return masterDataStore?.agents || [];
};

export const getMembers = (): Member[] => {
  return masterDataStore?.members || [];
};

export const getBanks = (): Bank[] => {
  return masterDataStore?.banks || [];
};

export const getSchedules = (): Schedule[] => {
  return masterDataStore?.schedules || [];
};

export const getBookingNumber = (): string => {
  return masterDataStore?.booking_number || "";
};

// Helper: Filter recommended packages for a person based on criteria
export interface PersonCriteria {
  level: string; // Can be any string, will be matched partially
  age: "A" | "Ad" | "Ch1" | "Ch2";
  type: "L" | "ST" | Array<"L" | "ST">;
  bookingDate?: string; // format: YYYY-MM-DD
  personCount?: number;
  totalPaxCount?: number; // Total adult + children count for minimum pax validation
}

export const getRecommendedPackagesForPerson = (criteria: PersonCriteria) => {
  const packages = getPackages();

  // Early return if no packages available
  if (!packages || packages.length === 0) {
    return [];
  }

  const filteredPackages = packages.filter((pkg) => {
    if (pkg.active !== "Y") return false;

    // Type check (supports single or array) - make it more lenient
    if (criteria.type && criteria.type.length > 0) {
      const allowedTypes: Array<"L" | "ST"> = Array.isArray(criteria.type)
        ? criteria.type
        : [criteria.type];
      if (!allowedTypes.includes(pkg.type as "L" | "ST")) return false;
    }

    // Level check - make it more lenient, show packages even if level doesn't match exactly
    if (criteria.level && criteria.level.trim() !== "") {
      const userLevel = criteria.level.toUpperCase();
      const packageLevel = pkg.level.toUpperCase();

      // More lenient level matching - check if any letter from user level exists in package level
      const levelMatch = userLevel
        .split("")
        .some((letter) => packageLevel.includes(letter));

      if (!levelMatch) return false;
    }

    // Age check - make it more lenient
    if (criteria.age && criteria.age.trim() !== "") {
      const agePass = pkg.age === "A" || pkg.age === criteria.age;
      if (!agePass) return false;
    }

    // Person count check - show all packages regardless of pax requirements
    // The warning will be handled in the UI, not in package filtering

    return true;
  });

  return filteredPackages;
};

// Helper untuk konversi package dari API ke format yang dibutuhkan UI
export const convertPackageToUIFormat = (pkg: Package) => {
  // Console log untuk debugging
  // console.log('masterDataHelpers - convertPackageToUIFormat called with:', pkg);

  // Resolve image URL from API - simplified approach
  let resolvedImage: string = "/placeholder.svg";
  if (
    pkg.image &&
    typeof pkg.image === "string" &&
    pkg.image.trim().length > 0
  ) {
    const img = pkg.image.trim();
    console.log("Image resolution for package:", pkg.code, "Image:", img);

    // Use the correct path directly
    resolvedImage = `https://odyssey.softcomp.io/storage/assets/package/${img}`;
    console.log("Using direct path:", resolvedImage);
  } else {
    console.log("No image or empty image for package:", pkg.code);
  }

  const converted = {
    // Basic info
    id: parseInt(pkg.code.replace("PKG", "")),
    code: pkg.code,
    name: pkg.name,
    type: pkg.type,

    // Descriptions
    desc_short: pkg.desc_short,
    desc_long: pkg.desc_long,

    // Package details
    level: pkg.level,
    person: pkg.pax,
    age: pkg.age,
    active: pkg.active,
    active_from: pkg.active_from,
    active_to: pkg.active_to,

    // Price
    price: parseFloat(pkg.price),

    // UI formatting (for backward compatibility)
    title: pkg.name,
    description: pkg.desc_long,
    category:
      pkg.type === "L" ? "Lesson" : pkg.type === "ST" ? "Surf Tour" : "Package",
    categoryBadges: [
      pkg.type === "L" ? "LESSON" : pkg.type === "ST" ? "TOUR" : "PACKAGE",
    ],
    image: resolvedImage,
    reviews: 50, // Default value
    rating: 5, // Default value
    duration: "2.5 Hours", // Default value
    maxStudents: pkg.pax,
    level_display:
      pkg.level === "B"
        ? "Beginner"
        : pkg.level === "I"
        ? "Intermediate"
        : "Advanced",
    includes: ["Equipment", "Insurance", "Photos"], // Default includes
  };

  // console.log('masterDataHelpers - converted package:', converted);
  return converted;
};

// Helper untuk mendapatkan packages yang sudah dikonversi
export const getConvertedPackages = () => {
  const packages = getPackages();
  return packages.map(convertPackageToUIFormat);
};

// Helper untuk mendapatkan agent berdasarkan code
export const getAgentByCode = (code: string): Agent | undefined => {
  return getAgents().find((agent) => agent.code === code);
};

// Helper untuk mendapatkan member berdasarkan code
export const getMemberByCode = (code: string): Member | undefined => {
  return getMembers().find((member) => member.code === code);
};

// Helper untuk mendapatkan bank berdasarkan code
export const getBankByCode = (code: string): Bank | undefined => {
  return getBanks().find((bank) => bank.code === code);
};

// Helper untuk mengecek pax requirements
export const checkPaxRequirements = (
  selectedPackages: {
    [key: string]: any;
  },
  paxGroups?: Array<{
    id: string;
    ageGroup: string;
    count: number;
    surfLevel: string;
    recommendedPackage: string;
  }>,
  showAgentSection?: boolean
) => {
  const packageGroups: {
    [packageCode: string]: { pax: number; people: string[] };
  } = {};

  if (showAgentSection && paxGroups && paxGroups.length > 0) {
    // Agent mode: Use paxGroups data
    paxGroups.forEach((group) => {
      if (group.recommendedPackage && group.count > 0) {
        const packageCode = group.recommendedPackage;
        if (!packageGroups[packageCode]) {
          // Find package details from selectedPackages
          const packageKey = Object.keys(selectedPackages).find((key) =>
            key.includes(group.id)
          );
          const packageData = packageKey ? selectedPackages[packageKey] : null;
          const packagePax = packageData?.pkg?.pax || 0;

          packageGroups[packageCode] = {
            pax: packagePax,
            people: [],
          };
        }
        // Add people based on count
        for (let i = 1; i <= group.count; i++) {
          packageGroups[packageCode].people.push(`${group.ageGroup} ${i}`);
        }
      }
    });
  } else {
    // Customer mode: Use individual recommendations from formData
    // This is a simplified check - we'll assume requirements are met if packages are selected
    // The detailed validation is handled in the UI warnings
    return {
      allRequirementsMet: true,
      paxWarnings: {},
      packageGroups: {},
    };
  }

  // Check each package group for pax requirements
  const paxWarnings: {
    [packageCode: string]: { required: number; current: number };
  } = {};
  const allRequirementsMet = Object.entries(packageGroups).every(
    ([packageCode, group]) => {
      const currentCount = group.people.length;
      const requiredPax = group.pax;

      if (currentCount < requiredPax) {
        paxWarnings[packageCode] = {
          required: requiredPax,
          current: currentCount,
        };
        return false;
      }
      return true;
    }
  );

  return {
    allRequirementsMet,
    paxWarnings,
    packageGroups,
  };
};
