"use client"

import { useState, useEffect, useMemo, useCallback } from "react"
import { Star } from "lucide-react"
import { setMasterData, getRecommendedPackagesForPerson } from "@/lib/masterDataHelpers"
import { MasterDataResponse } from "@/lib/masterDataTypes"
import { 
  getSchedulesUrl, 
  getMasterDataUrl, 
  getBookingUrl,
  getWithoutPaymentUrl,
  getApplyVoucherUrl,
  getGenerateNumberUrl,
  getUpdatePaymentStatusUrl,
  getApiUrl,
  API_CONFIG,
  MIDTRANS_CONFIG,
  PAYPAL_CONFIG
} from "@/lib/config"
import BookingHeader from "./BookingHeader"
import BookingFooter from "./BookingFooter"
import BookingSummary from "./BookingSummary"
import CustomerInformationForm from "./CustomerInformationForm"
import ParticipantForm from "./ParticipantForm"
import LessonsToursSpecifications from "./LessonsToursSpecifications"
import AgentCodeSection from "./AgentCodeSection"
import LoadingSpinner from "./LoadingSpinner"
import BookingSuccessModal from "./BookingSuccessModal"
import BookingConfirmationModal from "./BookingConfirmationModal"
import PaymentLoadingOverlay from "./PaymentLoadingOverlay"
import { AlertContainer } from "./AlertToast"
import { sendEmail, sendEmailAuto } from "@/service/bookingApi"
import { getBookingPayPalStatusByQuery, getBookingPayPalStatus, sendPayPalConfirmationEmail, getBackendBookingPaymentStatus } from "@/service/paypalApi"

type ActivityKey = "surfLessons" | "surfTours";

interface AdultForm {
  name: string;
  level: string;
  surfLevel?: string;
  medical?: string[];
  medical_other?: string;
  recommendation?: string;
}

interface ChildForm {
  name: string;
  age: string;
  ageGroup?: string;
  level: string;
  surfLevel?: string;
  medical?: string[];
  medical_other?: string;
  recommendation?: string;
}

const activityLabelMap: Record<string, string> = {
  surfLessons: "Surf Lessons",
  surfTours: "Surf Tours",
};

type VoucherData = {
  voucher_code: string;
  discount_type: string;
  discount_value: number;
  discount_amount: number;
  net_amount: number;
  promo_no: string;
  message: string;
};

// Fixed exchange rate for USD payments: 1 USD = 13000 IDR
// Rate (USD per 1 IDR) = 1 / 13000
const USD_FIXED_RATE = 1 / 13000; // 0.000076923076923...

export default function SurfSchoolBooking() {
  // Add client-side rendering check
  const [isClient, setIsClient] = useState(false);

  useEffect(() => {
    setIsClient(true);
  }, []);

  // Check URL parameters for agent mode
  useEffect(() => {
    if (isClient) {
      const urlParams = new URLSearchParams(window.location.search);
      const hasAgentParam = urlParams.has('agent');
      setShowAgentSection(hasAgentParam);
    }
  }, [isClient]);

  const [selectedActivities, setSelectedActivities] = useState<Record<ActivityKey, boolean>>({
    surfLessons: true,
    surfTours: false,
  })

  const [formData, setFormData] = useState<{
    adults: AdultForm[];
    children: ChildForm[];
  }>({
    adults: [],
    children: [],
  })

  const [duration, setDuration] = useState("");

  const [reservationDays, setReservationDays] = useState<Array<{ date: string; time: string }>>([]);

  const [adultCount, setAdultCount] = useState(0);
  const [childrenCount, setChildrenCount] = useState(0);
  const [packageList, setPackageList] = useState([]);

  const [selectedPackages, setSelectedPackages] = useState<{ [key: string]: any }>({});
  const [schedules, setSchedules] = useState<any[]>([]);
  const [loadingSchedules, setLoadingSchedules] = useState(false);

  // Agent states
  const [agentCode, setAgentCode] = useState<string>('');
  const [selectedAgent, setSelectedAgent] = useState<any>(null);
  const [agentCommission, setAgentCommission] = useState<number>(0);
  const [splitPayments, setSplitPayments] = useState<any[]>([]);
  const [loadingAgent, setLoadingAgent] = useState(false);
  const [showAgentSection, setShowAgentSection] = useState(false);
  // Exchange rate (USD per 1 IDR). Null until provided by CurrencySelector or fetched.
  const [usdRate, setUsdRate] = useState<number | null>(null);

  // Debug agentCommission changes
  useEffect(() => {
    console.log('🔄 agentCommission changed to:', agentCommission);
  }, [agentCommission]);

  // Debug selectedAgent changes
  useEffect(() => {
    console.log('🔄 selectedAgent changed to:', selectedAgent);
  }, [selectedAgent]);

  // Agent pax management states
  const [totalPax, setTotalPax] = useState(0);
  const [paxGroups, setPaxGroups] = useState<Array<{
    id: string;
    ageGroup: string;
    count: number;
    surfLevel: string;
    recommendedPackage: string;
  }>>([]);

  // Agent payment amounts state
  const [agentPaymentAmounts, setAgentPaymentAmounts] = useState<{[key: string]: number}>({});

  // Payment states
  const [paymentMethod, setPaymentMethod] = useState("bank");
  const [agreeTerms, setAgreeTerms] = useState(false);
  const [promoCode, setPromoCode] = useState("");
  const [promoMessage, setPromoMessage] = useState("");
  const [promoSuccess, setPromoSuccess] = useState(false);
  const [voucherData, setVoucherData] = useState<VoucherData | null>(null);
  const [isPaymentLoading, setIsPaymentLoading] = useState(false);

  // Convert pax groups to selected packages for agent mode
  useEffect(() => {
    if (showAgentSection && paxGroups.length > 0) {
      const newSelectedPackages: { [key: string]: any } = {};
      
      paxGroups.forEach((group) => {
        if (group.recommendedPackage) {
          // Generate criteria to get package details
          let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
          if (selectedActivities.surfLessons && selectedActivities.surfTours) {
            typeCriteria = ['L', 'ST'];
          } else if (selectedActivities.surfTours) {
            typeCriteria = 'ST';
          } else {
            typeCriteria = 'L';
          }

          const groupCriteria = {
            level: (group.surfLevel === 'beginner' ? 'B' : group.surfLevel === 'intermediate' ? 'I' : 'A'),
            age: (group.ageGroup === 'adult' ? 'Ad' : group.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
            type: typeCriteria,
            bookingDate: reservationDays[0]?.date,
            personCount: group.count,
          };

          const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
          const selectedPackage = recommendedPackages.find(pkg => pkg.code === group.recommendedPackage);
          
          if (selectedPackage) {
            // Create a unique key for this group's package selection
            const packageKey = `${group.id}_${group.recommendedPackage}`;
            newSelectedPackages[packageKey] = {
              pkg: selectedPackage,
              price: selectedPackage.price,
              count: group.count,
              ageGroup: group.ageGroup,
              surfLevel: group.surfLevel
            };
          }
        }
      });
      
      setSelectedPackages(newSelectedPackages);
    } else if (showAgentSection && paxGroups.length === 0) {
      // Clear selected packages when no pax groups
      setSelectedPackages({});
    }
  }, [paxGroups, showAgentSection, selectedActivities, reservationDays]);

  // Helper function to extract packages from customer form data
  const getCustomerPackages = () => {
    if (showAgentSection) return [];
    
    const packages: Array<{ name: string; price: number }> = [];
    
    // Process adults
    (formData?.adults || []).forEach((adult: any) => {
      if (adult.recommendation) {
        // Generate criteria to get package details
        let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
        if (selectedActivities.surfLessons && selectedActivities.surfTours) {
          typeCriteria = ['L', 'ST'];
        } else if (selectedActivities.surfTours) {
          typeCriteria = 'ST';
        } else {
          typeCriteria = 'L';
        }

        const personCriteria = {
          level: (adult.surfLevel === 'beginner' ? 'B' : adult.surfLevel === 'intermediate' ? 'I' : 'A'),
          age: 'Ad' as 'A' | 'Ad' | 'Ch1' | 'Ch2',
          type: typeCriteria,
          bookingDate: reservationDays[0]?.date,
          personCount: 1, // Each adult is counted as 1 person
        };

        const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
        const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === adult.recommendation);
        
        if (selectedPackage) {
          packages.push({
            name: selectedPackage.name || "N/A",
            price: typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price)) || 0
          });
        }
      }
    });
    
    // Process children
    (formData?.children || []).forEach((child: any) => {
      if (child.recommendation) {
        // Generate criteria to get package details
        let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
        if (selectedActivities.surfLessons && selectedActivities.surfTours) {
          typeCriteria = ['L', 'ST'];
        } else if (selectedActivities.surfTours) {
          typeCriteria = 'ST';
        } else {
          typeCriteria = 'L';
        }

        const personCriteria = {
          level: (child.surfLevel === 'beginner' ? 'B' : child.surfLevel === 'intermediate' ? 'I' : 'A'),
          age: (child.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
          type: typeCriteria,
          bookingDate: reservationDays[0]?.date,
          personCount: 1, // Each child is counted as 1 person
        };

        const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
        const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === child.recommendation);
        
        if (selectedPackage) {
          packages.push({
            name: selectedPackage.name || "N/A",
            price: typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price)) || 0
          });
        }
      }
    });
    
    return packages;
  };

  // Calculate total amount for customer mode from formData recommendations
  const calculateCustomerTotalAmount = () => {
    if (showAgentSection) return 0;
    
    const customerPackages = getCustomerPackages();
    return customerPackages.reduce((total, pkg) => total + pkg.price, 0);
  };

  // Calculate total amount from selected packages
  const selectedPackagesArr = Object.values(selectedPackages);
  const totalAmount = showAgentSection ? selectedPackagesArr.reduce((sum, pkg) => {
    const price = pkg?.pkg?.price || 0;
    const count = pkg?.count || 1; // Use count from pax group, default to 1
    const itemTotal = typeof price === 'number' ? price * count : 0;
    return sum + itemTotal;
  }, 0) : calculateCustomerTotalAmount();

  // Calculate total amount directly from pax groups for agent mode
  const calculateAgentTotalAmount = () => {
    if (!showAgentSection || paxGroups.length === 0) return 0;
    
    let total = 0;
    paxGroups.forEach((group) => {
      if (group.recommendedPackage) {
        // Generate criteria to get package details
        let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
        if (selectedActivities.surfLessons && selectedActivities.surfTours) {
          typeCriteria = ['L', 'ST'];
        } else if (selectedActivities.surfTours) {
          typeCriteria = 'ST';
        } else {
          typeCriteria = 'L';
        }

        const groupCriteria = {
          level: (group.surfLevel === 'beginner' ? 'B' : group.surfLevel === 'intermediate' ? 'I' : 'A'),
          age: (group.ageGroup === 'adult' ? 'Ad' : group.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
          type: typeCriteria,
          bookingDate: reservationDays[0]?.date,
          personCount: group.count,
        };

        const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
        const selectedPackage = recommendedPackages.find(pkg => pkg.code === group.recommendedPackage);
        
        if (selectedPackage && selectedPackage.price) {
          const price = typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price));
          const count = typeof group.count === 'number' ? group.count : parseInt(String(group.count));
          total += price * count;
        }
      }
    });
    
    return total;
  };

  // Use agent total calculation if in agent mode, otherwise use customer total
  const finalTotalAmount = showAgentSection ? calculateAgentTotalAmount() : totalAmount;

  // Function to re-apply voucher when total amount changes
  const reapplyVoucherIfActive = useCallback(async () => {
    // Only re-apply if there's an active voucher and promo code
    if (voucherData && promoCode && promoCode.trim() !== "") {
      try {
        const response = await fetch(getApplyVoucherUrl(), {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
            "Authorization": `Bearer ${API_CONFIG.TOKEN}`,
          },
          body: JSON.stringify({
            voucher_code: promoCode,
            gross_amount: finalTotalAmount,
          }),
          credentials: "omit",
        });

        const data = await response.json();

        if (response.ok && data.success) {
          // Update voucher data with new amount
          setVoucherData(data.data);
        } else {
          // If voucher becomes invalid with new amount, clear it
          setVoucherData(null);
          setPromoMessage("");
          setPromoSuccess(false);
        }
      } catch (err) {
        // On error, clear voucher data
        setVoucherData(null);
        setPromoMessage("");
        setPromoSuccess(false);
      }
    }
  }, [voucherData, promoCode, finalTotalAmount]);

  // Re-apply voucher when total amount changes
  useEffect(() => {
    // Only re-apply if there's an active voucher and promo code
    if (voucherData && promoCode && promoCode.trim() !== "") {
      // Add a small delay to avoid too many API calls
      const timeoutId = setTimeout(() => {
        reapplyVoucherIfActive();
      }, 500);

      return () => clearTimeout(timeoutId);
    }
  }, [finalTotalAmount, reapplyVoucherIfActive]);

  // Fetch schedules from API
  const fetchSchedules = async () => {
    try {
      setLoadingSchedules(true);
      const response = await fetch(getSchedulesUrl(), {
        headers: {
          'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
          'Content-Type': 'application/json'
        }
      });
      const data = await response.json();
      
      if (data.success) {
        setSchedules(data.data);
      } else {
        console.error('Failed to fetch schedules:', data.message);
      }
    } catch (error) {
      console.error('Error fetching schedules:', error);
    } finally {
      setLoadingSchedules(false);
    }
  };

  // Fetch master data from API
  const fetchMasterData = async () => {
    try {
      const response = await fetch(getMasterDataUrl(), {
        headers: {
          'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
          'Content-Type': 'application/json'
        }
      });
      const data: MasterDataResponse = await response.json();
      
      if (data.success) {
        setMasterData(data.data); // Store raw master data
        
        // Console log untuk debugging
        // console.log('SurfSchoolBooking - Master data received:', data.data);
        // console.log('SurfSchoolBooking - Packages:', data.data.packages);
        // console.log('SurfSchoolBooking - Agents:', data.data.agents);
        
        // Store agents data if available
        if (data.data.agents) {
          // Agents data is available in master data
          // console.log('Agents loaded from master data:', data.data.agents.length);
        }
      } else {
        console.error('Failed to fetch master data:', data.message);
      }
    } catch (error) {
      console.error('Error fetching master data:', error);
    }
  };

  // Fetch agent by code from master data
  const fetchAgentByCode = async (code: string) => {
    try {
      // Validate input
      if (!code || code.trim() === '') {
        showAlert('Please enter an agent code', 'warning');
        return;
      }

      setLoadingAgent(true);

      // Get agents from master data
      const response = await fetch(getMasterDataUrl(), {
        headers: {
          'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      
      if (data.success && data.data && data.data.agents) {
        const agent = data.data.agents.find((a: any) => a.code === code.toUpperCase());
        
        if (agent) {
          console.log('🔍 Agent found:', agent);
          console.log('🔍 Agent commission from API:', agent.commission);
          console.log('🔍 Agent commission type:', typeof agent.commission);
          
          setSelectedAgent(agent);
          setAgentCode(code.toUpperCase());
          setAgentCommission(Number(agent.commission) || 0);
          
          console.log('🔍 Set agentCommission to:', Number(agent.commission) || 0);
          validateField('agentCode', code.toUpperCase());
          showAlert(`Agent ${agent.name} applied successfully!`, 'success');
        } else {
          // Agent not found - this is normal, not an error
          showAlert(`Agent code "${code.toUpperCase()}" not found. Please check the code and try again.`, 'warning');
          setSelectedAgent(null);
          setAgentCommission(0);
          setAgentCode('');
          validateField('agentCode', '');
        }
      } else {
        console.error('Failed to fetch master data:', data.message);
        showAlert('Unable to fetch agent data. Please try again later.', 'error');
        setSelectedAgent(null);
        setAgentCommission(0);
        validateField('agentCode', code.toUpperCase());
      }
    } catch (error: any) {
      console.error('Error fetching agent:', error);
      
      // Handle different types of errors
      if (error.name === 'TypeError' && error.message.includes('fetch')) {
        showAlert('Network error. Please check your internet connection and try again.', 'error');
              } else if (error.message.includes('HTTP error')) {
          showAlert('Server error. Please try again later.', 'error');
        } else {
          showAlert('An unexpected error occurred. Please try again.', 'error');
        }
      
      setSelectedAgent(null);
      setAgentCommission(0);
      validateField('agentCode', code.toUpperCase());
    } finally {
      setLoadingAgent(false);
    }
  };

  // Load schedules on component mount
  useEffect(() => {
    if (isClient) {
      fetchSchedules();
      fetchMasterData();
    }
  }, [isClient]);

  // Clear agent code error when valid agent is selected
  useEffect(() => {
    if (selectedAgent && errors.agentCode) {
      setErrors((prev: any) => {
        const newErrors = { ...prev };
        delete newErrors.agentCode;
        return newErrors;
      });
    }
  }, [selectedAgent]);

  // Load data from localStorage on mount - DISABLED
  // useEffect(() => {
  //   if (typeof window !== "undefined" && isClient) {
  //     const raw = localStorage.getItem("surf-booking-data");
  //     if (raw) {
  //       try {
  //         const saved = JSON.parse(raw);
  //         setFormData(saved.formData || { 
  //           adults: [], 
  //           children: [] 
  //         });
  //         setSelectedActivities(saved.selectedActivities || { surfLessons: true, surfTours: false });
  //         setDuration(saved.duration || "1-day");
  //         setReservationDays(saved.reservationDays || [{ date: "", time: "" }]);
  //         setAdultCount(saved.adultCount || 0);
  //         setChildrenCount(saved.childrenCount || 0);
  //         setSelectedPackages(saved.selectedPackages || {});
  //       } catch (error) {
  //         console.error('Error parsing localStorage data:', error);
  //       }
  //     }
  //   }
  // }, [isClient]);

  // Save data to localStorage on change - DISABLED
  // useEffect(() => {
  //   if (typeof window !== "undefined" && isClient) {
  //     localStorage.setItem("surf-booking-data", JSON.stringify({
  //       formData,
  //       selectedActivities,
  //       duration,
  //       reservationDays,
  //       adultCount,
  //       childrenCount,
  //       selectedPackages,
  //     }));
  //   }
  // }, [formData, selectedActivities, duration, reservationDays, adultCount, childrenCount, selectedPackages, isClient]);

  useEffect(() => {
    if (!duration) {
      setReservationDays([]);
      return;
    }
    const dayCount = parseInt(duration); // "1-day" => 1, "2-days" => 2, dst
    if (Number.isNaN(dayCount) || dayCount <= 0) {
      setReservationDays([]);
      return;
    }
    setReservationDays((prev) => {
      const newArr = [...prev];
      if (newArr.length < dayCount) {
        while (newArr.length < dayCount) newArr.push({ date: "", time: "" });
      } else if (newArr.length > dayCount) {
        newArr.length = dayCount;
      }
      return newArr;
    });
  }, [duration]);

  // Build dynamic summary from props
  const activityLabels = Object.entries(selectedActivities)
    .filter(([_, v]) => v)
    .map(([k]) => activityLabelMap[k] || k)
    .join(", ");
  const durationLabel = !duration ? "Select Duration" : (duration === "1-day" ? "1 Day" : duration === "2-days" ? "2 Days" : "3 Days");
  const criteria = `${activityLabels} | ${durationLabel} | ${adultCount} Adult${childrenCount > 0 ? ` | ${childrenCount} Children` : ""}`;

  // Ambil level unik dari adults & children
  const allLevels = [
    ...formData.adults.map(a => a.level),
    ...formData.children.map(c => c.level)
  ].filter(Boolean);
  const uniqueLevels = Array.from(new Set(allLevels));
  // Ambil inisial nama pertama (jika ada)
  const firstName = formData.adults[0]?.name || formData.children[0]?.name || "-";
  const initials = firstName ? firstName.split(" ").map(s => s[0]).join("").slice(0,3).toLowerCase() : "-";
  const levelLabel = uniqueLevels.length > 0 ? uniqueLevels.join(", ") : "-";

  // Currency state & options
  const [currency, setCurrency] = useState("IDR");
  const currencyOptions = useMemo(() => [
    { code: "IDR", label: "Indonesian Rupiah", rate: 1 },
    { code: "USD", label: "United State Dollar", rate: 0.000067 },
    { code: "AUD", label: "Australian Dollar", rate: 0.0001 },
    { code: "SGD", label: "Singapore Dollar", rate: 0.00009 },
    { code: "JPY", label: "Japanese Yen", rate: 0.0098 },
    { code: "EUR", label: "Euro", rate: 0.00006 },
  ], []);
  const selectedCurrency = currencyOptions.find(c => c.code === currency) || currencyOptions[0];

  // Memoize setCurrency function to prevent unnecessary re-renders
  const handleSetCurrency = useCallback((newCurrency: string) => {
    setCurrency(newCurrency);
  }, []);

  function getDisplayPrice(price: any) {
    let num = typeof price === 'number' ? price : parseFloat((price || '').toString().replace(/,/g, ''));
    
    if (isNaN(num)) return '-';
    
    // Use fixed exchange rate for USD: 1 USD = 13000 IDR
    const rate = currency === 'USD' ? USD_FIXED_RATE : selectedCurrency.rate;
    const converted = num * rate;
    
    if (currency === 'USD') {
      // For USD, show 2 decimal places
      return `$${converted.toFixed(2)}`;
    } else {
      // For other currencies, round to whole numbers
      const rounded = Math.round(converted);
      return `${selectedCurrency.code} ${rounded.toLocaleString()}`;
    }
  }

  // Agent pax management helper functions
  const addPaxGroup = () => {
    const newGroup = {
      id: Date.now().toString(),
      ageGroup: 'adult',
      count: 1,
      surfLevel: 'beginner',
      recommendedPackage: ''
    };
    setPaxGroups(prev => [...prev, newGroup]);
  };

  const updatePaxGroup = (id: string, field: string, value: any) => {
    setPaxGroups(prev => prev.map(group => 
      group.id === id ? { ...group, [field]: value } : group
    ));
  };

  const removePaxGroup = (id: string) => {
    setPaxGroups(prev => prev.filter(group => group.id !== id));
  };

  const getRemainingPax = () => {
    const usedPax = paxGroups.reduce((sum, group) => sum + group.count, 0);
    return Math.max(0, totalPax - usedPax);
  };

  const isPaxValid = () => {
    const usedPax = paxGroups.reduce((sum, group) => sum + group.count, 0);
    return usedPax === totalPax && totalPax > 0;
  };

  // Package requirement validation for agent mode
  const getPackageRequirementMessage = () => {
    if (!showAgentSection || paxGroups.length === 0) return null;
    
    const packageGroups = paxGroups.filter(group => group.recommendedPackage);
    const packageCounts: { [key: string]: number } = {};
    
    packageGroups.forEach(group => {
      const packageCode = group.recommendedPackage;
      packageCounts[packageCode] = (packageCounts[packageCode] || 0) + group.count;
    });
    
    const messages: Array<{package: string, packageName: string, current: number, required: number, needed: number}> = [];
    
    // Get package details for each selected package
    Object.entries(packageCounts).forEach(([packageCode, count]) => {
      // Find package details from any group that has this package
      const groupWithPackage = paxGroups.find(group => group.recommendedPackage === packageCode);
      if (groupWithPackage) {
        // Generate criteria to get package details
        let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
        if (selectedActivities.surfLessons && selectedActivities.surfTours) {
          typeCriteria = ['L', 'ST'];
        } else if (selectedActivities.surfTours) {
          typeCriteria = 'ST';
        } else {
          typeCriteria = 'L';
        }
        
        const groupCriteria = {
          level: (groupWithPackage.surfLevel === 'beginner' ? 'B' : groupWithPackage.surfLevel === 'intermediate' ? 'I' : 'A'),
          age: (groupWithPackage.ageGroup === 'adult' ? 'Ad' : groupWithPackage.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
          type: typeCriteria,
          bookingDate: reservationDays[0]?.date,
          personCount: groupWithPackage.count,
        };
        
        const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
        const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === packageCode);
        
        if (selectedPackage && selectedPackage.pax) {
          const requiredPax = selectedPackage.pax;
          
          // Check if current count meets requirements (must be multiple of required pax)
          if (count < requiredPax || count % requiredPax !== 0) {
            let needed = 0;
            if (count < requiredPax) {
              needed = requiredPax - count;
            } else {
              needed = requiredPax - (count % requiredPax);
            }
            
            messages.push({
              package: packageCode,
              packageName: selectedPackage.name,
              current: count,
              required: requiredPax,
              needed: needed
            });
          }
        }
      }
    });
    
    return messages.length > 0 ? messages : null;
  };

  const selectedActivityLabels = Object.entries(selectedActivities)
    .filter(([_, v]) => v)
    .map(([k]) => activityLabelMap[k] || k)
    .join(", ");

  const adultsSummary = formData.adults.length
    ? formData.adults.map((a, i) => `${a.name || '-'} (${a.level || '-'})`).join(', ')
    : '-';
  const childrenSummary = formData.children.length
    ? formData.children.map((c, i) => `${c.name || '-'} (${c.age || '-'}, ${c.level || '-'})`).join(', ')
    : '-';

  // State for activity selection logic (individual/peer)
  const [selectedActivityType, setSelectedActivityType] = useState<"individual" | "peer" | null>(null);
  
  // State for last alert message
  const [alerts, setAlerts] = useState<Array<{ id: string; message: string; type: 'success' | 'error' | 'warning' | 'info' }>>([]);

  // State for booking success modal
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [successBookingData, setSuccessBookingData] = useState<{
    bookingNumber: string;
    bookingDetails?: {
      date: string;
      duration: string;
      participants: string;
      totalAmount: string;
      customerName: string;
    };
  } | null>(null);

  // State for Midtrans payment
  const [snapToken, setSnapToken] = useState<string | null>(null);
  const [showMidtransModal, setShowMidtransModal] = useState(false);

  // State for PayPal payment
  const [showBookingConfirmationModal, setShowBookingConfirmationModal] = useState(false);
  const [bookingConfirmationData, setBookingConfirmationData] = useState<any>(null);
  const [isProcessingPayPal, setIsProcessingPayPal] = useState(false);
  const [paypalApprovalUrl, setPaypalApprovalUrl] = useState<string | null>(null);
  const [paypalBookingNumber, setPaypalBookingNumber] = useState<string | null>(null);
  const [paypalPaymentStatus, setPaypalPaymentStatus] = useState<string | null>(null);
  const [isCheckingPayPalStatus, setIsCheckingPayPalStatus] = useState(false);
  const [isMonitoringPayPal, setIsMonitoringPayPal] = useState(false);

  // Send PayPal email with retry (non-blocking)
  const sendPayPalEmailWithRetry = async (bookingNo: string, attempts = 3) => {
    for (let i = 1; i <= attempts; i++) {
      try {
        await sendPayPalConfirmationEmail(bookingNo);
        console.log(`✅ PayPal confirmation email sent (attempt ${i})`);
        return true;
      } catch (err) {
        console.warn(`❌ Email send failed (attempt ${i})`, err);
        if (i < attempts) {
          await new Promise(res => setTimeout(res, i * 1500));
          continue;
        }
        showAlert('Email confirmation gagal dikirim. Anda bisa mencoba lagi dari riwayat booking.', 'error');
        return false;
      }
    }
    return false;
  };

  // Add payment status modal states
  const [showPaymentFailedModal, setShowPaymentFailedModal] = useState(false);
  const [showPaymentSettlementModal, setShowPaymentSettlementModal] = useState(false);
  const [showPaymentPendingModal, setShowPaymentPendingModal] = useState(false);
  const [showPaymentExpiredModal, setShowPaymentExpiredModal] = useState(false);
  const [showPaymentDeniedModal, setShowPaymentDeniedModal] = useState(false);

  // Toast alert function
  function showAlert(msg: string, type: 'success' | 'error' | 'warning' | 'info' = 'info') {
    const id = `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
    setAlerts(alerts => [...alerts, { id, message: msg, type }]);
    setTimeout(() => {
      setAlerts(alerts => alerts.filter(alert => alert.id !== id));
    }, 4000); 
  }

  // Helper: get key for a recommendation (for individual: name, for peer: joined names)
  function getRecKey(rec: any) {
    return `${rec.type}:${rec.people.join("|")}`;
  }

  // Handler when a package is selected
  function handleSelect(rec: any) {
    // Validasi nama sebelum memilih paket
    const missingNames: string[] = [];
    
    rec.people.forEach((personName: string) => {
      // Skip jika nama kosong atau hanya whitespace
      if (!personName || personName.trim() === "") {
        missingNames.push("Participant");
        return;
      }
      
      // Cari participant berdasarkan nama
      const adult = formData.adults.find((a: any) => a.name === personName);
      const child = formData.children.find((c: any) => c.name === personName);
      
      if (!adult && !child) {
        missingNames.push(personName);
      }
    });
    
    if (missingNames.length > 0) {
      const participantType = missingNames.length === 1 ? 'participant' : 'participants';
      const namesList = missingNames.includes("Participant") ? "participant" : missingNames.join(', ');
      showAlert(`Please fill in the name for ${participantType}: ${namesList}`, 'warning');
      return;
    }
    
    const isIndividual = rec.people.length === 1;
    const isPeer = rec.people.length > 1;
    let newSelected = { ...selectedPackages };

    if (isIndividual) {
      // Remove peer packages that include this person
      Object.entries(newSelected).forEach(([key, selRec]) => {
        if (selRec.people.length > 1 && selRec.people.includes(rec.people[0])) {
          delete newSelected[key];
        }
      });
      // Remove individual packages for the same person and same type (now handled by unique key)
      Object.entries(newSelected).forEach(([key, selRec]) => {
        if (selRec.people.length === 1 && selRec.people[0] === rec.people[0] && selRec.type === rec.type) {
          delete newSelected[key];
        }
      });
      newSelected[getRecKey(rec)] = rec;
    } else if (isPeer) {
      // Remove individual packages for people in this peer group (regardless of type)
      rec.people.forEach((person: string) => {
        Object.entries(newSelected).forEach(([key, selRec]) => {
          if (selRec.people.length === 1 && selRec.people[0] === person) {
            delete newSelected[key];
          }
        });
      });
      // Remove other peer packages that overlap with this one
      Object.entries(newSelected).forEach(([key, selRec]) => {
        if (selRec.people.length > 1 && selRec.people.some((p: string) => rec.people.includes(p))) {
          delete newSelected[key];
        }
      });
      newSelected[getRecKey(rec)] = rec;
    }
    
    setSelectedPackages(newSelected);
    
    // Update participant recommendation field when package is selected
    if (isIndividual && rec.people.length === 1) {
      const personName = rec.people[0];
      const packageCode = rec.pkg?.code;
      const participantType = rec.type; // 'adult' or 'children'
      
      if (packageCode) {
        setFormData((prev: any) => {
          const newFormData = { ...prev };
          
          // Only update recommendation for the specific participant type
          if (participantType === 'adult') {
            const adultIndex = newFormData.adults.findIndex((a: any) => a.name === personName);
            if (adultIndex !== -1) {
              newFormData.adults[adultIndex] = {
                ...newFormData.adults[adultIndex],
                recommendation: packageCode
              };
            }
          } else if (participantType === 'children') {
            const childIndex = newFormData.children.findIndex((c: any) => c.name === personName);
            if (childIndex !== -1) {
              newFormData.children[childIndex] = {
                ...newFormData.children[childIndex],
                recommendation: packageCode
              };
            }
          }
          
          return newFormData;
        });
      }
    }
  }

  function handleCancel(recKey: string) {
    // Get the package being cancelled to find affected participants
    const cancelledPackage = selectedPackages[recKey];
    
    setSelectedPackages(prev => {
      const newSelected = { ...prev };
      delete newSelected[recKey];
      return newSelected;
    });
    
    // Clear recommendation field for participants when package is cancelled
    if (cancelledPackage && cancelledPackage.people.length === 1) {
      const personName = cancelledPackage.people[0];
      const participantType = cancelledPackage.type; // 'adult' or 'children'
      
      setFormData((prev: any) => {
        const newFormData = { ...prev };
        
        // Only clear recommendation for the specific participant type
        if (participantType === 'adult') {
          const adultIndex = newFormData.adults.findIndex((a: any) => a.name === personName);
          if (adultIndex !== -1) {
            newFormData.adults[adultIndex] = {
              ...newFormData.adults[adultIndex],
              recommendation: ""
            };
          }
        } else if (participantType === 'children') {
          const childIndex = newFormData.children.findIndex((c: any) => c.name === personName);
          if (childIndex !== -1) {
            newFormData.children[childIndex] = {
              ...newFormData.children[childIndex],
              recommendation: ""
            };
          }
        }
        
        return newFormData;
      });
    }
  }

  // Helper: get all participant names
  const allParticipantNames = [
    ...formData.adults.map(a => a.name),
    ...formData.children.map(c => c.name)
  ].filter(Boolean);

  // Helper: check if a participant is covered by any selected package
  function isParticipantCovered(name: string) {
    return Object.values(selectedPackages).some((rec: any) => rec.people.includes(name));
  }

  // Validation: are all participants covered?
  const allParticipantsSelected = allParticipantNames.length > 0 && allParticipantNames.every(isParticipantCovered);
  
  const [formData2, setFormData2] = useState({
    title: "",
    fullName: "",
    email: "",
    mobilePhone: "",
    hotel: "",
    hotelAddress: "",
    bookingName: "",
    dateOfArrival: "",
    country: "",
    hotelTransfer: "No",
    roomNumber: "", // ✅ Added room number field
    notes: "",
  })

  const [agreeAndContinue, setAgreeAndContinue] = useState(false)

  const dummySurfExperience = {
    year: '2023',
    months: '6',
    weeks: '3',
    locations: 'Bali, Lombok, Mentawai',
    boardSize: '6.2 ft',
    stance: 'Regular',
    stanceOther: '',
    waveSize: 'Medium to large (4 - 6 ft)',
    otherSports: ['Skateboarding', 'Snowboarding'],
    lastSurf: 'Two weeks ago at Kuta Beach, Bali',
  };

  const [surfExperience, setSurfExperience] = useState(dummySurfExperience);

  const [errors, setErrors] = useState<any>({});

  // Validasi terpusat untuk semua form
  function validateAll() {
    const newErrors: any = {};

    if (showAgentSection) {
      // AGENT MODE VALIDATION - Clean and simple
      
      // 1. Agent code validation
      if (!agentCode || agentCode.trim() === '') {
        newErrors.agentCode = "Agent code is required";
      } else if (!selectedAgent) {
        newErrors.agentCode = "Please search for a valid agent code";
      }

      // 2. Activities validation
      if (!Object.values(selectedActivities).some(Boolean)) {
        newErrors.activities = "Please select at least one surfing activity";
      }

      // 3. Reservation dates validation
      if (reservationDays.length === 0) {
        newErrors.reservation = "Please select at least one reservation date";
      } else {
        reservationDays.forEach((r, i) => {
          if (!r.date) {
            newErrors[`reservation_date_${i}`] = "Date required";
          }
          // Only validate time if time field is visible (when date exists and surf lessons are selected)
          if (r.date && selectedActivities.surfLessons && !r.time) {
            newErrors[`reservation_time_${i}`] = "Time required";
          }
        });
      }

      // 4. Pax groups validation
      if (paxGroups.length === 0) {
        newErrors.paxGroups = "Please add at least one pax group";
      } else {
        // Check if all pax groups have packages
        const groupsWithoutPackages = paxGroups.filter(group => !group.recommendedPackage || group.count === 0);
        if (groupsWithoutPackages.length > 0) {
          newErrors.selectedPackages = "Please select packages for all pax groups";
        }
      }

      // 5. Terms validation
      if (!agreeTerms) {
        newErrors.agreeTerms = "You must agree to the terms";
      }

    } else {
      // CUSTOMER MODE VALIDATION - Original logic
      
      // Customer info validation
      if (!formData2.fullName) newErrors.fullName = "Full name is required";
      if (!formData2.email) newErrors.email = "Email is required";
      else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData2.email)) {
        newErrors.email = "Please enter a valid email address";
      }
      if (!formData2.hotelTransfer) newErrors.hotelTransfer = "Hotel transfer is required";
      
      // Hotel fields are only required if hotel transfer is "Yes"
      if (formData2.hotelTransfer === "Yes") {
        if (!formData2.hotel) newErrors.hotel = "Hotel is required";
        if (!formData2.hotelAddress) newErrors.hotelAddress = "Hotel address is required";
        if (!formData2.bookingName) newErrors.bookingName = "Booking name is required";
        if (!formData2.roomNumber) newErrors.roomNumber = "Room number is required";
      }

      // Agent code validation - SKIPPED for customer mode
      // Customer mode doesn't require agent code

      // Activities validation
      if (!Object.values(selectedActivities).some(Boolean)) {
        newErrors.activities = "Please select at least one surfing activity";
      }

      // Participants validation
      if (adultCount === 0 && childrenCount === 0) {
        newErrors.participants = "At least one participant required";
      }
      formData.adults.forEach((a, i) => {
        if (!a.name) newErrors[`adult_name_${i}`] = "Name required";
        if (!a.surfLevel) newErrors[`adult_level_${i}`] = "Level required";
      });
      formData.children.forEach((c, i) => {
        if (!c.name) newErrors[`child_name_${i}`] = "Name required";
        if (!c.ageGroup) newErrors[`child_age_${i}`] = "Age required";
        if (!c.surfLevel) newErrors[`child_level_${i}`] = "Level required";
      });

      // Reservation validation
      const selectedDates = reservationDays.map(r => r.date).filter(Boolean);
      const uniqueDates = [...new Set(selectedDates)];
      
      reservationDays.forEach((r, i) => {
        if (!r.date) {
          newErrors[`reservation_date_${i}`] = "Date required";
        } else {
          // Check for duplicate dates
          const dateCount = selectedDates.filter(date => date === r.date).length;
          if (dateCount > 1) {
            newErrors[`reservation_date_${i}`] = "Each day must have a different date";
          }
        }
        
        // Only validate time if time field is visible (when date exists and surf lessons are selected)
        if (r.date && selectedActivities.surfLessons && !r.time) {
          newErrors[`reservation_time_${i}`] = "Time required";
        }
      });

      // Packages validation - Customer mode
      const allParticipantNames = [
        ...formData.adults.map(a => a.name),
        ...formData.children.map(c => c.name)
      ].filter(name => name.trim() !== "");

      if (allParticipantNames.length === 0) {
        newErrors.participants = "At least one participant required";
      } else {
        // Check if all participants have packages
        const participantsWithoutPackages = allParticipantNames.filter(name => {
          const adult = formData.adults.find(a => a.name === name);
          const child = formData.children.find(c => c.name === name);
          const participant = adult || child;
          const hasRecommendation = participant?.recommendation && participant.recommendation.trim() !== "";
          console.log(`🔍 Participant ${name}:`, { participant, hasRecommendation, recommendation: participant?.recommendation });
          return !hasRecommendation;
        });

        console.log('🔍 All participant names:', allParticipantNames);
        console.log('🔍 Participants without packages:', participantsWithoutPackages);

        if (participantsWithoutPackages.length > 0) {
          newErrors.packageSelection = `The following participants haven't selected a package: ${participantsWithoutPackages.join(', ')}`;
        } else {
          delete newErrors.packageSelection;
        }
      }

      // Payment method validation
      if (!paymentMethod) {
        newErrors.paymentMethod = "Please select a payment method";
      } else {
        // Check if payment method is valid for selected currency and agent
        if (currency === 'USD') {
          if (selectedAgent) {
            // USD + Agent: Only PayPal allowed with fixed rate
            if (paymentMethod !== 'paypal') {
              newErrors.paymentMethod = "USD currency with agent only supports PayPal payments";
            }
          } else {
            // USD + No Agent: Only PayPal allowed
            if (paymentMethod !== 'paypal') {
              newErrors.paymentMethod = "USD currency without agent only supports PayPal payments";
            }
          }
        } else {
          // IDR currency
          if (paymentMethod === 'paypal') {
            newErrors.paymentMethod = "PayPal only supports USD currency";
          } else if (selectedAgent) {
            // Validasi untuk agent payment - menggunakan agentPaymentAmounts
            const agentPaymentMethods = selectedAgent.payment || '';
            const availableMethods = [];
            
            if (agentPaymentMethods.includes('B')) availableMethods.push('bank');
            if (agentPaymentMethods.includes('C')) availableMethods.push('credit');
            if (agentPaymentMethods.includes('O')) availableMethods.push('onsite');
            if (agentPaymentMethods.includes('S')) availableMethods.push('saldo');
            
            // Validasi untuk agent: pastikan setidaknya satu payment method memiliki amount > 0
            const totalAgentAmount = splitPayments.reduce((sum, payment) => sum + (payment.amount || 0), 0);
            const netAmount = voucherData?.net_amount || finalTotalAmount;
            
            // Skip payment amount validation for agent mode
            if (false) { // Disabled for agent mode
              if (totalAgentAmount === 0) {
                newErrors.paymentMethod = "Please enter payment amount for at least one payment method";
              } else if (Math.abs(totalAgentAmount - netAmount) > 0.01) {
                newErrors.paymentMethod = `Total payment amount (${getDisplayPrice(totalAgentAmount)}) must equal booking amount (${getDisplayPrice(netAmount)})`;
              }
            }
            
            // Validasi saldo limit - keep this validation
            const saldoPayment = splitPayments.find(p => p.method === 'saldo');
            const saldoAmount = saldoPayment?.amount || 0;
            const maxSaldo = selectedAgent?.saldo ? parseFloat(selectedAgent.saldo) : 0;
            
            if (saldoAmount > maxSaldo) {
              newErrors.paymentMethod = `Saldo payment (${getDisplayPrice(saldoAmount)}) exceeds available balance (${getDisplayPrice(maxSaldo)})`;
            }
          } else {
            // Validasi untuk non-agent payment (Midtrans/PayPal)
            const validNonAgentMethods = ['midtrans', 'paypal'];
            if (!validNonAgentMethods.includes(paymentMethod)) {
              newErrors.paymentMethod = "Please select a valid payment method (Midtrans or PayPal)";
            }
          }
        }
      }

      // Terms validation
      if (!agreeTerms) {
        newErrors.agreeTerms = "You must agree to the terms";
      }
    }

    return newErrors;
  }

  // Get available payment methods based on currency and agent
  const getAvailablePaymentMethods = () => {
    if (currency === 'USD') {
      // USD currency supports PayPal (with fixed rate for agent, dynamic rate for customer)
      return [
        { code: 'paypal', name: 'PayPal', description: selectedAgent ? 'International payment (USD) - Fixed rate 1 USD = 13000 IDR' : 'International payment (USD)', icon: '💳' }
      ];
    } else {
      // IDR currency supports agent payments and Midtrans
      if (selectedAgent) {
        // Agent is selected, use agent's payment methods
        const agentPaymentMethods = selectedAgent.payment || '';
        const methods = [];
        
        if (agentPaymentMethods.includes('B')) methods.push({ code: 'bank', name: 'Bank Transfer', description: 'BCA, Mandiri, BNI', icon: '🏛️' });
        if (agentPaymentMethods.includes('C')) methods.push({ code: 'credit', name: 'Credit', description: 'Remaining payment stored as credit', icon: '💳' });
        if (agentPaymentMethods.includes('O')) methods.push({ code: 'onsite', name: 'Onsite Payment', description: 'Pay at location', icon: '🏪' });
        if (agentPaymentMethods.includes('S')) methods.push({ code: 'saldo', name: 'Saldo/Balance', description: 'Account balance', icon: '💰' });
        
        return methods;
      } else {
        // No agent selected, use Midtrans for IDR
        return [
          { code: 'midtrans', name: 'Midtrans', description: 'Local payment gateway', icon: '🏦' }
        ];
      }
    }
  };

  // Handler tombol Payment
  async function handlePayment() {
    const validationErrors = validateAll();
    setErrors(validationErrors);
  
    if (Object.keys(validationErrors).length > 0) {
      // Recalculate voucher if it exists and validation fails
      if (promoCode && promoCode.trim() !== '') {
        try {
          await applyVoucher();
        } catch (error) {
          console.error('Error recalculating voucher after validation failure:', error);
        }
      }
      
      // Show toast with validation errors
      const errorMessages = Object.values(validationErrors).join('\n');
      showAlert(`Please complete the following data:\n\n${errorMessages}`, 'error');
      return;
    }
  
    // Set loading state
    setIsPaymentLoading(true);
  
    try {
      // 1. Generate booking number
      const bookingNumber = await generateBookingNumber();
  
      // 2. Mapping data ke payload backend
      let gross = finalTotalAmount || 0;
      const disc = voucherData?.discount_amount ? Math.round(Number(voucherData.discount_amount)) : 0;
      const discp = voucherData?.discount_type === "percentage" ? Number(voucherData.discount_value || 0) : 0;
      let nett = voucherData?.net_amount ? Math.round(Number(voucherData.net_amount)) : gross;
      const promo_no = voucherData?.promo_no || "";
      const voucher_code = voucherData?.voucher_code || "";

  
      // Group booking details by package code and sum up quantities and amounts
      let groupedBookingDetails: any[] = [];
      
      if (showAgentSection) {
        // AGENT MODE: Build booking details from paxGroups
        const packageGroups: { [key: string]: { count: number; packageInfo: any } } = {};
        
        paxGroups.forEach((group) => {
          if (group.recommendedPackage && group.count > 0) {
            // Generate criteria to get package details
            let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
            if (selectedActivities.surfLessons && selectedActivities.surfTours) {
              typeCriteria = ['L', 'ST'];
            } else if (selectedActivities.surfTours) {
              typeCriteria = 'ST';
            } else {
              typeCriteria = 'L';
            }

            const groupCriteria = {
              level: (group.surfLevel === 'beginner' ? 'B' : group.surfLevel === 'intermediate' ? 'I' : 'A'),
              age: (group.ageGroup === 'adult' ? 'Ad' : group.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
              type: typeCriteria,
              bookingDate: reservationDays[0]?.date,
              personCount: group.count,
            };

            const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
            const selectedPackage = recommendedPackages.find(pkg => pkg.code === group.recommendedPackage);
            
            if (selectedPackage) {
              const packageCode = selectedPackage.code;
              const packagePrice = typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price));
              
              if (packageGroups[packageCode]) {
                packageGroups[packageCode].count += group.count;
              } else {
                packageGroups[packageCode] = {
                  count: group.count,
                  packageInfo: selectedPackage
                };
              }
            }
          }
        });
        
        // Convert package groups to booking details format
        groupedBookingDetails = Object.values(packageGroups).map((group: any) => ({
          code: group.packageInfo.code,
          name: group.packageInfo.name || group.packageInfo.title || "",
          price: group.packageInfo.price,
          qty: group.count,
          gross: group.packageInfo.price * group.count,
          disc: 0,
          discp: 0,
          nett: group.packageInfo.price * group.count,
        }));
      } else {
        // CUSTOMER MODE: Create booking details from participant recommendations
        const packageGroups: { [key: string]: { count: number; name: string; price: number } } = {};
        
        // Process adults
        (formData?.adults || []).forEach((adult: any) => {
          if (adult.recommendation && adult.recommendation.trim() !== "") {
            const packageCode = adult.recommendation;
            
            // Get package details
            let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
            if (selectedActivities.surfLessons && selectedActivities.surfTours) {
              typeCriteria = ['L', 'ST'];
            } else if (selectedActivities.surfTours) {
              typeCriteria = 'ST';
            } else {
              typeCriteria = 'L';
            }

            const personCriteria = {
              level: (adult.surfLevel === 'beginner' ? 'B' : adult.surfLevel === 'intermediate' ? 'I' : 'A'),
              age: 'Ad' as 'A' | 'Ad' | 'Ch1' | 'Ch2',
              type: typeCriteria,
              bookingDate: reservationDays[0]?.date,
              personCount: 1, // Each adult is counted as 1 person
            };

            const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
            const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === adult.recommendation);
            
            if (selectedPackage) {
              if (!packageGroups[packageCode]) {
                packageGroups[packageCode] = {
                  count: 0,
                  name: selectedPackage.name,
                  price: typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price))
                };
              }
              packageGroups[packageCode].count += 1;
            } else {
              // Fallback: create basic package entry if lookup fails
              if (!packageGroups[packageCode]) {
                packageGroups[packageCode] = {
                  count: 0,
                  name: `Package ${packageCode}`,
                  price: 0
                };
              }
              packageGroups[packageCode].count += 1;
            }
          }
        });
        
        // Process children
        (formData?.children || []).forEach((child: any) => {
          if (child.recommendation && child.recommendation.trim() !== "") {
            const packageCode = child.recommendation;
            
            // Get package details
            let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
            if (selectedActivities.surfLessons && selectedActivities.surfTours) {
              typeCriteria = ['L', 'ST'];
            } else if (selectedActivities.surfTours) {
              typeCriteria = 'ST';
            } else {
              typeCriteria = 'L';
            }

            const personCriteria = {
              level: (child.surfLevel === 'beginner' ? 'B' : child.surfLevel === 'intermediate' ? 'I' : 'A'),
              age: (child.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
              type: typeCriteria,
              bookingDate: reservationDays[0]?.date,
              personCount: 1, // Each child is counted as 1 person
            };

            const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
            const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === child.recommendation);
            
            if (selectedPackage) {
              if (!packageGroups[packageCode]) {
                packageGroups[packageCode] = {
                  count: 0,
                  name: selectedPackage.name,
                  price: typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price))
                };
              }
              packageGroups[packageCode].count += 1;
            } else {
              // Fallback: create basic package entry if lookup fails
              if (!packageGroups[packageCode]) {
                packageGroups[packageCode] = {
                  count: 0,
                  name: `Package ${packageCode}`,
                  price: 0
                };
              }
              packageGroups[packageCode].count += 1;
            }
          }
        });
        
        // Convert package groups to booking details
        groupedBookingDetails = Object.entries(packageGroups).map(([packageCode, group]: [string, any]) => ({
          code: packageCode,
          name: group.name,
          price: group.price,
          qty: group.count,
          gross: group.price * group.count,
          disc: 0,
          discp: 0,
          nett: group.price * group.count,
        }));
      }

      // Calculate gross amount from grouped details to avoid double counting
      const grossFromGrouped = groupedBookingDetails.reduce((sum: number, pkg: any) => sum + (pkg.gross || 0), 0);
      const finalGross = Number(grossFromGrouped);
      const finalNett = voucherData?.net_amount ? Number(voucherData.net_amount) : finalGross;
      
      // Update gross and nett with calculated values for both modes
      gross = finalGross;
      nett = finalNett;

      // Calculate total quantity from selected packages
      const totalQty = groupedBookingDetails.reduce((sum: number, pkg: any) => sum + (pkg.qty || 0), 0);

      // Apply proportional voucher discount to booking details if voucher exists
      let processedBookingDetails = groupedBookingDetails;
      if (voucherData && voucherData.discount_amount > 0) {
        const totalDiscount = finalGross - finalNett;
        
        processedBookingDetails = groupedBookingDetails.map((detail: any) => {
          const itemGross = Number(detail.gross || 0);
          const proportion = finalGross > 0 ? itemGross / finalGross : 0;
          const proportionalDiscount = totalDiscount * proportion;
          const itemNett = itemGross - proportionalDiscount;
          const itemDisc = proportionalDiscount;
          const itemDiscp = finalGross > 0 ? (proportionalDiscount / itemGross) * 100 : 0;
          
          return {
            ...detail,
            gross: itemGross,
            nett: Math.max(0, itemNett), // Ensure nett is not negative
            disc: itemDisc,
            discp: itemDiscp,
          };
        });
      }

      // Filter out any booking details with missing required fields and ensure all fields are properly typed
      const booking_details = processedBookingDetails
        .filter((detail: any) => detail.code && detail.name && detail.name.trim() !== "")
        .map((detail: any) => ({
          ...detail,
          gross: Number(detail.gross || 0),
          nett: Number(detail.nett || 0),
          price: Number(detail.price || 0),
          qty: Number(detail.qty || 0),
          disc: Number(detail.disc || 0),
          discp: Number(detail.discp || 0),
          date: reservationDays?.[0]?.date || "",
          time: reservationDays?.[0]?.time || "",
        }));

      // Validate that we have valid booking details
      if (booking_details.length === 0) {
        showAlert("No valid packages selected. Please select at least one package.", 'error');
        return;
      }
  
      // Create booking_persons based on mode (customer vs agent)
      let booking_persons: Array<{
        code: string;
        name: string;
        medical: string[];
        medical_other: string;
      }> = [];

      if (showAgentSection && paxGroups.length > 0) {
        // Agent mode: Create individual person entries for each pax in each group
        paxGroups.forEach((group) => {
          if (group.recommendedPackage && group.count > 0) {
            // Generate criteria to get package details
            let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
            if (selectedActivities.surfLessons && selectedActivities.surfTours) {
              typeCriteria = ['L', 'ST'];
            } else if (selectedActivities.surfTours) {
              typeCriteria = 'ST';
            } else {
              typeCriteria = 'L';
            }

            const groupCriteria = {
              level: (group.surfLevel === 'beginner' ? 'B' : group.surfLevel === 'intermediate' ? 'I' : 'A'),
              age: (group.ageGroup === 'adult' ? 'Ad' : group.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
              type: typeCriteria,
              bookingDate: reservationDays[0]?.date,
              personCount: group.count,
            };

            const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
            const selectedPackage = recommendedPackages.find(pkg => pkg.code === group.recommendedPackage);
            const packageCode = selectedPackage?.code || group.recommendedPackage;
            
            // Create individual person entries for each pax in this group
            for (let i = 1; i <= group.count; i++) {
              const personName = `${group.ageGroup === 'adult' ? 'Adult' : group.ageGroup === 'teen' ? 'Teen' : 'Child'} ${i} (${group.surfLevel})`;
              
              booking_persons.push({
                code: packageCode,
                name: personName,
                medical: ["no_medical"],
                medical_other: ""
              });
            }
          }
        });
      } else {
        // Customer mode: Use recommendation field directly from formData
        booking_persons = [
          ...(formData?.adults || []).map((a: any) => {
            return { 
              code: a.recommendation || "",
              name: a.name, 
              medical: a.medical || ["no_medical"],
              medical_other: a.medical_other || ""
            };
          }),
          ...(formData?.children || []).map((c: any) => {
            return { 
              code: c.recommendation || "",
              name: c.name, 
              medical: c.medical || ["no_medical"],
              medical_other: c.medical_other || ""
            };
          }),
        ];
      }
  
      // Calculate payment amounts based on agent and split payment
      let paid_b = 0, paid_c = 0, paid_o = 0, paid_s = 0;
      let bank_code_b = "";

      if (selectedAgent) {
        // Agent payment mode - use splitPayments
        const bankPayment = splitPayments.find(p => p.method === 'bank');
        const creditPayment = splitPayments.find(p => p.method === 'credit');
        const onsitePayment = splitPayments.find(p => p.method === 'onsite');
        const saldoPayment = splitPayments.find(p => p.method === 'saldo');
        
        paid_b = bankPayment?.amount || 0;
        paid_c = creditPayment?.amount || 0;
        paid_o = onsitePayment?.amount || 0;
        paid_s = saldoPayment?.amount || 0;
        
        if (paid_b > 0) {
          bank_code_b = "BCA"; // Default bank
        }
        
        // Validate agent payment total
        const totalAgentAmount = splitPayments.reduce((sum, payment) => sum + (payment.amount || 0), 0);
        if (Math.abs(totalAgentAmount - nett) > 0.01) {
          showAlert(`Total payment amount (${getDisplayPrice(totalAgentAmount)}) must equal booking amount (${getDisplayPrice(nett)})`, 'error');
          return;
        }
        
        // Validate saldo limit
        const maxSaldo = selectedAgent?.saldo ? parseFloat(selectedAgent.saldo) : 0;
        if (paid_s > maxSaldo) {
          showAlert(`Saldo payment (${getDisplayPrice(paid_s)}) exceeds available balance (${getDisplayPrice(maxSaldo)})`, 'error');
          return;
        }
      } else {
        // Single payment mode
        switch (paymentMethod) {
          case 'bank':
            paid_b = nett;
            bank_code_b = "BCA";
            break;
          case 'credit':
            paid_c = nett;
            break;
          case 'onsite':
            paid_o = nett;
            break;
          case 'saldo':
            paid_s = nett;
            break;
          case 'midtrans':
            paid_b = nett;
            bank_code_b = "MIDTRANS";
            break;
          case 'paypal':
            paid_b = nett;
            bank_code_b = "PAYPAL";
            break;
        }
      }

      console.log('🚀 Building booking payload...');
      console.log('🚀 selectedAgent:', selectedAgent);
      console.log('🚀 agentCommission state:', agentCommission);
      console.log('🚀 showAgentSection:', showAgentSection);
      
      const bookingPayload = {
        booking_number: bookingNumber,
        date: reservationDays?.[0]?.date || "",
        notes: "", // Empty notes as requested
        type: "O",
        agent: selectedAgent ? "Y" : "N",
        agent_code: selectedAgent?.code || "",
        comm: selectedAgent ? Number(agentCommission) : 0,
        member_code: null,
        member_type: "I",
        qty: Number(totalQty),
        gross: Number(finalGross),
        disc: Number(disc),
        discp: Number(discp),
        nett: Number(finalNett),
        paid_b,
        paid_c,
        paid_o,
        paid_s,
        bank_code_b,
        promo_no,
        voucher_code,
        save_as_draft: false,
        booking_details,
        booking_persons,
        email: showAgentSection ? (selectedAgent?.email || "") : (formData2?.email || ""),
        phone: showAgentSection ? (selectedAgent?.phone || "") : (formData2?.mobilePhone || ""),
        country: formData2?.country || "", // ✅ Added country from Customer Information
        room_number: formData2?.roomNumber || "", // ✅ Added room number from Customer Information
        hotel_transport_requested: formData2?.hotelTransfer === "Yes" ? "Y" : "N", // ✅ Added hotel transport request
        hotel_name: formData2?.hotel || "", // ✅ Added hotel name
        booking_nameinhotel: formData2?.bookingName || "", // ✅ Added booking name in hotel
      };

      console.log('🚀 Final booking payload comm:', bookingPayload.comm);
      console.log('🚀 Full booking payload:', bookingPayload);

      // Determine which endpoint to use based on payment method
      const isAgentPayment = selectedAgent && (paymentMethod === 'bank' || paymentMethod === 'credit' || paymentMethod === 'onsite' || paymentMethod === 'saldo');
      
      // Skip API call for PayPal - it will be handled in PayPal confirmation
      if (paymentMethod === 'paypal') {
        // PayPal payment - show confirmation modal first
        const customerPackages = getCustomerPackages();
        console.log('🔍 Customer packages for PayPal confirmation:', customerPackages);
        console.log('🔍 showAgentSection:', showAgentSection);
        console.log('🔍 selectedPackages:', selectedPackages);
        
        // Use fixed exchange rate: 1 USD = 13000 IDR
        const exchangeRate = USD_FIXED_RATE;
        
        // Convert packages to USD for PayPal confirmation
        const convertPackagesToUSD = (packages: Array<{ name: string; price: number }>) => {
          return packages.map(pkg => ({
            name: pkg.name,
            price: parseFloat((pkg.price * exchangeRate).toFixed(2))
          }));
        };

        // Group identical packages into one line (sum price, show qty in name)
        const groupPackagesByName = (packages: Array<{ name: string; price: number }>) => {
          const groups: { [key: string]: { count: number; total: number } } = {};
          packages.forEach(p => {
            if (!groups[p.name]) {
              groups[p.name] = { count: 0, total: 0 };
            }
            groups[p.name].count += 1;
            groups[p.name].total += p.price;
          });
          return Object.entries(groups).map(([name, g]) => ({
            name: g.count > 1 ? `${name} x${g.count}` : name,
            price: parseFloat(g.total.toFixed(2))
          }));
        };
        
        // Calculate USD total amount
        const netAmount = voucherData?.net_amount || finalTotalAmount;
        const usdTotalAmount = parseFloat((netAmount * exchangeRate).toFixed(2));
        
        const confirmationData = {
          bookingNumber,
          date: reservationDays?.[0]?.date || "N/A",
          duration: duration === "1-day" ? "1 Day" : duration === "2-days" ? "2 Days" : "3 Days",
          participants: `${adultCount > 0 ? `${adultCount} Adult` : ''}${adultCount > 0 && childrenCount > 0 ? ' | ' : ''}${childrenCount > 0 ? `${childrenCount} Children` : ''}`,
          totalAmount: `$${usdTotalAmount.toFixed(2)}`, // ✅ Show USD total amount
          customerName: formData2?.fullName || "N/A",
          hotel: formData2?.hotel || "N/A",
          hotelAddress: formData2?.hotelAddress || "N/A",
          bookingName: formData2?.bookingName || "N/A",
          hotelTransfer: formData2?.hotelTransfer || "N/A",
          packages: showAgentSection 
            ? groupPackagesByName(
                convertPackagesToUSD(
                  Object.values(selectedPackages || {}).map((pkg: any) => ({
                    name: pkg.pkg?.title || "N/A",
                    price: pkg.pkg?.price || 0
                  }))
                )
              )
            : groupPackagesByName(
                convertPackagesToUSD(customerPackages)
              ) // ✅ Use customer packages for customer mode
        };

        setBookingConfirmationData(confirmationData);
        setShowBookingConfirmationModal(true);
        return; // Skip the general booking API call for PayPal
      }
      
      // Use appropriate endpoint based on payment method
      let endpoint;
      if (isAgentPayment) {
        endpoint = getBookingUrl(); // BCOS (Agent payments) use /booking
      } else if (paymentMethod === 'midtrans') {
        endpoint = getWithoutPaymentUrl(); // Midtrans use /booking/without-payment
      } else {
        endpoint = getWithoutPaymentUrl(); // Draft/without payment
      }
      
      const response = await fetch(endpoint, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${API_CONFIG.TOKEN}`,
        },
        body: JSON.stringify(bookingPayload),
        credentials: "omit",
      });
      
      const data = await response.json();
        
      if (data.success) {
        if (isAgentPayment) {
          // Agent payment - show success modal
          const bookingDetails = {
            date: reservationDays?.[0]?.date || "N/A",
            duration: duration === "1-day" ? "1 Day" : duration === "2-days" ? "2 Days" : "3 Days",
            participants: showAgentSection ? `${totalPax} Pax` : `${adultCount > 0 ? `${adultCount} Adult` : ''}${adultCount > 0 && childrenCount > 0 ? ' | ' : ''}${childrenCount > 0 ? `${childrenCount} Children` : ''}`,
            totalAmount: getDisplayPrice(voucherData?.net_amount || finalTotalAmount),
            customerName: formData2?.fullName || "N/A",
            agentName: selectedAgent?.name || "N/A",
            agentEmail: selectedAgent?.email || "",
            isAgentMode: showAgentSection
          };

          setSuccessBookingData({
            bookingNumber,
            bookingDetails
          });
          setShowSuccessModal(true);
        } else {
          // Handle different payment methods
          if (paymentMethod === 'midtrans') {
            // Midtrans payment - handle snap token
            console.log('Midtrans response data:', data);
            
            if (data.data && data.data.snap_token) {
              setSnapToken(data.data.snap_token);
              setShowMidtransModal(true);
              
              // Set success data for after payment completion
              const bookingDetails = {
                date: reservationDays?.[0]?.date || "N/A",
                duration: duration === "1-day" ? "1 Day" : duration === "2-days" ? "2 Days" : "3 Days",
                participants: showAgentSection ? `${totalPax} Pax` : `${adultCount > 0 ? `${adultCount} Adult` : ''}${adultCount > 0 && childrenCount > 0 ? ' | ' : ''}${childrenCount > 0 ? `${childrenCount} Children` : ''}`,
                totalAmount: getDisplayPrice(voucherData?.net_amount || finalTotalAmount),
                customerName: formData2?.fullName || "N/A",
                agentName: selectedAgent?.name || "N/A",
                agentEmail: selectedAgent?.email || "",
                isAgentMode: showAgentSection
              };

              setSuccessBookingData({
                bookingNumber,
                bookingDetails
              });
              // Don't show success modal yet - wait for payment completion
            } else {
              // Enhanced error handling for missing snap token
              console.error('Midtrans token error:', {
                hasData: !!data.data,
                dataKeys: data.data ? Object.keys(data.data) : 'no data',
                fullResponse: data,
                bookingPayload: bookingPayload
              });
              
              // Show detailed error message
              const errorDetails = data.data ? 
                `Response data: ${JSON.stringify(data.data)}` : 
                'No response data received';
              
              showAlert(
                `Booking created but payment token not received.\n\nError details: ${errorDetails}\n\nPlease try again or contact support.`, 
                'error'
              );
              
              // Reset payment states to allow retry
              setSnapToken(null);
              setShowMidtransModal(false);
            }
          }
        }
      } else {
        // Handle different types of errors from API response
        if (data.errors) {
          // Handle validation errors - format them nicely
          const errorMessages = Object.values(data.errors).flat().join(', ');
          showAlert(`Validation error: ${errorMessages}`, 'error');
        } else {
          showAlert(data.message || "Booking failed", 'error');
        }
      }
    } catch (err: any) {
      showAlert(err?.message || "An error occurred while processing the booking.", 'error');
      console.error("Booking error:", err);
    } finally {
      setIsPaymentLoading(false);
    }
  }

  // Load formData2 and other states from localStorage - DISABLED
  // useEffect(() => {
  //   if (typeof window !== "undefined" && isClient) {
  //     const raw = localStorage.getItem("surf-booking-data");
  //     if (raw) {
  //       try {
  //           const saved = JSON.parse(raw);
  //           setFormData2(saved.formData2 || {
  //             title: "Mr",
  //             fullName: "",
  //             email: "",
  //             mobilePhone: "",
  //             hotel: "",
  //             hotelAddress: "",
  //             bookingName: "",
  //             dateOfArrival: "",
  //             country: "Indonesia",
  //             nationality: "",
  //             hotelTransfer: "No",
  //             notes: "",
  //           });
  //           setSurfExperience(saved.surfExperience || dummySurfExperience);
  //           setAgreeTerms(saved.agreeTerms || false);
  //           
  //           // Auto-fill customer info if available
  //           if (saved.customerInfo) {
  //             setFormData2(prev => ({
  //               ...prev,
  //               fullName: saved.customerInfo.fullName || prev.fullName,
  //               email: saved.customerInfo.email || prev.email,
  //               mobilePhone: saved.customerInfo.mobile || prev.mobilePhone,
  //               hotel: saved.customerInfo.hotel || prev.hotel,
  //               hotelAddress: saved.customerInfo.hotelAddress || prev.hotelAddress,
  //               bookingName: saved.customerInfo.bookingName || prev.bookingName,
  //               dateOfArrival: saved.customerInfo.dateOfArrival || prev.dateOfArrival,
  //               country: saved.customerInfo.country || prev.country,
  //               nationality: saved.customerInfo.nationality || prev.nationality,
  //               notes: saved.customerInfo.notes || prev.notes,
  //             }));
  //           }
  //         } catch (error) {
  //           console.error('Error parsing localStorage data for formData2:', error);
  //         }
  //       }
  //     }
  //   }, [isClient]);

  // Save formData2 and other states to localStorage - DISABLED
  // useEffect(() => {
  //   if (typeof window !== "undefined" && isClient) {
  //     try {
  //       const existing = localStorage.getItem("surf-booking-data");
  //       const existingData = existing ? JSON.parse(existing) : {};
  //       localStorage.setItem("surf-booking-data", JSON.stringify({
  //         ...existingData,
  //         formData2,
  //         surfExperience,
  //         agreeTerms,
  //       }));
  //     } catch (error) {
  //       console.error('Error saving to localStorage:', error);
  //     }
  //   }
  // }, [formData2, surfExperience, agreeTerms, isClient]);

  const bookingData2 = {
    dateOfLesson: "19-07-2025",
    duration: "1 Day",
    pax: "1 Adult | 1 Children",
  }

  const customerInfo = {
    fullName: showAgentSection ? (selectedAgent?.name || "") : formData2.fullName,
    email: showAgentSection ? (selectedAgent?.email || "") : formData2.email,
    mobile: showAgentSection ? (selectedAgent?.phone || "") : formData2.mobilePhone,
    hotel: formData2.hotel,
    hotelAddress: formData2.hotelAddress,
    bookingName: formData2.bookingName,
    dateOfArrival: formData2.dateOfArrival,
    country: formData2.country,
    roomNumber: formData2.roomNumber,
    hotelTransfer: formData2.hotelTransfer, // ✅ Added hotel transfer
    notes: formData2.notes,
  }

  async function applyVoucher() {
    setPromoMessage("");
    setPromoSuccess(false);

    try {
      const response = await fetch(getApplyVoucherUrl(), {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${API_CONFIG.TOKEN}`,
        },
        body: JSON.stringify({
          voucher_code: promoCode,
          gross_amount: finalTotalAmount,
        }),
        credentials: "omit",
      });

      const data = await response.json();

      if (!response.ok || !data.success) {
        // Handle different types of errors from API response
        if (data.errors) {
          // Handle validation errors - format them nicely
          const errorMessages = Object.values(data.errors).flat().join(', ');
          setPromoMessage(`Validation error: ${errorMessages}`);
          setPromoSuccess(false);
          setVoucherData(null);
          showAlert(`Validation error: ${errorMessages}`, 'error');
        } else {
          setPromoMessage(data.message || "Voucher not valid.");
          setPromoSuccess(false);
          setVoucherData(null);
          showAlert(data.message || "Voucher not valid.", 'error');
        }
        return;
      }

      setPromoMessage(data.data?.message || data.message || "Voucher applied successfully!");
      setPromoSuccess(true);
      setVoucherData(data.data);
      showAlert("Voucher applied successfully!", 'success');
    } catch (err) {
      setPromoMessage("Failed to apply voucher. Please try again.");
      setPromoSuccess(false);
      setVoucherData(null);
      showAlert("Failed to apply voucher. Please try again.", 'error');
    }
  }

  async function generateBookingNumber() {
    try {
      const response = await fetch(getGenerateNumberUrl(), {
        method: "GET",
        headers: {
          "Authorization": `Bearer ${API_CONFIG.TOKEN}`,
        },
        credentials: "omit",
      });
      const data = await response.json();
      if (data.success) {
        return data.data.booking_number;
      } else {
        throw new Error(data.message || "Failed to generate booking number");
      }
    } catch (err) {
      throw new Error("Failed to generate booking number");
    }
  }



  // Function to reset all form data
  function resetAllData() {
    // Clear localStorage (disabled - no longer saving to localStorage)
    // if (typeof window !== "undefined") {
    //   localStorage.removeItem("surf-booking-data");
    // }

    // Reset all state to initial values
    setFormData({
      adults: [],
      children: [],
    });
    setSelectedActivities({
      surfLessons: true,
      surfTours: false,
    });
    setDuration("1-day");
    setReservationDays([{ date: "", time: "" }]);
    setAdultCount(0);
    setChildrenCount(0);
    setPackageList([]);
    setSelectedPackages({});
    setSchedules([]);
    setLoadingSchedules(false);

    setAgentCode('');
    setSelectedAgent(null);
    setAgentCommission(0);
    setSplitPayments([]);
    setAgentPaymentAmounts({});
    setCurrency("IDR");
    setSelectedActivityType(null);
    setAlerts([]);
    setShowSuccessModal(false);
    setSuccessBookingData(null);
    setFormData2({
      title: "Mr",
      fullName: "",
      email: "",
      mobilePhone: "",
      hotel: "",
      hotelAddress: "",
      bookingName: "",
      dateOfArrival: "",
      country: "Indonesia",
      hotelTransfer: "No",
      roomNumber: "",
      notes: "",
    });
    setAgreeAndContinue(false);
    setSurfExperience(dummySurfExperience);
    setErrors({});
    setPaymentMethod("bank");
    setAgreeTerms(false);
    setPromoCode("");
    setPromoMessage("");
    setPromoSuccess(false);
    setVoucherData(null);
    
    // Reset payment-related states
    setSnapToken(null);
    setShowMidtransModal(false);
    setShowBookingConfirmationModal(false);
    setBookingConfirmationData(null);
    setIsProcessingPayPal(false);
    setPaypalApprovalUrl(null);
    setPaypalBookingNumber(null);
    setPaypalPaymentStatus(null);
    setIsCheckingPayPalStatus(false);
    
    // Reset payment status modals
    setShowPaymentFailedModal(false);
    setShowPaymentSettlementModal(false);
    setShowPaymentPendingModal(false);
    setShowPaymentExpiredModal(false);
    setShowPaymentDeniedModal(false);
  }

  // Function to force reset everything and reload page
  function forceResetAndReload() {
    resetAllData();
    // Clear any remaining localStorage
    if (typeof window !== "undefined") {
      localStorage.clear();
      sessionStorage.clear();
    }
    // Reload the page to ensure complete fresh start
    window.location.reload();
  }

  // Clean up selected packages when participant count changes
  useEffect(() => {
    if (typeof window !== "undefined" && isClient) {
      const allParticipantNames = [
        ...formData.adults.map(a => a.name),
        ...formData.children.map(c => c.name)
      ].filter(name => name.trim() !== "");

      // If no participants, clear all packages
      if (allParticipantNames.length === 0) {
        setSelectedPackages({});
        return;
      }

      // Remove packages that reference non-existent participants
      const cleanedPackages = { ...selectedPackages };
      Object.keys(cleanedPackages).forEach(key => {
        const packageData = cleanedPackages[key];
        if (packageData.people) {
          // Filter out people that no longer exist
          const validPeople = packageData.people.filter((person: string) => 
            allParticipantNames.includes(person)
          );
          
          if (validPeople.length === 0) {
            // Remove package if no valid people remain
            delete cleanedPackages[key];
          } else if (validPeople.length !== packageData.people.length) {
            // Update package with only valid people
            cleanedPackages[key] = {
              ...packageData,
              people: validPeople
            };
          }
        }
      });

      // Only update if there are changes
      if (JSON.stringify(cleanedPackages) !== JSON.stringify(selectedPackages)) {
        setSelectedPackages(cleanedPackages);
      }
    }
  }, [adultCount, childrenCount, formData.adults, formData.children, isClient]);

  // Clean up packages when participant names change
  useEffect(() => {
    if (typeof window !== "undefined" && isClient && Object.keys(selectedPackages).length > 0) {
      const allParticipantNames = [
        ...formData.adults.map(a => a.name),
        ...formData.children.map(c => c.name)
      ].filter(name => name.trim() !== "");

      // Check if any packages reference non-existent participants
      let hasChanges = false;
      const cleanedPackages = { ...selectedPackages };
      
      Object.keys(cleanedPackages).forEach(key => {
        const packageData = cleanedPackages[key];
        if (packageData.people) {
          const validPeople = packageData.people.filter((person: string) => 
            allParticipantNames.includes(person)
          );
          
          if (validPeople.length === 0) {
            // Remove package if no valid people remain
            delete cleanedPackages[key];
            hasChanges = true;
          } else if (validPeople.length !== packageData.people.length) {
            // Update package with only valid people
            cleanedPackages[key] = {
              ...packageData,
              people: validPeople
            };
            hasChanges = true;
          }
        }
      });

      // Only update if there are changes
      if (hasChanges) {
        setSelectedPackages(cleanedPackages);
      }
    }
  }, [formData.adults, formData.children, isClient]);

  // Add Midtrans Snap script loading
  useEffect(() => {
    if (isClient) {
      // Load Midtrans Snap script
      const script = document.createElement('script');
      script.src = `${MIDTRANS_CONFIG.BASE_URL}/snap/snap.js`;
      script.setAttribute('data-client-key', MIDTRANS_CONFIG.CLIENT_KEY);
      script.async = true;
      
      script.onload = () => {
        // console.log('Midtrans Snap script loaded successfully');
      };
      
      script.onerror = () => {
        console.error('Failed to load Midtrans Snap script');
      };
      
      document.head.appendChild(script);
      
      return () => {
        // Cleanup script on unmount
        const existingScript = document.querySelector(`script[src="${script.src}"]`);
        if (existingScript) {
          document.head.removeChild(existingScript);
        }
      };
    }
  }, [isClient]);

  // Add function to handle Midtrans payment
  const handleMidtransPayment = (snapToken: string) => {
    if (typeof window !== 'undefined' && (window as any).snap) {
      try {
        (window as any).snap.pay(snapToken, {
          onSuccess: function(result: any) {
            // console.log('Payment success:', result);
            
            // Show success modal directly since Midtrans handles the status
            setShowSuccessModal(true);
            setShowMidtransModal(false);
            setSnapToken(null);
          },
          onPending: function(result: any) {
            // console.log('Payment pending:', result);
            
            // Show pending modal
            setShowPaymentPendingModal(true);
            setShowMidtransModal(false);
            // Keep snapToken for reopening the modal
          },
          onError: function(result: any) {
            // console.log('Payment error:', result);
            
            // Show error modal
            setShowPaymentFailedModal(true);
            setShowMidtransModal(false);
            setSnapToken(null);
          },
          onClose: function() {
            // console.log('Payment modal closed');
            
            // Show error modal
            setShowPaymentFailedModal(true);
            setShowMidtransModal(false);
            setSnapToken(null);
          }
        });
      } catch (error) {
        console.error('Error initializing Midtrans payment:', error);
        
        // Show error modal
        setShowPaymentFailedModal(true);
        setShowMidtransModal(false);
        setSnapToken(null);
      }
    } else {
      console.error('Midtrans Snap not loaded');
      showAlert('Payment system not ready. Please refresh the page and try again.', 'error');
    }
  };

  // Add function to handle PayPal payment confirmation
  const handlePayPalConfirmation = async () => {
    if (!bookingConfirmationData) return;
    
    setIsProcessingPayPal(true);
    
    try {
      // ✅ Generate fresh booking number for PayPal to avoid duplicates
      const freshBookingNumber = await generateBookingNumber();
      
      // Group booking details by package code first to calculate total quantity
      const packageQuantityDetails = Object.values(selectedPackages || {}).reduce((acc: any, pkg: any) => {
        const packageCode = pkg.pkg?.code || "";
        const packagePrice = pkg.pkg?.price || 0;
        const existingPackage = acc.find((item: any) => item.code === packageCode);
        
        if (existingPackage) {
          // If package already exists, sum up the values
          existingPackage.qty += 1;
          existingPackage.gross += packagePrice;
          existingPackage.nett += packagePrice;
        } else {
          // If package doesn't exist, add new entry
          acc.push({
            code: packageCode,
            name: pkg.pkg?.title || "",
            price: packagePrice,
            qty: 1,
            gross: packagePrice,
            disc: 0,
            discp: 0,
            nett: packagePrice,
          });
        }
        
        return acc;
      }, []);

      // Calculate total quantity from selected packages
      let totalQty = 0;
      if (showAgentSection) {
        // Agent mode: use packageQuantityDetails
        totalQty = packageQuantityDetails.reduce((sum: number, pkg: any) => sum + (pkg.qty || 0), 0);
      } else {
        // Customer mode: count from formData recommendations
        const adultCount = (formData?.adults || []).filter(adult => adult.recommendation).length;
        const childCount = (formData?.children || []).filter(child => child.recommendation).length;
        totalQty = adultCount + childCount;
      }
      
      // Recalculate values for PayPal
      const gross = finalTotalAmount || 0;
      const disc = voucherData?.discount_amount ? Math.round(Number(voucherData.discount_amount)) : 0;
      const discp = voucherData?.discount_type === "percentage" ? Number(voucherData.discount_value || 0) : 0;
      const nett = voucherData?.net_amount ? Math.round(Number(voucherData.net_amount)) : gross;
      const voucher_code = voucherData?.voucher_code || "";

      // Use fixed exchange rate: 1 USD = 13000 IDR
      const exchangeRate = USD_FIXED_RATE;
      
      console.log('💳 USD Calculation Debug:');
      console.log('💳 nett (IDR):', nett);
      console.log('💳 exchangeRate:', exchangeRate, '(Fixed Rate: 1 USD = 13000 IDR)');

      // Generate PayPal booking details from customer packages
      const getPayPalBookingDetails = () => {
        if (showAgentSection) {
          // Agent mode: use selectedPackages and group by package code + age type
          return Object.values(selectedPackages || {}).reduce((acc: any, pkg: any) => {
            const packageCode = pkg.pkg?.code || "";
            const packagePrice = pkg.pkg?.price || 0;
            const packageUsdPrice = Math.round(packagePrice * exchangeRate * 100) / 100;
            const ageGroup = (pkg.ageGroup || '').toLowerCase(); // 'adult' | 'teen' | 'child'
            const ageLabel = ageGroup === 'adult' ? 'Adult' : ageGroup === 'teen' ? 'Teen' : 'Child';
            const displayName = `${pkg.pkg?.title || ''} (${ageLabel})`;

            const existingPackage = acc.find((item: any) => item.code === packageCode && item.name === displayName);
            
            if (existingPackage) {
              existingPackage.qty += 1;
              existingPackage.gross += packagePrice;
              existingPackage.nett += packagePrice;
              // Don't add to price_usd - it should be the unit price, not total
              existingPackage.date = reservationDays?.[0]?.date || "";
              existingPackage.time = reservationDays?.[0]?.time || "";
            } else {
              acc.push({
                code: packageCode,
                name: displayName,
                price: packagePrice,
                qty: 1,
                gross: packagePrice,
                disc: 0,
                discp: 0,
                nett: packagePrice,
                price_usd: packageUsdPrice, // Unit price
                date: reservationDays?.[0]?.date || "",
                time: reservationDays?.[0]?.time || "",
              });
            }
            return acc;
          }, []);
        } else {
          // Customer mode: use formData recommendations
          const packageGroups: { [key: string]: any } = {};
          
          // Process adults
          (formData?.adults || []).forEach((adult: any) => {
            if (adult.recommendation) {
              const packageCode = adult.recommendation;
              const ageKey = 'Ad';
              
              const groupKey = `${packageCode}|${ageKey}`;
              if (!packageGroups[groupKey]) {
                // Get package details
                let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                  typeCriteria = ['L', 'ST'];
                } else if (selectedActivities.surfTours) {
                  typeCriteria = 'ST';
                } else {
                  typeCriteria = 'L';
                }

                const personCriteria = {
                  level: (adult.surfLevel === 'beginner' ? 'B' : adult.surfLevel === 'intermediate' ? 'I' : 'A'),
                  age: 'Ad' as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                  type: typeCriteria,
                  bookingDate: reservationDays[0]?.date,
                  personCount: 1,
                };

                const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === packageCode);
                
                if (selectedPackage) {
                  const packagePrice = typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price)) || 0;
                  const packageUsdPrice = Math.round(packagePrice * exchangeRate * 100) / 100;
                  const displayName = `${selectedPackage.name || ''} (Adult)`;
                  
                  packageGroups[groupKey] = {
                    code: packageCode,
                    name: displayName,
                    price: packagePrice,
                    qty: 0,
                    gross: 0,
                    disc: 0,
                    discp: 0,
                    nett: 0,
                    price_usd: 0,
                    date: reservationDays?.[0]?.date || "",
                    time: reservationDays?.[0]?.time || "",
                  };
                }
              }
              
              if (packageGroups[groupKey]) {
                const packagePrice = packageGroups[groupKey].price;
                const packageUsdPrice = Math.round(packagePrice * exchangeRate * 100) / 100;
                
                packageGroups[groupKey].qty += 1;
                packageGroups[groupKey].gross += packagePrice;
                packageGroups[groupKey].nett += packagePrice;
                // Don't add to price_usd - it should be the unit price, not total
                // packageGroups[groupKey].price_usd += packageUsdPrice;
              }
            }
          });
          
          // Process children
          (formData?.children || []).forEach((child: any) => {
            if (child.recommendation) {
              const packageCode = child.recommendation;
              const ageKey = (child.ageGroup === 'teen' ? 'Ch2' : 'Ch1');
              
              const groupKey = `${packageCode}|${ageKey}`;
              if (!packageGroups[groupKey]) {
                // Get package details
                let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                  typeCriteria = ['L', 'ST'];
                } else if (selectedActivities.surfTours) {
                  typeCriteria = 'ST';
                } else {
                  typeCriteria = 'L';
                }

                const personCriteria = {
                  level: (child.surfLevel === 'beginner' ? 'B' : child.surfLevel === 'intermediate' ? 'I' : 'A'),
                  age: (child.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                  type: typeCriteria,
                  bookingDate: reservationDays[0]?.date,
                  personCount: 1,
                };

                const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === packageCode);
                
                if (selectedPackage) {
                  const packagePrice = typeof selectedPackage.price === 'number' ? selectedPackage.price : parseFloat(String(selectedPackage.price)) || 0;
                  const packageUsdPrice = Math.round(packagePrice * exchangeRate * 100) / 100;
                  const displayName = `${selectedPackage.name || ''} (${child.ageGroup === 'teen' ? 'Teen' : 'Child'})`;
                  
                  packageGroups[groupKey] = {
                    code: packageCode,
                    name: displayName,
                    price: packagePrice,
                    qty: 0,
                    gross: 0,
                    disc: 0,
                    discp: 0,
                    nett: 0,
                    price_usd: 0,
                    date: reservationDays?.[0]?.date || "",
                    time: reservationDays?.[0]?.time || "",
                  };
                }
              }
              
              if (packageGroups[groupKey]) {
                const packagePrice = packageGroups[groupKey].price;
                const packageUsdPrice = Math.round(packagePrice * exchangeRate * 100) / 100;
                
                packageGroups[groupKey].qty += 1;
                packageGroups[groupKey].gross += packagePrice;
                packageGroups[groupKey].nett += packagePrice;
                // Don't add to price_usd - it should be the unit price, not total
                // packageGroups[groupKey].price_usd += packageUsdPrice;
              }
            }
          });
          
          // Set price_usd to unit price for each package group
          Object.values(packageGroups).forEach((group: any) => {
            if (group.price > 0) {
              group.price_usd = Math.round(group.price * exchangeRate * 100) / 100;
            }
          });
          
          return Object.values(packageGroups);
        }
      };

      const booking_details = getPayPalBookingDetails();

      // Apply proportional voucher discount to PayPal booking details if voucher exists
      let processedBookingDetails = booking_details;
      if (voucherData && voucherData.discount_amount > 0) {
        const totalDiscount = gross - nett;
        const totalGross = booking_details.reduce((sum: number, item: any) => sum + (item.gross || 0), 0);
        
        processedBookingDetails = booking_details.map((detail: any) => {
          const itemGross = Number(detail.gross || 0);
          const proportion = totalGross > 0 ? itemGross / totalGross : 0;
          const proportionalDiscount = totalDiscount * proportion;
          const itemNett = itemGross - proportionalDiscount;
          const itemDisc = proportionalDiscount;
          const itemDiscp = itemGross > 0 ? (proportionalDiscount / itemGross) * 100 : 0;
          
          // Calculate price_usd as unit price (per item) after discount using fixed rate
          const itemQty = detail.qty || 1;
          const itemNettPerUnit = Math.max(0, itemNett) / itemQty; // Price per unit after discount
          const priceUsdPerUnit = Math.round(itemNettPerUnit * exchangeRate * 100) / 100;
          
          return {
            ...detail,
            gross: itemGross,
            nett: Math.max(0, itemNett), // Total nett after discount
            disc: itemDisc,
            discp: itemDiscp,
            price_usd: priceUsdPerUnit, // Unit price in USD after discount
          };
        });
      }
      
      // Calculate usd_amount from total of all items (after discount) using fixed rate
      const totalItemNett = processedBookingDetails.reduce((sum: number, item: any) => {
        return sum + (item.nett || 0);
      }, 0);
      const usd_amount = Math.round(totalItemNett * exchangeRate * 100) / 100;
      
      console.log('💳 USD Calculation Debug:');
      console.log('💳 totalItemNett (IDR):', totalItemNett);
      console.log('💳 exchangeRate:', exchangeRate, '(Fixed Rate: 1 USD = 13000 IDR)');
      console.log('💳 calculated usd_amount:', usd_amount);

      const booking_persons = [
        ...(formData?.adults || []).map((a: any) => {
          if (showAgentSection) {
            // Agent mode: use selectedPackages
            const selectedPackage = Object.values(selectedPackages || {}).find((pkg: any) => 
              pkg.people.includes(a.name)
            );
            return { 
              code: selectedPackage?.pkg?.code || "",
              name: a.name, 
              medical: a.medical || ["no_medical"],
              medical_other: a.medical_other || ""
            };
          } else {
            // Customer mode: use recommendation directly
            return { 
              code: a.recommendation || "",
              name: a.name, 
              medical: a.medical || ["no_medical"],
              medical_other: a.medical_other || ""
            };
          }
        }),
        ...(formData?.children || []).map((c: any) => {
          if (showAgentSection) {
            // Agent mode: use selectedPackages
            const selectedPackage = Object.values(selectedPackages || {}).find((pkg: any) => 
              pkg.people.includes(c.name)
            );
            return { 
              code: selectedPackage?.pkg?.code || "",
              name: c.name, 
              medical: c.medical || ["no_medical"],
              medical_other: c.medical_other || ""
            };
          } else {
            // Customer mode: use recommendation directly
            return { 
              code: c.recommendation || "",
              name: c.name, 
              medical: c.medical || ["no_medical"],
              medical_other: c.medical_other || ""
            };
          }
        }),
      ];

      console.log('💳 Building PayPal booking payload...');
      console.log('💳 selectedAgent for PayPal:', selectedAgent);
      console.log('💳 agentCommission for PayPal:', agentCommission);
      console.log('💳 booking_details length:', booking_details.length);
      console.log('💳 booking_persons length:', booking_persons.length);
      console.log('💳 exchangeRate:', exchangeRate, '(Fixed Rate: 1 USD = 13000 IDR)');
      console.log('💳 usd_amount:', usd_amount);
      
      // Convert all monetary values to USD for PayPal API using fixed rate
      const grossUsd = parseFloat((gross * exchangeRate).toFixed(2));
      const discUsd = parseFloat((disc * exchangeRate).toFixed(2));
      const nettUsd = parseFloat((nett * exchangeRate).toFixed(2));
      
      console.log('💳 Currency Conversion for PayPal:');
      console.log('💳 Original IDR values (stored in DB) - gross:', gross, 'disc:', disc, 'nett:', nett);
      console.log('💳 Converted to USD for PayPal API - gross:', grossUsd, 'disc:', discUsd, 'nett:', nettUsd);
      console.log('💳 usd_amount field stores IDR:', nett, '(for database consistency)');
      console.log('💳 Exchange rate used:', exchangeRate, '(Fixed Rate: 1 USD = 13000 IDR)');
      
      const paypalBookingData = {
        booking_no: freshBookingNumber, // ✅ Use fresh booking number instead of cached one
        date: reservationDays?.[0]?.date || new Date().toISOString().split('T')[0],
        notes: "", // Empty notes as requested
        type: 'O', // Walk-in
        agent: selectedAgent ? "Y" : "N", // ✅ Include agent if selected
        agent_code: selectedAgent?.code || "", // ✅ Agent code if agent is selected
        comm: selectedAgent ? agentCommission : 0, // ✅ Commission if agent is selected
        member_type: 'I', // Individual
        qty: totalQty,
        gross: grossUsd, // ✅ Convert IDR to USD for PayPal API
        disc: discUsd, // ✅ Convert IDR to USD for PayPal API
        discp: discp, // Percentage stays the same
        nett: nettUsd, // ✅ Convert IDR to USD for PayPal API
        usd_amount: usd_amount, // ✅ USD amount calculated from total items using fixed rate
        booking_details: processedBookingDetails, // Use processed booking details with discount applied
        booking_persons: booking_persons,
        voucher_code: voucher_code,
        currency: 'USD', // USD for PayPal processing
        email: formData2?.email || "", // ✅ Always use customer email for PayPal payments
        phone: formData2?.mobilePhone || "", // ✅ Always use customer phone for PayPal payments
        country: formData2?.country || "", // ✅ Added country from Customer Information
        room_number: formData2?.roomNumber || "", // ✅ Added room number from Customer Information
        hotel_transport_requested: formData2?.hotelTransfer === "Yes" ? "Y" : "N", // ✅ Added hotel transport request
        hotel_name: formData2?.hotel || "", // ✅ Added hotel name
        booking_nameinhotel: formData2?.bookingName || "" // ✅ Added booking name in hotel
      };

      console.log('💳 Final PayPal payload comm:', paypalBookingData.comm);
      console.log('💳 Full PayPal payload:', paypalBookingData);
      
      // Detailed logging for PayPal validation
      console.log('💳 PayPal Validation Details:');
      console.log('💳 usd_amount:', paypalBookingData.usd_amount);
      console.log('💳 booking_details:', paypalBookingData.booking_details);
      
      // Calculate expected item total for validation
      let expectedItemTotal = 0;
      paypalBookingData.booking_details.forEach((item: any, index: number) => {
        const itemTotal = item.price_usd * item.qty;
        expectedItemTotal += itemTotal;
        console.log(`💳 Item ${index + 1}: ${item.name} - Unit: $${item.price_usd} x ${item.qty} = $${itemTotal.toFixed(2)}`);
      });
      console.log('💳 Expected item_total:', expectedItemTotal.toFixed(2));
      console.log('💳 Actual usd_amount:', paypalBookingData.usd_amount);
      console.log('💳 Difference:', Math.abs(expectedItemTotal - paypalBookingData.usd_amount).toFixed(2));
      
      // Fix usd_amount if it doesn't match item total (PayPal validation requirement)
      if (Math.abs(expectedItemTotal - paypalBookingData.usd_amount) > 0.01) {
        console.log('💳 Fixing usd_amount to match item total for PayPal validation');
        paypalBookingData.usd_amount = Math.round(expectedItemTotal * 100) / 100;
        paypalBookingData.nett = paypalBookingData.usd_amount; // Also update nett to match
        console.log('💳 Corrected usd_amount:', paypalBookingData.usd_amount);
        console.log('💳 Corrected nett:', paypalBookingData.nett);
      }
      
      // Validate required fields
      if (!paypalBookingData.booking_no) {
        throw new Error('Missing booking number');
      }
      if (!paypalBookingData.email) {
        throw new Error('Missing customer email');
      }
      if (paypalBookingData.booking_details.length === 0) {
        throw new Error('No booking details found');
      }
      if (paypalBookingData.booking_persons.length === 0) {
        throw new Error('No booking persons found');
      }
      if (paypalBookingData.usd_amount <= 0) {
        throw new Error('Invalid USD amount');
      }

      const apiUrl = getApiUrl('/api/v1/paypal/create-booking-payment');
      console.log('💳 PayPal API URL:', apiUrl);
      console.log('💳 API_CONFIG.BASE_URL:', API_CONFIG.BASE_URL);
      console.log('💳 API_CONFIG.TOKEN:', API_CONFIG.TOKEN ? 'Present' : 'Missing');
      
      const paypalResponse = await fetch(apiUrl, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(paypalBookingData)
      });

      console.log('💳 PayPal API Response Status:', paypalResponse.status);
      console.log('💳 PayPal API Response URL:', paypalResponse.url);
      console.log('💳 PayPal API Response Headers:', Object.fromEntries(paypalResponse.headers.entries()));
      
      if (!paypalResponse.ok) {
        const errorText = await paypalResponse.text();
        console.error('💳 PayPal API Error Response:', errorText);
        throw new Error(`PayPal API Error: ${paypalResponse.status} - ${errorText}`);
      }
      
      const paypalData = await paypalResponse.json();
      console.log('💳 PayPal API Response Data:', paypalData);

              if (paypalData.success) {
          // Keep confirmation modal open for status checking
          
          // ✅ Open in-app PayPal modal with approval URL; user will choose to continue
          if (paypalData.data?.approval_url) {
            // Show success message
            showAlert('PayPal payment created. Opening PayPal in new tab...', 'success');
            // Store approval URL and booking number, then open PayPal in new tab
                  setPaypalApprovalUrl(paypalData.data.approval_url || null);
                  setPaypalBookingNumber(paypalData.data.booking_no || freshBookingNumber);
                  
                  // Open PayPal payment in new tab (keep modal open)
                  if (paypalData.data.approval_url) {
                    const paypalWindow = window.open(paypalData.data.approval_url, '_blank', 'noopener,noreferrer');
                    
                    // Start monitoring for when user returns from PayPal
                    const startPayPalMonitoring = () => {
                      console.log('🔄 Starting PayPal payment monitoring...');
                      setIsMonitoringPayPal(true);
                      const finalizedKey = `paypalFinalized:${paypalData.data.booking_no}`;
                      let isFinalized = false;
                      try {
                        if (typeof window !== 'undefined') {
                          isFinalized = sessionStorage.getItem(finalizedKey) === 'true';
                        }
                      } catch {}
                      
                      // Helper to check now
                      const checkNow = async () => {
                        try {
                          if (isFinalized) return;
                          console.log('🔄 Checking payment status automatically...');
                          const response = await getBookingPayPalStatusByQuery(paypalData.data.booking_no);
                          const data = response.data || {};
                          console.log('🔄 Auto-check response:', data);
                          
                          if (data.success) {
                            const paymentState = data?.data?.payment_state ?? '';
                            const statusField  = data?.data?.status ?? '';
                            const paymentStatus= data?.data?.payment_status ?? '';
                            console.log('🔍 Status fields:', { paymentState, statusField, paymentStatus });
                            const upper = (v: any) => String(v || '').toUpperCase();
                            const flags = [upper(paymentState), upper(statusField), upper(paymentStatus)];
                            const isSuccess = ['APPROVED','COMPLETED','SUCCESS','PAID','SETTLEMENT'].some(v => flags.includes(v));

                            if (isSuccess) {
                              console.log('✅ Payment successful detected!');
                              clearInterval(checkInterval);
                              isFinalized = true;
                              try {
                                if (typeof window !== 'undefined') {
                                  sessionStorage.setItem(finalizedKey, 'true');
                                }
                              } catch {}
                              
                              // Show success modal
                              const bookingDetails = {
                                date: bookingConfirmationData?.date || "N/A",
                                duration: bookingConfirmationData?.duration || "N/A", 
                                participants: bookingConfirmationData?.participants || "N/A",
                                totalAmount: bookingConfirmationData?.totalAmount || "N/A",
                                customerName: bookingConfirmationData?.customerName || "N/A",
                                hotel: bookingConfirmationData?.hotel || "N/A",
                                hotelAddress: bookingConfirmationData?.hotelAddress || "N/A",
                                bookingName: bookingConfirmationData?.bookingName || "N/A",
                                hotelTransfer: bookingConfirmationData?.hotelTransfer || "N/A",
                                packages: bookingConfirmationData?.packages || [],
                                agentName: selectedAgent?.name || "N/A",
                                agentEmail: selectedAgent?.email || "",
                                isAgentMode: showAgentSection
                              };

                              setSuccessBookingData({
                                bookingNumber: paypalData.data.booking_no,
                                bookingDetails
                              });
                              setShowSuccessModal(true);
                              setShowBookingConfirmationModal(false);
                              
                              // Clear PayPal-related states
                              setPaypalBookingNumber(null);
                              setPaypalApprovalUrl(null);
                              setPaypalPaymentStatus(null);
                              setIsProcessingPayPal(false);
                              setIsMonitoringPayPal(false);
                              
                              showAlert('Payment successful! Your booking has been confirmed.', 'success');
                              
                              // Send confirmation email automatically (PayPal specific)
                              console.log('📧 Sending PayPal confirmation email for booking:', paypalData.data.booking_no);
                              await sendPayPalEmailWithRetry(paypalData.data.booking_no);
                            } else {
                              // Backend fallback if local proxy says pending
                              try {
                                const backendResp = await getBackendBookingPaymentStatus(paypalData.data.booking_no)
                                const b = backendResp?.data || {}
                                const bState = b?.data?.payment_state ?? ''
                                const bStatus = b?.data?.status ?? b?.data?.payment_status ?? ''
                                const upper = (v: any) => String(v || '').toUpperCase()
                                const bFlags = [upper(bState), upper(bStatus)]
                                const ok = ['APPROVED','COMPLETED','SUCCESS','PAID','SETTLEMENT'].some(v => bFlags.includes(v))
                                if (ok) {
                                  console.log('✅ Payment successful via backend fallback!')
                                  clearInterval(checkInterval)
                                  isFinalized = true
                                  try { if (typeof window !== 'undefined') sessionStorage.setItem(finalizedKey, 'true') } catch {}
                                  const bookingDetails = {
                                    date: bookingConfirmationData?.date || "N/A",
                                    duration: bookingConfirmationData?.duration || "N/A",
                                    participants: bookingConfirmationData?.participants || "N/A",
                                    totalAmount: bookingConfirmationData?.totalAmount || "N/A",
                                    customerName: bookingConfirmationData?.customerName || "N/A",
                                    hotel: bookingConfirmationData?.hotel || "N/A",
                                    hotelAddress: bookingConfirmationData?.hotelAddress || "N/A",
                                    bookingName: bookingConfirmationData?.bookingName || "N/A",
                                    hotelTransfer: bookingConfirmationData?.hotelTransfer || "N/A",
                                    packages: bookingConfirmationData?.packages || [],
                                    agentName: selectedAgent?.name || "N/A",
                                    agentEmail: selectedAgent?.email || "",
                                    isAgentMode: showAgentSection
                                  }
                                  setSuccessBookingData({ bookingNumber: paypalData.data.booking_no, bookingDetails })
                                  setShowSuccessModal(true)
                                  setShowBookingConfirmationModal(false)
                                  setPaypalBookingNumber(null)
                                  setPaypalApprovalUrl(null)
                                  setPaypalPaymentStatus(null)
                                  setIsProcessingPayPal(false)
                                  setIsMonitoringPayPal(false)
                                  showAlert('Payment successful! Your booking has been confirmed.', 'success')
                                  await sendPayPalEmailWithRetry(paypalData.data.booking_no)
                                }
                              } catch {}
                            }
                          }
                        } catch (error: any) {
                          console.log('🔄 Auto-check error (will retry):', error);
                        }
                      };

                      // Start with immediate check, then poll faster
                      checkNow();
                      const checkInterval = setInterval(checkNow, 1000);
                      
                      // Stop monitoring after 5 minutes
                      setTimeout(() => {
                        clearInterval(checkInterval);
                        setIsMonitoringPayPal(false);
                        console.log('🛑 Stopped PayPal monitoring after timeout');
                      }, 300000); // 5 minutes
                    };
                    
                    // Start monitoring immediately
                    startPayPalMonitoring();
                    
                    showAlert('PayPal payment opened in new tab. We\'ll automatically detect when payment is completed!', 'success');
                  } else {
                    showAlert('PayPal approval URL not received', 'error');
                  }
            
            // Detailed logging for PayPal response
            console.log('🔍 PayPal API Response Details:');
            console.log('🔍 - Full response:', paypalData);
            console.log('🔍 - Success:', paypalData.success);
            console.log('🔍 - Data:', paypalData.data);
            console.log('🔍 - Approval URL:', paypalData.data?.approval_url);
            console.log('🔍 - Booking Number:', paypalData.data?.booking_no);
            console.log('🔍 - Payment ID:', paypalData.data?.payment_id);
            console.log('🔍 - Amount:', paypalData.data?.amount);
            console.log('🔍 - USD Amount:', paypalData.data?.usd_amount);
            console.log('🔍 - Currency:', paypalData.data?.currency);
            console.log('🔍 - Status:', paypalData.data?.status);
            
            // Validate approval URL
            if (paypalData.data?.approval_url) {
              try {
                const url = new URL(paypalData.data.approval_url);
                console.log('🔍 Approval URL Analysis:');
                console.log('🔍 - Protocol:', url.protocol);
                console.log('🔍 - Hostname:', url.hostname);
                console.log('🔍 - Pathname:', url.pathname);
                console.log('🔍 - Search params:', url.search);
                console.log('🔍 - Is PayPal URL:', url.hostname.includes('paypal.com'));
              } catch (urlError) {
                console.error('❌ Invalid approval URL format:', urlError);
              }
            } else {
              console.error('❌ No approval URL in PayPal response');
            }
          } else {
            showAlert('PayPal approval URL not received', 'error');
          }
        } else {
          // Handle specific error cases from documentation
          if (paypalData.message?.includes('already exists')) {
            showAlert('Payment processing error. Please try again.', 'error');
          } else if (paypalData.message?.includes('active PayPal payment')) {
            showAlert('Booking already has an active PayPal payment', 'error');
          } else if (paypalData.errors) {
            // Handle validation errors
            const errorMessages = Object.values(paypalData.errors).flat().join(', ');
            showAlert(`Validation error: ${errorMessages}`, 'error');
          } else {
            showAlert(paypalData.message || 'Failed to create PayPal payment', 'error');
          }
        }
    } catch (error) {
      console.error('Error creating PayPal payment:', error);
      showAlert('Failed to create PayPal payment. Please try again.', 'error');
    } finally {
      setIsProcessingPayPal(false);
    }
  };

  // Add function to check PayPal payment status
  const handleCheckPayPalStatus = async () => {
    if (!paypalBookingNumber) {
      showAlert('No booking number available for status check', 'error');
      return;
    }

    console.log('🔍 Starting Payment Status Check (READ-ONLY)');
    console.log('🔍 - Current Status in State:', paypalPaymentStatus);
    console.log('🔍 - Booking Number:', paypalBookingNumber);
    
    setIsCheckingPayPalStatus(true);
    
    try {
      // Check payment status using the booking number
      const statusUrl = `${process.env.NEXT_PUBLIC_API_URL}/paypal/booking-payment-status/${paypalBookingNumber}`;
      console.log('🔍 Payment Status Check Debug:');
      console.log('🔍 - Booking Number:', paypalBookingNumber);
      console.log('🔍 - Status URL:', statusUrl);
      console.log('🔍 - API URL:', process.env.NEXT_PUBLIC_API_URL);
      console.log('🔍 - Bearer Token:', API_CONFIG.TOKEN);
      console.log('🔍 - Session Token:', snapToken);
      
      const response = await fetch(statusUrl, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      console.log('🔍 PayPal Status Check Response:', data);

      if (data.success) {
        const status = data.data?.status || data.data?.payment_status || '';
        console.log('🔍 Payment Status from API:', status);
        console.log('🔍 Previous Status in State:', paypalPaymentStatus);
        
        // Only update payment status state if it's different
        if (status !== paypalPaymentStatus) {
          console.log('🔄 Status changed from', paypalPaymentStatus, 'to', status);
          setPaypalPaymentStatus(status);
        } else {
          console.log('✅ Status unchanged:', status);
        }
        
        if (['APPROVED', 'COMPLETED', 'SUCCESS', 'PAID', 'SETTLEMENT'].includes(String(status).toUpperCase())) {
          // Handle approved payment like Midtrans success
          const bookingDetails = {
            date: bookingConfirmationData?.date || "N/A",
            duration: bookingConfirmationData?.duration || "N/A", 
            participants: bookingConfirmationData?.participants || "N/A",
            totalAmount: bookingConfirmationData?.totalAmount || "N/A",
            customerName: bookingConfirmationData?.customerName || "N/A",
            hotel: bookingConfirmationData?.hotel || "N/A",
            hotelAddress: bookingConfirmationData?.hotelAddress || "N/A",
            bookingName: bookingConfirmationData?.bookingName || "N/A",
            hotelTransfer: bookingConfirmationData?.hotelTransfer || "N/A",
            packages: bookingConfirmationData?.packages || [],
            agentName: selectedAgent?.name || "N/A",
            agentEmail: selectedAgent?.email || "",
            isAgentMode: showAgentSection
          };

          setSuccessBookingData({
            bookingNumber: paypalBookingNumber,
            bookingDetails
          });
          setShowSuccessModal(true);
          setShowBookingConfirmationModal(false);
          
          // Clear PayPal-related states
          setPaypalBookingNumber(null);
          setPaypalApprovalUrl(null);
          setPaypalPaymentStatus(null);
          setIsCheckingPayPalStatus(false);
          
          // Send confirmation email automatically (PayPal specific)
          try {
            console.log('📧 Sending PayPal confirmation email for booking:', paypalBookingNumber);
            await sendPayPalConfirmationEmail(paypalBookingNumber);
            console.log('✅ PayPal confirmation email sent successfully');
          } catch (emailError) {
            console.error('❌ Failed to send PayPal confirmation email:', emailError);
            // Don't show error to user, email sending failure shouldn't block the success flow
          }
          
        } else if (['PENDING', 'PROCESSING'].includes(String(status).toUpperCase())) {
          showAlert(`Payment status checked: ${status}. Payment is still being processed. Please wait a moment and check again.`, 'info');
        } else if (['FAILED', 'CANCELLED', 'DENIED'].includes(String(status).toUpperCase())) {
          showAlert(`Payment status checked: ${status}. Payment was not successful. Please try again.`, 'error');
        } else {
          showAlert(`Payment status checked: ${status}. Please complete the payment or try again.`, 'info');
        }
      } else {
        showAlert(data.message || 'Failed to check payment status', 'error');
      }
    } catch (error) {
      console.error('Error checking PayPal status:', error);
      showAlert('Failed to check payment status. Please try again.', 'error');
    } finally {
      console.log('🔍 Payment Status Check Completed');
      setIsCheckingPayPalStatus(false);
    }
  };

  // Add function to handle PayPal payment
  const handlePayPalPayment = async (paymentDetails: any, bookingNumber: string) => {
    try {
      // console.log('PayPal payment details:', paymentDetails);
      
      // If paymentDetails doesn't include PayPal IDs (status-based success), just finalize UI feedback
      if (!paymentDetails?.id || !paymentDetails?.payer?.payer_id) {
        showAlert('PayPal payment completed successfully!', 'success');
        return;
      }

      // Execute PayPal payment using backend API (with path fallbacks)
      const executeUrls = [
        getApiUrl('/api/v1/paypal/execute-booking-payment'),
        getApiUrl('/api/v1/paypal/execute-payment'),
        getApiUrl('/paypal/execute-booking-payment'),
        getApiUrl('/paypal/execute-payment'),
      ];

      let data: any = null;
      let lastError: any = null;
      for (const url of executeUrls) {
        try {
          const resp = await fetch(url, {
            method: 'POST',
            headers: {
              'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
              'Content-Type': 'application/json'
            },
            body: JSON.stringify({
              paymentId: paymentDetails.id,
              PayerID: paymentDetails.payer.payer_id,
              booking_no: bookingNumber
            })
          });
          if (!resp.ok) {
            const txt = await resp.text();
            // Try next url on 404
            if (resp.status === 404) {
              console.warn('Execute endpoint 404, trying next:', url);
              lastError = txt;
              continue;
            }
            console.error('Execute endpoint error:', resp.status, txt);
            throw new Error(`Execute payment failed (${resp.status})`);
          }
          data = await resp.json();
          break;
        } catch (e) {
          lastError = e;
        }
      }
      if (!data) {
        throw new Error(`All execute payment endpoints failed. Last error: ${lastError}`);
      }
      
      if (data.success) {
        // PayPal payment success - don't show success modal
        // const bookingDetails = {
        //   date: reservationDays?.[0]?.date || "N/A",
        //   duration: duration === "1-day" ? "1 Day" : duration === "2-days" ? "2 Days" : "3 Days",
        //   participants: `${adultCount > 0 ? `${adultCount} Adult` : ''}${adultCount > 0 && childrenCount > 0 ? ' | ' : ''}${childrenCount > 0 ? `${childrenCount} Children` : ''}`,
        //   totalAmount: getDisplayPrice(voucherData?.net_amount || totalAmount),
        //   customerName: formData2?.fullName || "N/A"
        // };

        // setSuccessBookingData({
        //   bookingNumber,
        //   bookingDetails
        // });
        // setShowSuccessModal(true);
        showAlert('PayPal payment completed successfully!', 'success');
      } else {
        // Handle different types of errors from API response
        if (data.errors) {
          // Handle validation errors - format them nicely
          const errorMessages = Object.values(data.errors).flat().join(', ');
          showAlert(`Validation error: ${errorMessages}`, 'error');
        } else {
          showAlert(data.message || 'Payment verification failed. Please contact support.', 'error');
        }
      }
    } catch (error) {
      console.error('Error processing PayPal payment:', error);
      showAlert('Payment processing failed. Please try again.', 'error');
    }
  };





  // Real-time validation function
  function validateField(fieldName: string, value: any, context?: any) {
    const newErrors = { ...errors };
    
    switch (fieldName) {
      case 'fullName':
        // Skip validation for agent mode
        if (!showAgentSection) {
          if (!value || value.trim() === '') {
            newErrors.fullName = "Full name is required";
          } else {
            delete newErrors.fullName;
          }
        }
        break;
        
      case 'email':
        // Skip validation for agent mode
        if (!showAgentSection) {
          if (!value || value.trim() === '') {
            newErrors.email = "Email is required";
          } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value)) {
            newErrors.email = "Please enter a valid email address";
          } else {
            delete newErrors.email;
          }
        }
        break;
        
      case 'hotel':
        // Only validate if hotel transfer is "Yes"
        if (formData2.hotelTransfer === "Yes" && (!value || value.trim() === '')) {
          newErrors.hotel = "Hotel is required";
        } else {
          delete newErrors.hotel;
        }
        break;
        
      case 'hotelAddress':
        // Only validate if hotel transfer is "Yes"
        if (formData2.hotelTransfer === "Yes" && (!value || value.trim() === '')) {
          newErrors.hotelAddress = "Hotel address is required";
        } else {
          delete newErrors.hotelAddress;
        }
        break;
        
      case 'bookingName':
        // Only validate if hotel transfer is "Yes"
        if (formData2.hotelTransfer === "Yes" && (!value || value.trim() === '')) {
          newErrors.bookingName = "Booking name is required";
        } else {
          delete newErrors.bookingName;
        }
        break;
        
      case 'roomNumber':
        // Only validate if hotel transfer is "Yes"
        if (formData2.hotelTransfer === "Yes" && (!value || value.trim() === '')) {
          newErrors.roomNumber = "Room number is required";
        } else {
          delete newErrors.roomNumber;
        }
        break;
        
      case 'hotelTransfer':
        if (!value) {
          newErrors.hotelTransfer = "Hotel transfer is required";
        } else {
          delete newErrors.hotelTransfer;
          
          // If hotel transfer is "No", clear hotel field errors
          if (value === "No") {
            delete newErrors.hotel;
            delete newErrors.hotelAddress;
            delete newErrors.bookingName;
            delete newErrors.roomNumber;
          }
        }
        break;
        
      case 'agentCode':
        if (!value || value.trim() === '') {
          newErrors.agentCode = "Agent code is required";
        } else if (selectedAgent) {
          // Clear error when a valid agent is selected
          delete newErrors.agentCode;
        } else {
          // Clear error when user types a code - validation will happen on form submit
          delete newErrors.agentCode;
        }
        break;
        
      case 'participantName':
        const { participantType, index } = context;
        const fieldKey = `${participantType}_name_${index}`;
        if (!value || value.trim() === '') {
          newErrors[fieldKey] = "Name required";
        } else {
          delete newErrors[fieldKey];
        }
        break;
        
      case 'participantLevel':
        const { participantType: levelType, index: levelIndex } = context;
        const levelFieldKey = `${levelType}_level_${levelIndex}`;
        if (!value) {
          newErrors[levelFieldKey] = "Level required";
        } else {
          delete newErrors[levelFieldKey];
        }
        break;
        
      case 'childAge':
        const { index: ageIndex } = context;
        const ageFieldKey = `child_age_${ageIndex}`;
        if (!value) {
          newErrors[ageFieldKey] = "Age required";
        } else {
          delete newErrors[ageFieldKey];
        }
        break;
        
      case 'reservationDate':
        const { index: dateIndex } = context;
        const dateFieldKey = `reservation_date_${dateIndex}`;
        if (!value) {
          newErrors[dateFieldKey] = "Date required";
        } else {
          // Check for duplicate dates
          const selectedDates = reservationDays.map(r => r.date).filter(Boolean);
          const dateCount = selectedDates.filter(date => date === value).length;
          if (dateCount > 1) {
            newErrors[dateFieldKey] = "Each day must have a different date";
          } else {
            delete newErrors[dateFieldKey];
          }
        }
        break;
        
      case 'reservationTime':
        const { index: timeIndex } = context;
        const timeFieldKey = `reservation_time_${timeIndex}`;
        const reservation = reservationDays[timeIndex];
        // Only validate time if time field is visible (when date exists and surf lessons are selected)
        if (reservation?.date && selectedActivities.surfLessons && !value) {
          newErrors[timeFieldKey] = "Time required";
        } else {
          delete newErrors[timeFieldKey];
        }
        break;
        
      case 'activities':
        if (!Object.values(selectedActivities).some(Boolean)) {
          newErrors.activities = "Please select at least one surfing activity";
        } else {
          delete newErrors.activities;
        }
        break;
        
      case 'participants':
        // Skip validation for agent mode
        if (!showAgentSection) {
          if (adultCount === 0 && childrenCount === 0) {
            newErrors.participants = "At least one participant required";
          } else {
            delete newErrors.participants;
          }
        }
        break;
        
      case 'selectedPackages':
        if (Object.keys(selectedPackages).length === 0) {
          newErrors.selectedPackages = "Please select at least one package";
        } else {
          // Check if all participants have packages
          const allParticipantNames = [
            ...formData.adults.map(a => a.name),
            ...formData.children.map(c => c.name)
          ].filter(name => name.trim() !== "");

          let participantsWithoutPackages: string[] = [];
          
          if (showAgentSection) {
            // Agent mode: Check selectedPackages
            const participantsWithPackages = Object.values(selectedPackages || {}).flatMap((pkg: any) => 
              pkg.people || []
            );
            participantsWithoutPackages = allParticipantNames.filter(name => 
              !participantsWithPackages.includes(name)
            );
          } else {
            // Customer mode: Check recommendation field
            participantsWithoutPackages = allParticipantNames.filter(name => {
              const adult = formData.adults.find(a => a.name === name);
              const child = formData.children.find(c => c.name === name);
              const participant = adult || child;
              return !participant?.recommendation || participant.recommendation.trim() === "";
            });
          }

          if (participantsWithoutPackages.length > 0) {
            newErrors.packageSelection = `The following participants haven't selected a package: ${participantsWithoutPackages.join(', ')}`;
          } else {
            delete newErrors.packageSelection;
          }
          delete newErrors.selectedPackages;
        }
        break;
        
      case 'agreeTerms':
        if (!value) {
          newErrors.agreeTerms = "You must agree to the terms";
        } else {
          delete newErrors.agreeTerms;
        }
        break;
    }
    
    setErrors(newErrors);
  }

  // Prevent hydration error by not rendering until client-side
  if (!isClient) {
    return <LoadingSpinner />;
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <BookingHeader />
      
      {/* Main Content */}
      <div className="max-w-7xl mx-auto p-6">
        <div className="bg-white shadow-sm rounded-xl border border-gray-100">
          <div className="p-6">
            {/* Title Section */}
            <div className="border-b border-gray-200 pb-6 mb-8">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className="w-10 h-10 bg-orange-500 rounded-lg flex items-center justify-center">
                    <Star className="w-5 h-5 text-white" />
                  </div>
                  <div>
                    <h1 className="text-lg sm:text-lg md:text-xl lg:text-2xl font-bold text-gray-900">Your Surf Booking</h1>
                    <p className="text-sm sm:text-sm md:text-base text-gray-600 mt-1">Complete your surfing adventure</p>
                  </div>
                </div>
                
                {/* Reset Button */}
                <button
                  onClick={() => {
                    if (confirm('Are you sure you want to reset all data? This will clear all form data and start fresh.')) {
                      forceResetAndReload();
                    }
                  }}
                  className="px-4 py-2 text-sm font-medium text-red-600 bg-red-50 border border-red-200 rounded-lg hover:bg-red-100 hover:border-red-300 transition-colors"
                  title="Reset all form data"
                >
                  🔄 Reset All
                </button>
              </div>
            </div>

            {/* Agent Code Section */}
            <AgentCodeSection
              currency={currency}
              showAgentSection={showAgentSection}
              agentCode={agentCode}
              setAgentCode={setAgentCode}
              selectedAgent={selectedAgent}
              setSelectedAgent={setSelectedAgent}
              fetchAgentByCode={fetchAgentByCode}
              loadingAgent={loadingAgent}
              errors={errors}
              validateField={validateField}
            />

            {/* Lessons & Tours Specifications - Always shown */}
            <LessonsToursSpecifications
              currency={currency}
              setCurrency={handleSetCurrency}
              onRateChange={(rate) => {
                setUsdRate(typeof rate === 'number' ? rate : null);
              }}
              paymentMethod={paymentMethod}
              selectedActivities={selectedActivities}
              setSelectedActivities={setSelectedActivities}
              duration={duration}
              setDuration={setDuration}
              reservationDays={reservationDays}
              setReservationDays={setReservationDays}
              schedules={schedules}
              errors={errors}
              validateField={validateField}
              showAgentSection={showAgentSection}
            />

            <ParticipantForm
              formData={formData}
              setFormData={setFormData}
              adultCount={adultCount}
              setAdultCount={setAdultCount}
              childrenCount={childrenCount}
              setChildrenCount={setChildrenCount}
              selectedActivities={selectedActivities}
              reservationDays={reservationDays}
              selectedPackages={selectedPackages}
              getDisplayPrice={getDisplayPrice}
              showAgentSection={showAgentSection}
              // Agent-specific props
              totalPax={totalPax}
              setTotalPax={setTotalPax}
              paxGroups={paxGroups}
              setPaxGroups={setPaxGroups}
              addPaxGroup={addPaxGroup}
              updatePaxGroup={updatePaxGroup}
              removePaxGroup={removePaxGroup}
              getRemainingPax={getRemainingPax}
              isPaxValid={isPaxValid}
              getPackageRequirementMessage={getPackageRequirementMessage}
            />

            {/* Toast Alert Container */}
            <AlertContainer 
              alerts={alerts}
              onRemove={(id: string) => setAlerts(alerts => alerts.filter(alert => alert.id !== id))}
            />

            {/* Customer Information Form - Hidden for Agent Booking */}
            {!showAgentSection && (
              <CustomerInformationForm 
                formData2={formData2} 
                setFormData2={setFormData2} 
                errors={errors}
                validateField={validateField}
              />
            )}

            {/* Information Box */}
            {/* <div className="mt-8 bg-yellow-50 border border-yellow-200 rounded-lg p-6">
              <div className="text-sm text-gray-700 space-y-2">
                <p className="italic">
                  Please arrange your own transportation and please come to the odysseys's shop 15 minutes earlier.
                </p>
                <p className="italic">
                  Office's location = inside Mercure kuta hotel - jalan pantai kuta ( next to Hard Rock hotel ),
                  Kuta-Bali, Indonesia
                </p>
                <p className="italic">
                  Map:{" "}
                  <a href="http://goo.gl/0WLN7B" className="text-blue-600 underline">
                    http://goo.gl/0WLN7B
                  </a>
                </p>
                <p className="italic">
                  Please note, there is no additional hours or re-schedule for the lesson if you come late to our office.
                </p>
              </div>
            </div> */}

            <BookingSummary
              reservationDays={reservationDays}
              duration={duration}
              adultCount={adultCount}
              childrenCount={childrenCount}
              customerInfo={customerInfo}
              selectedPackages={selectedPackages}
              getDisplayPrice={getDisplayPrice}
              handleCancel={handleCancel}
              promoCode={promoCode}
              setPromoCode={setPromoCode}
              applyVoucher={applyVoucher}
              promoMessage={promoMessage}
              promoSuccess={promoSuccess}
              setPromoMessage={setPromoMessage}
              setPromoSuccess={setPromoSuccess}
              setVoucherData={setVoucherData}
              totalAmount={finalTotalAmount}
              voucherData={voucherData}
              paymentMethod={paymentMethod}
              setPaymentMethod={setPaymentMethod}
              agreeTerms={agreeTerms}
              setAgreeTerms={setAgreeTerms}
              handlePayment={handlePayment}
              errors={errors}
              formData={formData}
              currency={currency}
              setCurrency={handleSetCurrency}
              onRateChange={(rate) => {
                setUsdRate(typeof rate === 'number' ? rate : null);
              }}
              // Agent-specific props
              splitPayments={splitPayments}
              setSplitPayments={setSplitPayments}
              showAgentSection={showAgentSection}
              totalPax={totalPax}
              paxGroups={paxGroups}
              selectedAgent={selectedAgent}
              agentPaymentAmounts={agentPaymentAmounts}
              setAgentPaymentAmounts={setAgentPaymentAmounts}
              // Payment status props
              paypalBookingNumber={paypalBookingNumber}
              paypalPaymentStatus={paypalPaymentStatus}
            />
          </div>
        </div>
      </div>

      <BookingFooter />

      {/* Booking Success Modal */}
      {successBookingData && (
        <BookingSuccessModal
          isOpen={showSuccessModal}
          onClose={() => {
            setShowSuccessModal(false);
            setSuccessBookingData(null);
          }}
          onReset={() => {
            setShowSuccessModal(false);
            setSuccessBookingData(null);
            // Reset all form data and clear localStorage
            resetAllData();
            // Reload page to ensure complete fresh start
            window.location.reload();
          }}
          bookingNumber={successBookingData.bookingNumber}
          bookingDetails={successBookingData.bookingDetails}
        />
      )}

      {/* Midtrans Payment Modal */}
      {showMidtransModal && snapToken && (
        <div className="fixed inset-0 z-50 flex items-center justify-center">
          <div className="absolute inset-0 bg-black/50 backdrop-blur-sm"></div>
          <div className="relative bg-white rounded-2xl shadow-2xl max-w-lg w-full mx-4 p-6">
            <div className="text-center mb-6">
              <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <span className="text-2xl">💳</span>
              </div>
              <h2 className="text-2xl font-bold text-gray-900 mb-2">
                Secure Payment
              </h2>
              <p className="text-gray-600">
                Your booking has been created successfully. Please complete the payment to confirm your reservation.
              </p>
            </div>
            
            {/* Payment Details */}
            <div className="bg-gradient-to-r from-blue-50 to-green-50 rounded-xl p-4 mb-6 border-2 border-blue-200">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="text-gray-600">Amount:</span>
                  <div className="font-bold text-lg text-gray-900">
                    {getDisplayPrice(voucherData?.net_amount || finalTotalAmount)}
                  </div>
                </div>
                <div>
                  <span className="text-gray-600">Booking:</span>
                  <div className="font-semibold text-gray-900">
                    {successBookingData?.bookingNumber || "N/A"}
                  </div>
                </div>
              </div>
            </div>

            {/* Payment Method Info */}
            <div className="bg-yellow-50 rounded-lg p-4 mb-6 border border-yellow-200">
              <div className="flex items-center gap-2 mb-2">
                <span className="text-yellow-600">🔒</span>
                <span className="font-semibold text-yellow-800">Secure Payment Gateway</span>
              </div>
              <p className="text-sm text-yellow-700">
                You'll be redirected to Midtrans secure payment page. Multiple payment methods available including credit cards, bank transfer, and e-wallets.
              </p>
            </div>



            <div className="space-y-3">
              <button
                onClick={() => {
                  handleMidtransPayment(snapToken);
                }}
                className="w-full bg-gradient-to-r from-blue-600 to-green-600 hover:from-blue-700 hover:to-green-700 text-white font-semibold py-4 px-6 rounded-xl shadow-lg transform transition-all duration-200 hover:scale-105"
              >
                <span className="flex items-center justify-center gap-2">
                  <span>💳</span>
                  <span>Proceed to Secure Payment</span>
                </span>
              </button>
              
              <button
                onClick={() => {
                  setShowMidtransModal(false);
                  setSnapToken(null);
                }}
                className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-lg transition-colors"
              >
                Cancel Payment
              </button>
            </div>
            
            {/* Debug info - remove in production */}
            {process.env.NODE_ENV === 'development' && (
              <div className="mt-4 p-3 bg-gray-100 rounded text-xs text-gray-600">
                <strong>Debug Info:</strong><br/>
                Snap Token: {snapToken.substring(0, 20)}...<br/>
                Client Key: {MIDTRANS_CONFIG.CLIENT_KEY}<br/>
                Environment: {MIDTRANS_CONFIG.IS_PRODUCTION ? 'Production' : 'Sandbox'}
              </div>
            )}
          </div>
        </div>
      )}



             {/* Payment Failed Modal */}
       {showPaymentFailedModal && (
         <div className="fixed inset-0 z-50 flex items-center justify-center">
           <div className="absolute inset-0 bg-black/50 backdrop-blur-sm"></div>
           <div className="relative bg-white rounded-2xl shadow-2xl max-w-lg w-full mx-4 p-6">
             <div className="text-center mb-6">
               <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                 <span className="text-2xl">❌</span>
               </div>
               <h2 className="text-2xl font-bold text-gray-900 mb-2">
                 Payment Failed
               </h2>
               <p className="text-gray-600">
                 Your payment was unsuccessful. Please try again or contact support.
               </p>
             </div>
             
             {/* Error Info */}
             <div className="bg-red-50 rounded-lg p-4 mb-6 border border-red-200">
               <div className="flex items-center gap-2 mb-2">
                 <span className="text-red-600">⚠️</span>
                 <span className="font-semibold text-red-800">Possible reasons:</span>
               </div>
               <ul className="text-sm text-red-700 space-y-1">
                 <li>• Insufficient funds in your account</li>
                 <li>• Card was declined by your bank</li>
                 <li>• Network connection issues</li>
                 <li>• Payment gateway temporarily unavailable</li>
               </ul>
             </div>
             
             <div className="space-y-3">
              <button
                onClick={() => {
                  setShowPaymentFailedModal(false);
                  // Try to re-create the booking and get a new snap token
                  handlePayment();
                }}
                className="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-4 px-6 rounded-xl shadow-lg transform transition-all duration-200 hover:scale-105"
              >
                <span className="flex items-center justify-center gap-2">
                  <span>🔄</span>
                  <span>Try Again</span>
                </span>
              </button>
               <button
                 onClick={() => {
                   setShowPaymentFailedModal(false);
                   setShowMidtransModal(false);
                   setSnapToken(null);
                 }}
                 className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-lg transition-colors"
               >
                 Cancel Payment
               </button>
             </div>
           </div>
         </div>
       )}

             {/* Payment Settlement Modal */}
       {showPaymentSettlementModal && (
         <div className="fixed inset-0 z-50 flex items-center justify-center">
           <div className="absolute inset-0 bg-black/50 backdrop-blur-sm"></div>
           <div className="relative bg-white rounded-2xl shadow-2xl max-w-lg w-full mx-4 p-6">
             <div className="text-center mb-6">
               <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                 <span className="text-2xl">✅</span>
               </div>
               <h2 className="text-2xl font-bold text-gray-900 mb-2">
                 Payment Successful
               </h2>
               <p className="text-gray-600">
                 Your payment has been settled. Your booking is now confirmed.
               </p>
             </div>
             
             {/* Booking Details */}
             <div className="bg-gradient-to-r from-green-50 to-blue-50 rounded-xl p-4 mb-6 border-2 border-green-300">
               <h3 className="font-semibold text-gray-900 mb-3">📋 Booking Details</h3>
               <div className="grid grid-cols-2 gap-4 text-sm">
                 <div>
                   <span className="text-gray-600">Booking Number:</span>
                   <div className="font-bold text-gray-900">
                     {successBookingData?.bookingNumber || "N/A"}
                   </div>
                 </div>
                 <div>
                   <span className="text-gray-600">Amount:</span>
                   <div className="font-bold text-green-600">
                     {getDisplayPrice(voucherData?.net_amount || finalTotalAmount)}
                   </div>
                 </div>
                 <div>
                   <span className="text-gray-600">Date:</span>
                   <div className="font-semibold text-gray-900">
                     {reservationDays?.[0]?.date || "N/A"}
                   </div>
                 </div>
                 <div>
                   <span className="text-gray-600">Duration:</span>
                   <div className="font-semibold text-gray-900">
                     {duration === "1-day" ? "1 Day" : duration === "2-days" ? "2 Days" : "3 Days"}
                   </div>
                 </div>
                 <div className="col-span-2">
                   <span className="text-gray-600">Participants:</span>
                   <div className="font-semibold text-gray-900">
                     {`${adultCount > 0 ? `${adultCount} Adult` : ''}${adultCount > 0 && childrenCount > 0 ? ' | ' : ''}${childrenCount > 0 ? `${childrenCount} Children` : ''}`}
                   </div>
                 </div>
                 <div className="col-span-2">
                   <span className="text-gray-600">Customer:</span>
                   <div className="font-semibold text-gray-900">
                     {formData2?.fullName || "N/A"}
                   </div>
                 </div>
               </div>
             </div>
             
             <div className="space-y-3">
               <button
                 onClick={() => {
                   setShowPaymentSettlementModal(false);
                   setShowMidtransModal(false);
                   setSnapToken(null);
                   
                   // Reset form for new booking
                   resetAllData();
                 }}
                 className="w-full bg-gradient-to-r from-green-600 to-blue-600 hover:from-green-700 hover:to-blue-700 text-white font-semibold py-4 px-6 rounded-xl shadow-lg transform transition-all duration-200 hover:scale-105"
               >
                 <span className="flex items-center justify-center gap-2">
                   <span>🎉</span>
                   <span>Start New Booking</span>
                 </span>
               </button>
               <button
                 onClick={() => {
                   setShowPaymentSettlementModal(false);
                   setShowMidtransModal(false);
                   setSnapToken(null);
                 }}
                 className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-lg transition-colors"
               >
                 Close
               </button>
             </div>
           </div>
         </div>
       )}

             {/* Payment Pending Modal */}
       {showPaymentPendingModal && (
         <div className="fixed inset-0 z-50 flex items-center justify-center">
           <div className="absolute inset-0 bg-black/50 backdrop-blur-sm"></div>
           <div className="relative bg-white rounded-2xl shadow-2xl max-w-lg w-full mx-4 p-6">
             <div className="text-center mb-6">
               <div className="w-16 h-16 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-4">
                 <span className="text-2xl">⏳</span>
               </div>
               <h2 className="text-2xl font-bold text-gray-900 mb-2">
                 Payment Pending
               </h2>
               <p className="text-gray-600">
                 Your payment is being processed. We'll notify you once it's completed.
               </p>
             </div> 
             
             {/* Status Info */}
             <div className="bg-yellow-50 rounded-lg p-4 mb-6 border border-yellow-200">
               <div className="flex items-center gap-2 mb-2">
                 <span className="text-yellow-600">ℹ️</span>
                 <span className="font-semibold text-yellow-800">What happens next?</span>
               </div>
               <ul className="text-sm text-yellow-700 space-y-1">
                 <li>• We're checking your payment status every 5 seconds</li>
                 <li>• You'll receive a notification when payment is confirmed</li>
                 <li>• Your booking will be automatically confirmed</li>
                 <li>• You can close this modal and check back later</li>
               </ul>
             </div>
             
             <div className="space-y-3">
               <button
                 onClick={() => {
                   setShowPaymentPendingModal(false);
                   // Reopen Midtrans payment modal with the existing snapToken
                   if (snapToken) {
                     handleMidtransPayment(snapToken);
                   } else {
                     showAlert('Payment session expired. Please create a new booking.', 'error');
                   }
                 }}
                 className="w-full bg-yellow-600 hover:bg-yellow-700 text-white font-semibold py-4 px-6 rounded-xl shadow-lg transform transition-all duration-200 hover:scale-105"
               >
                 <span className="flex items-center justify-center gap-2">
                   <span>⏳</span>
                   <span>Continue Monitoring</span>
                 </span>
               </button>
               <button
                 onClick={() => {
                   setShowPaymentPendingModal(false);
                   setShowMidtransModal(false);
                   setSnapToken(null);
                 }}
                 className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-lg transition-colors"
               >
                 Close
               </button>
             </div>
           </div>
         </div>
       )}

             {/* Payment Expired Modal */}
       {showPaymentExpiredModal && (
         <div className="fixed inset-0 z-50 flex items-center justify-center">
           <div className="absolute inset-0 bg-black/50 backdrop-blur-sm"></div>
           <div className="relative bg-white rounded-2xl shadow-2xl max-w-lg w-full mx-4 p-6">
             <div className="text-center mb-6">
               <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                 <span className="text-2xl">⏰</span>
               </div>
               <h2 className="text-2xl font-bold text-gray-900 mb-2">
                 Payment Expired
               </h2>
               <p className="text-gray-600">
                 Your payment session has expired. Please try again with a new payment session.
               </p>
             </div>
             
             {/* Expiry Info */}
             <div className="bg-orange-50 rounded-lg p-4 mb-6 border border-orange-200">
               <div className="flex items-center gap-2 mb-2">
                 <span className="text-orange-600">⏰</span>
                 <span className="font-semibold text-orange-800">Why did this happen?</span>
               </div>
               <ul className="text-sm text-orange-700 space-y-1">
                 <li>• Payment session timeout (usually 15 minutes)</li>
                 <li>• Browser was closed during payment</li>
                 <li>• Network interruption during payment</li>
                 <li>• Payment gateway session expired</li>
               </ul>
             </div>
             
             <div className="space-y-3">
               <button
                 onClick={() => {
                   setShowPaymentExpiredModal(false);
                   // Try to re-create the booking and get a new snap token
                   handlePayment();
                 }}
                 className="w-full bg-orange-600 hover:bg-orange-700 text-white font-semibold py-4 px-6 rounded-xl shadow-lg transform transition-all duration-200 hover:scale-105"
               >
                 <span className="flex items-center justify-center gap-2">
                   <span>🔄</span>
                   <span>Try Again</span>
                 </span>
               </button>
               <button
                 onClick={() => {
                   setShowPaymentExpiredModal(false);
                   setShowMidtransModal(false);
                   setSnapToken(null);
                 }}
                 className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-lg transition-colors"
               >
                 Cancel Payment
               </button>
             </div>
           </div>
         </div>
       )}

             {/* Payment Denied Modal */}
       {showPaymentDeniedModal && (
         <div className="fixed inset-0 z-50 flex items-center justify-center">
           <div className="absolute inset-0 bg-black/50 backdrop-blur-sm"></div>
           <div className="relative bg-white rounded-2xl shadow-2xl max-w-lg w-full mx-4 p-6">
             <div className="text-center mb-6">
               <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                 <span className="text-2xl">🚫</span>
               </div>
               <h2 className="text-2xl font-bold text-gray-900 mb-2">
                 Payment Denied
               </h2>
               <p className="text-gray-600">
                 Your payment was denied by the payment processor. Please try again or contact support.
               </p>
             </div>
             
             {/* Denial Info */}
             <div className="bg-red-50 rounded-lg p-4 mb-6 border border-red-200">
               <div className="flex items-center gap-2 mb-2">
                 <span className="text-red-600">🚫</span>
                 <span className="font-semibold text-red-800">Why was payment denied?</span>
               </div>
               <ul className="text-sm text-red-700 space-y-1">
                 <li>• Card was blocked by your bank</li>
                 <li>• Suspicious transaction detected</li>
                 <li>• Payment method not supported</li>
                 <li>• Transaction limit exceeded</li>
               </ul>
             </div>
             
             <div className="space-y-3">
               <button
                 onClick={() => {
                   setShowPaymentDeniedModal(false);
                   // Try to re-create the booking and get a new snap token
                   handlePayment();
                 }}
                 className="w-full bg-red-600 hover:bg-red-700 text-white font-semibold py-4 px-6 rounded-xl shadow-lg transform transition-all duration-200 hover:scale-105"
               >
                 <span className="flex items-center justify-center gap-2">
                   <span>🔄</span>
                   <span>Try Again</span>
                 </span>
               </button>
               <button
                 onClick={() => {
                   setShowPaymentDeniedModal(false);
                   setShowMidtransModal(false);
                   setSnapToken(null);
                 }}
                 className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-semibold py-3 px-6 rounded-lg transition-colors"
               >
                 Cancel Payment
               </button>
             </div>
           </div>
         </div>
       )}


      {/* Booking Confirmation Modal */}
      {showBookingConfirmationModal && bookingConfirmationData && (
        <BookingConfirmationModal
          isOpen={showBookingConfirmationModal}
          onClose={() => setShowBookingConfirmationModal(false)}
          onConfirm={handlePayPalConfirmation}
          onCheckStatus={handleCheckPayPalStatus}
          bookingData={bookingConfirmationData}
          isLoading={isProcessingPayPal}
          isCheckingStatus={isCheckingPayPalStatus}
          showCheckStatusButton={!!paypalBookingNumber}
          preventClose={isProcessingPayPal}
          isMonitoringPayPal={isMonitoringPayPal}
        />
      )}

      {/* Payment Loading Overlay */}
      <PaymentLoadingOverlay 
        isVisible={isPaymentLoading}
        message="Processing your payment..."
      />
    </div>
  )
} 