"use client"

import React, { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Loader2, X, RefreshCcw } from 'lucide-react';
import { getBookingPayPalStatus, getBookingPayPalStatusByQuery, getBackendBookingPaymentStatus } from '@/service/paypalApi';
import { PAYPAL_CONFIG } from '@/lib/config';

interface PayPalModalProps {
  isOpen: boolean;
  onClose: () => void;
  amount: number;
  bookingCode: string;
  approvalUrl?: string | null;
  onSuccess: (paymentDetails: any) => void;
  onError: (error: any) => void;
}

export default function PayPalModal({
  isOpen,
  onClose,
  amount,
  bookingCode,
  approvalUrl,
  onSuccess,
  onError
}: PayPalModalProps) {
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [showPaymentAddress, setShowPaymentAddress] = useState(false);
  const [isPaymentProcessing, setIsPaymentProcessing] = useState(false);
  const [paymentStatusInterval, setPaymentStatusInterval] = useState<NodeJS.Timeout | null>(null);

  // Debug logging when modal opens
  React.useEffect(() => {
    if (isOpen) {
      console.log('🔍 PayPalModal opened with data:');
      console.log('🔍 - approvalUrl:', approvalUrl);
      console.log('🔍 - bookingCode:', bookingCode);
      console.log('🔍 - amount:', amount);
      console.log('🔍 - approvalUrl type:', typeof approvalUrl);
      console.log('🔍 - approvalUrl length:', approvalUrl?.length);
      
      if (approvalUrl) {
        try {
          const url = new URL(approvalUrl);
          console.log('🔍 URL analysis:');
          console.log('🔍 - Full URL:', approvalUrl);
          console.log('🔍 - Protocol:', url.protocol);
          console.log('🔍 - Hostname:', url.hostname);
          console.log('🔍 - Pathname:', url.pathname);
          console.log('🔍 - Search params:', url.search);
          console.log('🔍 - Hash:', url.hash);
        } catch (urlError) {
          console.error('❌ Invalid URL format in modal:', urlError);
        }
      } else {
        console.warn('⚠️ No approval URL provided to PayPalModal');
      }
    }
  }, [isOpen, approvalUrl, bookingCode, amount]);

  // Cleanup interval on unmount
  React.useEffect(() => {
    return () => {
      if (paymentStatusInterval) {
        clearInterval(paymentStatusInterval);
      }
    };
  }, [paymentStatusInterval]);

  // Prevent flapping: once finalized, never process pending again
  const isFinalizedRef = React.useRef<boolean>(false);

  // Start automatic payment status checking
  const startPaymentStatusPolling = () => {
    if (paymentStatusInterval) {
      clearInterval(paymentStatusInterval);
    }
    
    console.log('🔄 Starting automatic payment status polling...');
    const checkNow = async () => {
      try {
        if (isFinalizedRef.current) return; // guard

        const response = await getBookingPayPalStatusByQuery(bookingCode);
        const data = (response as any)?.data || {};
        const paymentState = data?.data?.payment_state ?? '';
        const statusField  = data?.data?.status ?? '';
        const paymentStatus= data?.data?.payment_status ?? '';
        console.log('🔍 Status fields:', { paymentState, statusField, paymentStatus });
        const upper = (v: any) => String(v || '').toUpperCase();
        const flags = [upper(paymentState), upper(statusField), upper(paymentStatus)];
        let isSuccess = ['APPROVED','COMPLETED','SUCCESS','PAID','SETTLEMENT'].some(v => flags.includes(v));

        // Fallback: query backend directly if not success
        if (!isSuccess) {
          try {
            const backendResp = await getBackendBookingPaymentStatus(bookingCode);
            const b = (backendResp as any)?.data || {};
            const bState = b?.data?.payment_state ?? '';
            const bStatus = b?.data?.status ?? b?.data?.payment_status ?? '';
            const upperB = (v: any) => String(v || '').toUpperCase();
            const bFlags = [upperB(bState), upperB(bStatus)];
            isSuccess = ['APPROVED','COMPLETED','SUCCESS','PAID','SETTLEMENT'].some(v => bFlags.includes(v));
          } catch {}
        }

        if (isSuccess) {
          console.log('✅ Payment successful detected automatically!');
          clearInterval(interval);
          setPaymentStatusInterval(null);
          setIsPaymentProcessing(false);
          isFinalizedRef.current = true;
          try {
            if (typeof window !== 'undefined' && bookingCode) {
              sessionStorage.setItem(`paypalFinalized:${bookingCode}`, 'true');
            }
          } catch {}
          onSuccess({ status, bookingCode });
        }
      } catch (error) {
        console.log('🔄 Auto-check error (will retry):', error);
      }
    };
    // run once immediately for real-time feel
    checkNow();
    const interval = setInterval(checkNow, 1000);

    setPaymentStatusInterval(interval);
  };

  // Stop automatic payment status checking
  const stopPaymentStatusPolling = () => {
    if (paymentStatusInterval) {
      clearInterval(paymentStatusInterval);
      setPaymentStatusInterval(null);
      console.log('🛑 Stopped automatic payment status polling');
    }
  };

  const handleCheckStatus = async () => {
    if (!bookingCode) {
      setError('Missing booking code for status check');
      return;
    }
    setIsLoading(true);
    setError(null);
    try {
      // Prefer query param with booking_no first
      let response = await getBookingPayPalStatusByQuery(bookingCode);
      const data = (response as any)?.data || {};
      const paymentState = data?.data?.payment_state ?? '';
      const statusField  = data?.data?.status ?? '';
      const paymentStatus= data?.data?.payment_status ?? '';
      const upper = (v: any) => String(v || '').toUpperCase();
      const flags = [upper(paymentState), upper(statusField), upper(paymentStatus)];

      if (typeof window !== 'undefined') {
        // Optional: provide quick feedback via alert for now
        const message = `Current PayPal status: ${paymentState || statusField || paymentStatus || 'UNKNOWN'}`;
        // eslint-disable-next-line no-alert
        window.alert(message);
      }

      const isSuccess = ['APPROVED','COMPLETED','SUCCESS','PAID','SETTLEMENT'].some(v => flags.includes(v));

      if (isSuccess) {
        isFinalizedRef.current = true;
        try {
          if (typeof window !== 'undefined' && bookingCode) {
            sessionStorage.setItem(`paypalFinalized:${bookingCode}`, 'true');
          }
        } catch {}
        onSuccess({ status: paymentState || statusField || paymentStatus, bookingCode });
      }
    } catch (err: any) {
      const serverMsg = err?.response?.data?.message || err?.message;
      const httpStatus = err?.response?.status;
      setError(`Failed to check PayPal status${httpStatus ? ` (HTTP ${httpStatus})` : ''}${serverMsg ? `: ${serverMsg}` : ''}`);
      onError(err);
    } finally {
      setIsLoading(false);
    }
  };

  const handleClose = () => {
    setError(null);
    setIsLoading(false);
    setShowPaymentAddress(false);
    setIsPaymentProcessing(false);
    stopPaymentStatusPolling();
    isFinalizedRef.current = false;
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-4xl max-h-[90vh] flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="flex items-center justify-between">
            <span>PayPal Payment</span>
            <Button
              variant="ghost"
              size="sm"
              onClick={handleClose}
              disabled={isLoading}
            >
              <X className="h-4 w-4" />
            </Button>
          </DialogTitle>
        </DialogHeader>

        <div className="flex-1 overflow-y-auto space-y-4">
          {error && (
            <div className="bg-red-50 border border-red-200 rounded-md p-3">
              <p className="text-sm text-red-600">{error}</p>
            </div>
          )}

          <div className="text-center space-y-2">
            <p className="text-sm text-gray-600">
              Complete your payment securely with PayPal
            </p>
            <p className="text-lg font-semibold">
              Total Amount: ${amount.toFixed(2)} USD
            </p>
          </div>

          {isLoading && (
            <div className="flex items-center justify-center py-6">
              <Loader2 className="h-6 w-6 animate-spin text-blue-600" />
              <span className="ml-2 text-sm text-gray-600">Processing...</span>
            </div>
          )}

          {isPaymentProcessing && (
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
              <div className="flex items-center justify-center space-x-2">
                <Loader2 className="h-5 w-5 animate-spin text-blue-600" />
                <div className="text-center">
                  <p className="text-sm font-medium text-blue-800">Payment in Progress</p>
                  <p className="text-xs text-blue-600">Please complete your payment in the form below</p>
                  <p className="text-xs text-blue-500 mt-1">We'll automatically detect when payment is successful</p>
                </div>
              </div>
            </div>
          )}

          {!showPaymentAddress ? (
            <div className="space-y-3">
              <Button
                className="w-full"
                disabled={!approvalUrl || isLoading}
                onClick={() => {
                  if (approvalUrl) {
                    setShowPaymentAddress(true);
                  }
                }}
              >
                Continue Payment
              </Button>

              <Button
                variant="secondary"
                className="w-full"
                disabled={isLoading || !bookingCode || isPaymentProcessing}
                onClick={handleCheckStatus}
              >
                <RefreshCcw className="h-4 w-4 mr-2" />
                {isPaymentProcessing ? 'Auto-checking...' : 'Check Payment Status'}
              </Button>
            </div>
          ) : (
            <div className="space-y-4">
              {/* PayPal Payment Form */}
              <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-6 border-2 border-blue-200">
                <div className="text-center space-y-2 mb-4">
                  <p className="text-sm text-gray-600">
                    Complete your payment securely through PayPal
                  </p>
                  <p className="text-lg font-semibold text-blue-600">
                    Booking Code: {bookingCode}
                  </p>
                </div>

                <div className="bg-white rounded-lg p-6 border">
                  <div className="text-center mb-6">
                    <h4 className="text-xl font-semibold text-gray-900 mb-2">
                      PayPal Payment
                    </h4>
                    <p className="text-sm text-gray-600 mb-4">
                      Amount: ${amount.toFixed(2)} USD
                    </p>
                  </div>

                  {/* Payment Form */}
                  <div className="space-y-4 payment-container">
                    <div className="bg-gray-50 rounded-lg p-4">
                      <h5 className="font-semibold text-gray-800 mb-3">Payment Instructions:</h5>
                      <ol className="text-sm text-gray-700 space-y-2">
                        <li>1. Complete the payment process below</li>
                        <li>2. After payment, click "Check Payment Status" to verify</li>
                        <li>3. Your booking will be confirmed once payment is successful</li>
                      </ol>
                    </div>

                    {/* Payment Button */}
                    <div className="text-center">
                      <Button
                        className="w-full bg-blue-600 hover:bg-blue-700 text-white py-4 px-8 text-lg font-semibold"
                        onClick={() => {
                          console.log('🔍 PayPal Debug Information:');
                          console.log('🔍 approvalUrl:', approvalUrl);
                          console.log('🔍 approvalUrl type:', typeof approvalUrl);
                          console.log('🔍 approvalUrl length:', approvalUrl?.length);
                          console.log('🔍 bookingCode:', bookingCode);
                          console.log('🔍 amount:', amount);
                          
                          if (approvalUrl) {
                            // Validate URL format
                            try {
                              const url = new URL(approvalUrl);
                              console.log('🔍 URL validation successful:');
                              console.log('🔍 - Protocol:', url.protocol);
                              console.log('🔍 - Hostname:', url.hostname);
                              console.log('🔍 - Pathname:', url.pathname);
                              console.log('🔍 - Search params:', url.search);
                              
                              // Check if it's a PayPal URL
                              const isPayPalUrl = url.hostname.includes('paypal.com') || url.hostname.includes('sandbox.paypal.com');
                              console.log('🔍 - Is PayPal URL:', isPayPalUrl);
                              
                              if (!isPayPalUrl) {
                                console.error('❌ Invalid PayPal URL - hostname does not contain paypal.com');
                                alert('Invalid PayPal URL detected. Please contact support.');
                                return;
                              }
                              
                            } catch (urlError) {
                              console.error('❌ URL validation failed:', urlError);
                              alert('Invalid URL format. Please contact support.');
                              return;
                            }
                            
                            // Create iframe dynamically and replace the payment form
                            console.log('🔍 Creating iframe with URL:', approvalUrl);
                            const iframe = document.createElement('iframe');
                            iframe.src = approvalUrl;
                            iframe.style.width = '100%';
                            iframe.style.height = '500px';
                            iframe.style.border = 'none';
                            iframe.style.borderRadius = '8px';
                            
                            // Add error handling for iframe
                            iframe.onload = () => {
                              console.log('✅ PayPal iframe loaded successfully');
                              setIsPaymentProcessing(true);
                              // Start automatic status checking when iframe loads
                              startPaymentStatusPolling();
                            };
                            
                            iframe.onerror = (error) => {
                              console.error('❌ PayPal iframe failed to load:', error);
                              alert('Failed to load PayPal payment form. Please try again or contact support.');
                            };
                            
                            // Add postMessage listener for PayPal communication
                            const handleMessage = (event: MessageEvent) => {
                              console.log('📨 Received postMessage:', event.data);
                              
                              // Check if message is from PayPal
                              if (event.origin.includes('paypal.com') || event.origin.includes('sandbox.paypal.com')) {
                                console.log('📨 PayPal postMessage detected:', event.data);
                                
                                // Check for payment completion indicators
                                if (event.data && typeof event.data === 'object') {
                                  const data = event.data;
                                  
                                  // PayPal success indicators
                                  if (data.type === 'success' || 
                                      data.status === 'success' || 
                                      data.paymentStatus === 'COMPLETED' ||
                                      data.action === 'payment_completed') {
                                    console.log('✅ PayPal payment completion detected via postMessage!');
                                    stopPaymentStatusPolling();
                                    setIsPaymentProcessing(false);
                                    onSuccess({ status: 'COMPLETED', bookingCode });
                                  }
                                }
                              }
                            };
                            
                            // Add message listener
                            window.addEventListener('message', handleMessage);
                            
                            // Cleanup listener when iframe is removed
                            const cleanup = () => {
                              window.removeEventListener('message', handleMessage);
                            };
                            
                            // Replace the payment form with iframe
                            const paymentContainer = document.querySelector('.payment-container');
                            if (paymentContainer) {
                              console.log('🔍 Replacing payment form with iframe');
                              paymentContainer.innerHTML = '';
                              paymentContainer.appendChild(iframe);
                              
                              // Store cleanup function for later use
                              (paymentContainer as any).cleanup = cleanup;
                            } else {
                              console.error('❌ Payment container not found');
                            }
                          } else {
                            console.error('❌ No approval URL provided');
                            alert('No payment URL available. Please try again or contact support.');
                          }
                        }}
                      >
                        Complete Payment
                      </Button>
                    </div>
                  </div>
                </div>
              </div>

              <div className="flex gap-2">
                <Button
                  variant="outline"
                  className="flex-1"
                  onClick={() => setShowPaymentAddress(false)}
                >
                  Back to Summary
                </Button>
                <Button
                  variant="secondary"
                  className="flex-1"
                  disabled={isLoading || !bookingCode || isPaymentProcessing}
                  onClick={handleCheckStatus}
                >
                  <RefreshCcw className="h-4 w-4 mr-2" />
                  {isPaymentProcessing ? 'Auto-checking...' : 'Check Payment Status'}
                </Button>
              </div>
            </div>
          )}

          <div className="text-xs text-gray-500 text-center">
            <p>PayPal Client ID: {PAYPAL_CONFIG.CLIENT_ID ? 'Configured' : 'Not configured'}</p>
            <p>Environment: {PAYPAL_CONFIG.IS_PRODUCTION ? 'Production' : 'Sandbox'}</p>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}