import { NextRequest, NextResponse } from "next/server";
import crypto from "crypto";

// PayPal webhook verification and processing
export async function POST(request: NextRequest) {
  try {
    const body = await request.text();
    const signature = request.headers.get("paypal-transmission-id");
    const certId = request.headers.get("paypal-cert-id");
    const authAlgo = request.headers.get("paypal-auth-algo");
    const transmissionSig = request.headers.get("paypal-transmission-sig");
    const transmissionTime = request.headers.get("paypal-transmission-time");
    const webhookId = request.headers.get("paypal-webhook-id");

    console.log("🔔 PayPal Webhook Received:");
    console.log("🔔 - Signature:", signature);
    console.log("🔔 - Cert ID:", certId);
    console.log("🔔 - Auth Algo:", authAlgo);
    console.log("🔔 - Transmission Sig:", transmissionSig);
    console.log("🔔 - Transmission Time:", transmissionTime);
    console.log("🔔 - Webhook ID:", webhookId);
    console.log("🔔 - Body:", body);

    // Parse the webhook payload
    const webhookData = JSON.parse(body);
    console.log("🔔 - Webhook Data:", JSON.stringify(webhookData, null, 2));

    // Verify webhook signature (optional but recommended)
    // Note: You should implement proper PayPal webhook verification
    // const isValid = await verifyPayPalWebhook(body, signature, certId, authAlgo, transmissionSig, transmissionTime, webhookId);

    // For now, we'll process the webhook without verification
    // In production, you should implement proper verification

    // Process different webhook event types
    const eventType = webhookData.event_type;
    console.log("🔔 - Event Type:", eventType);

    switch (eventType) {
      case "PAYMENT.CAPTURE.COMPLETED":
        await handlePaymentCompleted(webhookData);
        break;
      case "PAYMENT.CAPTURE.DENIED":
        await handlePaymentDenied(webhookData);
        break;
      case "PAYMENT.CAPTURE.PENDING":
        await handlePaymentPending(webhookData);
        break;
      case "PAYMENT.CAPTURE.REFUNDED":
        await handlePaymentRefunded(webhookData);
        break;
      default:
        console.log("🔔 - Unhandled event type:", eventType);
    }

    return NextResponse.json({ status: "success" });
  } catch (error) {
    console.error("❌ PayPal Webhook Error:", error);
    return NextResponse.json(
      { error: "Webhook processing failed" },
      { status: 500 }
    );
  }
}

// Handle completed payment
async function handlePaymentCompleted(webhookData: any) {
  try {
    console.log("✅ Payment Completed Webhook Processing");

    const paymentId = webhookData.resource?.id;
    const status = webhookData.resource?.status;
    const amount = webhookData.resource?.amount?.value;
    const currency = webhookData.resource?.amount?.currency_code;

    console.log("✅ - Payment ID:", paymentId);
    console.log("✅ - Status:", status);
    console.log("✅ - Amount:", amount);
    console.log("✅ - Currency:", currency);

    // Extract booking number from custom_id or description
    const customId = webhookData.resource?.custom_id;
    const description = webhookData.resource?.description;

    console.log("✅ - Custom ID:", customId);
    console.log("✅ - Description:", description);

    // Try to extract booking number from custom_id or description
    let bookingNumber = customId;
    if (!bookingNumber && description) {
      // Extract booking number from description if it contains one
      const bookingMatch = description.match(/Booking[:\s]+([A-Z0-9-]+)/i);
      if (bookingMatch) {
        bookingNumber = bookingMatch[1];
      }
    }

    if (!bookingNumber) {
      console.error("❌ No booking number found in webhook data");
      return;
    }

    console.log("✅ - Extracted Booking Number:", bookingNumber);

    // Update payment status in your backend
    const updateResponse = await fetch(
      `${process.env.NEXT_PUBLIC_API_URL}/paypal/update-payment-status`,
      {
        method: "POST",
        headers: {
          Authorization: `Bearer ${process.env.NEXT_PUBLIC_API_TOKEN}`,
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          booking_no: bookingNumber,
          payment_id: paymentId,
          status: "APPROVED",
          paypal_status: status,
          amount: amount,
          currency: currency,
          webhook_data: webhookData,
        }),
      }
    );

    if (updateResponse.ok) {
      console.log("✅ Payment status updated successfully");
    } else {
      console.error(
        "❌ Failed to update payment status:",
        await updateResponse.text()
      );
    }
  } catch (error) {
    console.error("❌ Error processing payment completed webhook:", error);
  }
}

// Handle denied payment
async function handlePaymentDenied(webhookData: any) {
  try {
    console.log("❌ Payment Denied Webhook Processing");

    const paymentId = webhookData.resource?.id;
    const status = webhookData.resource?.status;
    const customId = webhookData.resource?.custom_id;

    console.log("❌ - Payment ID:", paymentId);
    console.log("❌ - Status:", status);
    console.log("❌ - Custom ID:", customId);

    if (customId) {
      // Update payment status to denied
      await fetch(
        `${process.env.NEXT_PUBLIC_API_URL}/paypal/update-payment-status`,
        {
          method: "POST",
          headers: {
            Authorization: `Bearer ${process.env.NEXT_PUBLIC_API_TOKEN}`,
            "Content-Type": "application/json",
          },
          body: JSON.stringify({
            booking_no: customId,
            payment_id: paymentId,
            status: "DENIED",
            paypal_status: status,
            webhook_data: webhookData,
          }),
        }
      );
    }
  } catch (error) {
    console.error("❌ Error processing payment denied webhook:", error);
  }
}

// Handle pending payment
async function handlePaymentPending(webhookData: any) {
  try {
    console.log("⏳ Payment Pending Webhook Processing");

    const paymentId = webhookData.resource?.id;
    const status = webhookData.resource?.status;
    const customId = webhookData.resource?.custom_id;

    console.log("⏳ - Payment ID:", paymentId);
    console.log("⏳ - Status:", status);
    console.log("⏳ - Custom ID:", customId);

    if (customId) {
      // Update payment status to pending
      await fetch(
        `${process.env.NEXT_PUBLIC_API_URL}/paypal/update-payment-status`,
        {
          method: "POST",
          headers: {
            Authorization: `Bearer ${process.env.NEXT_PUBLIC_API_TOKEN}`,
            "Content-Type": "application/json",
          },
          body: JSON.stringify({
            booking_no: customId,
            payment_id: paymentId,
            status: "PENDING",
            paypal_status: status,
            webhook_data: webhookData,
          }),
        }
      );
    }
  } catch (error) {
    console.error("❌ Error processing payment pending webhook:", error);
  }
}

// Handle refunded payment
async function handlePaymentRefunded(webhookData: any) {
  try {
    console.log("💰 Payment Refunded Webhook Processing");

    const paymentId = webhookData.resource?.id;
    const status = webhookData.resource?.status;
    const customId = webhookData.resource?.custom_id;

    console.log("💰 - Payment ID:", paymentId);
    console.log("💰 - Status:", status);
    console.log("💰 - Custom ID:", customId);

    if (customId) {
      // Update payment status to refunded
      await fetch(
        `${process.env.NEXT_PUBLIC_API_URL}/paypal/update-payment-status`,
        {
          method: "POST",
          headers: {
            Authorization: `Bearer ${process.env.NEXT_PUBLIC_API_TOKEN}`,
            "Content-Type": "application/json",
          },
          body: JSON.stringify({
            booking_no: customId,
            payment_id: paymentId,
            status: "REFUNDED",
            paypal_status: status,
            webhook_data: webhookData,
          }),
        }
      );
    }
  } catch (error) {
    console.error("❌ Error processing payment refunded webhook:", error);
  }
}

// Optional: PayPal webhook verification function
async function verifyPayPalWebhook(
  body: string,
  signature: string | null,
  certId: string | null,
  authAlgo: string | null,
  transmissionSig: string | null,
  transmissionTime: string | null,
  webhookId: string | null
): Promise<boolean> {
  // Implement PayPal webhook verification here
  // This requires fetching PayPal's public certificate and verifying the signature
  // For now, we'll skip verification but you should implement this for production

  console.log("⚠️ Webhook verification not implemented - skipping for now");
  return true;
}
